/* $Id: filestore_stats.c 737 2006-06-13 20:19:04Z jim $
   teebu - An archiving tool
   Copyright (C) 2006 Jim Farrand

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the Free
   Software Foundation; either version 2 of the License, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT
   ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
   FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   this program; if not, write to the Free Software Foundation, Inc., 51
   Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <assert.h>
#include <stdio.h>

#include "logging.h"

#include "filestore_stats.h"

// FIXME: Argh!  This module is wrong, all wrong
// The error hierarchy is good, but should be read from a file
// Defining it in code like this is a pain

typedef struct print_params *print_params_t;

#define MAX_CHILD_SPACE 20

struct print_params
{
  status_t        pp_type;
  const char      *pp_empty_format, *pp_direct_format, *pp_indirect_format, *pp_both_format;
  print_params_t  *pp_children;
  int             pp_return_code;
};

static struct print_params print_params[STATUS_COUNT];
static bool print_params_inited = false;

static print_params_t pp_children_buffer[STATUS_COUNT][MAX_CHILD_SPACE];

static const char *
get_empty_format(status_t stat)
{
  switch (stat)
    {
    default: return NULL;
    }
}

static const char *
get_direct_format(status_t stat)
{
  switch (stat)
    {
    default: return NULL;
    }
}

static const char *
get_indirect_format(status_t stat)
{
  switch (stat)
    {
    default: return NULL;
    }
}

#define VERIFIED_FORMAT     "Verified:\t%d"
#define FAILED_FORMAT       "Failed:\t%d / %d"
#define SUCCESS_FORMAT      "Success:\t%d"
#define SKIP_FORMAT         "Skipped:\t%2$*d"
#define NOT_FOUND_FORMAT    "Not found:\t%2$*d"

static const char *
get_both_format(status_t stat)
{
  switch (stat)
    {
    case STAT_ARCHIVE: return "Archive";

    case STAT_ARCHIVE_WARNING:                return "Warnings:\t%d\t(%d)";
    case STAT_ARCHIVE_WARNING_UNRECOGNISED_TAG: return "Unrecognised tag:\t%d";
    case STAT_ARCHIVE_ERROR:                    return "Errors:\t%d\t(%d)";
    case STAT_ARCHIVE_INPUT_ERROR:              return "Input Errors:\t%d\t(%d)";
    case STAT_ARCHIVE_INPUT_ERROR_PROCESSING_STOPPED_EARLY: return "Processing stopped early:\t%d\t(%d)";
    case STAT_ARCHIVE_INPUT_ERROR_DATA_STOPPED_EARLY: return "Data stopped early:\t%d\t(%d)";
    case STAT_ARCHIVE_INPUT_ERROR_RECOVERY:     return "Error recovery:\t%d\t(%d)";
    case STAT_ARCHIVE_INPUT_ERROR_BAD_DATA:     return "Bad data:\t%d\t(%d)";
    case STAT_ARCHIVE_INPUT_ERROR_BAD_TIME:     return "Bad time:\t%d\t(%d)";
    case STAT_ARCHIVE_OUTPUT:                   return "Output";
    case STAT_ARCHIVE_OUTPUT_ERROR:             return "Output Errors:\t%d\t(%d)";
    case STAT_ARCHIVE_OUTPUT_FILE:              return "Files";
    case STAT_ARCHIVE_OUTPUT_FILE_OK:           return SUCCESS_FORMAT;
    case STAT_ARCHIVE_OUTPUT_FILE_CHECKSUM:     return "Checksums:\t%2$*d";
    case STAT_ARCHIVE_OUTPUT_FILE_CHECKSUM_SHA1: return "SHA1s:\t%d";
    case STAT_ARCHIVE_OUTPUT_FILE_CHECKSUM_MD5:  return "MD5s: \t%d";
    case STAT_ARCHIVE_OUTPUT_DIR:               return "Directories";
    case STAT_ARCHIVE_OUTPUT_DIR_OK:            return SUCCESS_FORMAT;
    case STAT_ARCHIVE_OUTPUT_LINK:              return "Links";
    case STAT_ARCHIVE_OUTPUT_LINK_OK:           return SUCCESS_FORMAT;
    case STAT_ARCHIVE_OUTPUT_SKIP:              return SKIP_FORMAT;
    case STAT_ARCHIVE_OUTPUT_SKIP_UNHANDLED:    return "Unhandled";
    case STAT_INPUT_ERROR:                      return "Input error:\t%d\t(%d)";
    case STAT_INPUT_ERROR_OPENING_FILE:         return "Opening file:\t%d";
    case STAT_INPUT_ERROR_STATING_FILE:         return "Stating filed:\t%d";
    case STAT_INPUT_ERROR_OPENING_DIR:          return "Opening directory:\t%d";
    case STAT_INPUT_ERROR_STATING_DIR:          return "Stating directory:\t%d";
    case STAT_INPUT_ERROR_OPENING_LINK:         return "Opening link:\t%d";
    case STAT_INPUT_ERROR_STATING_LINK:         return "Stating link:\t%d";
    case STAT_VERIFY:                           return "Verify";
    case STAT_VERIFY_FILE:                      return "Files";
    case STAT_VERIFY_FILE_OK:                   return SUCCESS_FORMAT;
    case STAT_VERIFY_FILE_FAILED:               return "Fail:\t%d\t(%d)";
    case STAT_VERIFY_FILE_DATA:                 return "Data:\t\t%2$*d";
    case STAT_VERIFY_FILE_DATA_OK:              return VERIFIED_FORMAT;
    case STAT_VERIFY_FILE_DATA_FAILED:          return FAILED_FORMAT;
    case STAT_VERIFY_FILE_CHECKSUM:             return "Checksums:\t%2$*d";
    case STAT_VERIFY_FILE_CHECKSUM_MD5:         return "MD5:\t\t%2$*d";
    case STAT_VERIFY_FILE_CHECKSUM_MD5_OK:      return VERIFIED_FORMAT;
    case STAT_VERIFY_FILE_CHECKSUM_MD5_FAILED:  return FAILED_FORMAT;
    case STAT_VERIFY_FILE_CHECKSUM_SHA1:        return "SHA1:\t\t%2$*d";
    case STAT_VERIFY_FILE_CHECKSUM_SHA1_OK:     return VERIFIED_FORMAT;
    case STAT_VERIFY_FILE_CHECKSUM_SHA1_FAILED: return FAILED_FORMAT;
    case STAT_VERIFY_PERMISSIONS:               return "Permissions:\t%2$*d";
    case STAT_VERIFY_PERMISSIONS_OK:            return VERIFIED_FORMAT;
    case STAT_VERIFY_PERMISSIONS_FAILED:        return FAILED_FORMAT;
    case STAT_VERIFY_CREATION_TIME:             return "Creation times:\t%2$*d";
    case STAT_VERIFY_CREATION_TIME_OK:          return VERIFIED_FORMAT;
    case STAT_VERIFY_CREATION_TIME_FAILED:      return FAILED_FORMAT;
    case STAT_VERIFY_MODIFICATION_TIME:         return "Modification times:\t%2$*d";
    case STAT_VERIFY_MODIFICATION_TIME_OK:      return VERIFIED_FORMAT;
    case STAT_VERIFY_MODIFICATION_TIME_FAILED:  return FAILED_FORMAT;
    case STAT_VERIFY_USER:                      return "User ownership:\t%2$*d";
    case STAT_VERIFY_USER_OK:                   return VERIFIED_FORMAT;
    case STAT_VERIFY_USER_FAILED:               return FAILED_FORMAT;
    case STAT_VERIFY_GROUP:                     return "Group ownership:\t%2$*d";
    case STAT_VERIFY_GROUP_OK:                  return VERIFIED_FORMAT;
    case STAT_VERIFY_GROUP_FAILED:              return FAILED_FORMAT;
    case STAT_VERIFY_DIR:                       return "Directories";
    case STAT_VERIFY_DIR_OK:                    return VERIFIED_FORMAT;
    case STAT_VERIFY_DIR_FAILED:                return FAILED_FORMAT;
    case STAT_VERIFY_DIR_FAILED_NOT_FOUND:      return NOT_FOUND_FORMAT;
    case STAT_VERIFY_LINK:                      return "Links";
    case STAT_VERIFY_LINK_TARGET:               return "Link targets:\t%2$*d";
    case STAT_VERIFY_LINK_TARGET_OK:            return VERIFIED_FORMAT;
    case STAT_VERIFY_LINK_TARGET_FAILED:        return FAILED_FORMAT;
    case STAT_CHECK: return "Stat check:\t%d\t(%d)";
    case STAT_CHECK_FILE: return "Stat check file:\t%d\t(%d)";
    case STAT_CHECK_FILE_OK: return "Stat check file ok:\t%d\t(%d)";
    case STAT_CHECK_FILE_FAILED: return "Stat check file failed:\t%d\t(%d)";
    case STAT_EXTRACT: return "Stat extract:\t%d\t(%d)";
    case STAT_EXTRACT_FILE_OK: return "Stat extract file ok:\t%d\t(%d)";
    case STAT_EXTRACT_FILE: return "Stat extract file:\t%d\t(%d)";
    case STAT_EXTRACT_FILE_EXCLUDED: return "Stat extract file excluded:\t%d\t(%d)";
    case STAT_EXTRACT_FILE_FAILED: return "Stat extract file failed:\t%d\t(%d)";
    case STAT_EXTRACT_DIR: return "Stat extract dir:\t%d\t(%d)";
    case STAT_EXTRACT_DIR_OK: return "Stat extract dir ok:\t%d\t(%d)";
    case STAT_EXTRACT_DIR_FAILED: return "Stat extract dir failed:\t%d\t(%d)";
    case STAT_EXTRACT_LINK: return "Stat extract link:\t%d\t(%d)";
    case STAT_EXTRACT_LINK_OK: return "Stat extract link ok:\t%d\t(%d)";
    case STAT_EXTRACT_LINK_FAILED: return "Stat extract link failed:\t%d\t(%d)";
    case STAT_EXTRACT_LINK_EXCLUDED: return "Stat extract link excluded:\t%d\t(%d)";
    case STAT_OUTPUT_ERROR: return "Stat output error:\t%d\t(%d)";
    case STAT_OUTPUT_ERROR_WRITING_FILE: return "Stat output error writing file:\t%d\t(%d)";
    case STAT_OUTPUT_ERROR_SETTING_FILE_TIME: return "Stat output error setting file time:\t%d\t(%d)";
    case STAT_OUTPUT_ERROR_SETTING_FILE_OWNER: return "Stat output error setting file owner:\t%d\t(%d)";
    case STAT_OUTPUT_ERROR_SETTING_FILE_GROUP: return "Stat output error setting file group:\t%d\t(%d)";
    case STAT_INTERNAL_ERROR: return "Stat internal error:\t%d\t(%d)";
    case STAT_OTHER: return "Stat other:\t%d\t(%d)";

    default: return NULL;
    }
}

static print_params_t *
get_children(status_t stat, print_params_t child_buf[MAX_CHILD_SPACE])
{
  switch (stat)
    {
    case STAT_ROOT:
      child_buf[0] = &print_params[STAT_ARCHIVE];
      child_buf[1] = &print_params[STAT_INPUT_ERROR];
      child_buf[2] = &print_params[STAT_OUTPUT_ERROR];
      child_buf[3] = &print_params[STAT_VERIFY];
      child_buf[4] = &print_params[STAT_CHECK];
      child_buf[5] = &print_params[STAT_EXTRACT];
      child_buf[6] = &print_params[STAT_INTERNAL_ERROR];
      child_buf[7] = &print_params[STAT_OTHER];
      child_buf[8] = NULL;
      return child_buf;
    case STAT_ARCHIVE:
      child_buf[0] = &print_params[STAT_ARCHIVE_OUTPUT];
      child_buf[1] = &print_params[STAT_ARCHIVE_WARNING];
      child_buf[2] = &print_params[STAT_ARCHIVE_ERROR];
      child_buf[3] = NULL;
      return child_buf;
    case STAT_ARCHIVE_WARNING:
      child_buf[0] = &print_params[STAT_ARCHIVE_WARNING_UNRECOGNISED_TAG];
      child_buf[1] = NULL;
      return child_buf;
    case STAT_ARCHIVE_ERROR:
      child_buf[0] = &print_params[STAT_ARCHIVE_INPUT_ERROR];
      child_buf[1] = &print_params[STAT_ARCHIVE_OUTPUT_ERROR];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_ARCHIVE_INPUT_ERROR:
      child_buf[0] = &print_params[STAT_ARCHIVE_INPUT_ERROR_PROCESSING_STOPPED_EARLY];
      child_buf[1] = &print_params[STAT_ARCHIVE_INPUT_ERROR_DATA_STOPPED_EARLY];
      child_buf[2] = &print_params[STAT_ARCHIVE_INPUT_ERROR_RECOVERY];
      child_buf[3] = &print_params[STAT_ARCHIVE_INPUT_ERROR_BAD_DATA];
      child_buf[4] = &print_params[STAT_ARCHIVE_INPUT_ERROR_BAD_TIME];
      child_buf[5] = NULL;
      return child_buf;
    case STAT_ARCHIVE_OUTPUT:
      child_buf[0] = &print_params[STAT_ARCHIVE_OUTPUT_FILE];
      child_buf[1] = &print_params[STAT_ARCHIVE_OUTPUT_DIR];
      child_buf[2] = &print_params[STAT_ARCHIVE_OUTPUT_LINK];
      child_buf[3] = &print_params[STAT_ARCHIVE_OUTPUT_SKIP];
      child_buf[4] = NULL;
      return child_buf;
    case STAT_ARCHIVE_OUTPUT_FILE:
      child_buf[0] = &print_params[STAT_ARCHIVE_OUTPUT_FILE_OK];
      child_buf[1] = &print_params[STAT_ARCHIVE_OUTPUT_FILE_CHECKSUM];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_ARCHIVE_OUTPUT_FILE_CHECKSUM:
      child_buf[0] = &print_params[STAT_ARCHIVE_OUTPUT_FILE_CHECKSUM_SHA1];
      child_buf[1] = &print_params[STAT_ARCHIVE_OUTPUT_FILE_CHECKSUM_MD5];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_ARCHIVE_OUTPUT_DIR:
      child_buf[0] = &print_params[STAT_ARCHIVE_OUTPUT_DIR_OK];
      child_buf[1] = NULL;
      return child_buf;
    case STAT_ARCHIVE_OUTPUT_LINK:
      child_buf[0] = &print_params[STAT_ARCHIVE_OUTPUT_LINK_OK];
      child_buf[1] = NULL;
      return child_buf;
    case STAT_ARCHIVE_OUTPUT_SKIP:
      child_buf[0] = &print_params[STAT_ARCHIVE_OUTPUT_SKIP_UNHANDLED];
      child_buf[1] = NULL;
      return child_buf;
    case STAT_INPUT_ERROR:
      child_buf[0] = &print_params[STAT_INPUT_ERROR_OPENING_FILE];
      child_buf[1] = &print_params[STAT_INPUT_ERROR_STATING_FILE];
      child_buf[2] = &print_params[STAT_INPUT_ERROR_OPENING_DIR];
      child_buf[3] = &print_params[STAT_INPUT_ERROR_STATING_DIR];
      child_buf[4] = &print_params[STAT_INPUT_ERROR_OPENING_LINK];
      child_buf[5] = &print_params[STAT_INPUT_ERROR_STATING_LINK];
      child_buf[6] = NULL;
      return child_buf;
    case STAT_VERIFY:
      child_buf[0] = &print_params[STAT_VERIFY_FILE];
      child_buf[1] = &print_params[STAT_VERIFY_PERMISSIONS];
      child_buf[2] = &print_params[STAT_VERIFY_MODIFICATION_TIME];
      child_buf[3] = &print_params[STAT_VERIFY_CREATION_TIME];
      child_buf[4] = &print_params[STAT_VERIFY_USER];
      child_buf[5] = &print_params[STAT_VERIFY_GROUP];
      child_buf[6] = &print_params[STAT_VERIFY_DIR];
      child_buf[7] = &print_params[STAT_VERIFY_LINK];
      child_buf[8] = NULL;
      return child_buf;
    case STAT_VERIFY_FILE:
      child_buf[0] = &print_params[STAT_VERIFY_FILE_OK];
      child_buf[1] = &print_params[STAT_VERIFY_FILE_FAILED];
      child_buf[2] = &print_params[STAT_VERIFY_FILE_DATA];
      child_buf[3] = &print_params[STAT_VERIFY_FILE_CHECKSUM];
      child_buf[4] = NULL;
      return child_buf;
    case STAT_VERIFY_FILE_DATA:
      child_buf[0] = &print_params[STAT_VERIFY_FILE_DATA_OK];
      child_buf[1] = &print_params[STAT_VERIFY_FILE_DATA_FAILED];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_VERIFY_FILE_CHECKSUM:
      child_buf[0] = &print_params[STAT_VERIFY_FILE_CHECKSUM_MD5];
      child_buf[1] = &print_params[STAT_VERIFY_FILE_CHECKSUM_SHA1];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_VERIFY_FILE_CHECKSUM_MD5:
      child_buf[0] = &print_params[STAT_VERIFY_FILE_CHECKSUM_MD5_OK];
      child_buf[1] = &print_params[STAT_VERIFY_FILE_CHECKSUM_MD5_FAILED];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_VERIFY_FILE_CHECKSUM_SHA1:
      child_buf[0] = &print_params[STAT_VERIFY_FILE_CHECKSUM_SHA1_OK];
      child_buf[1] = &print_params[STAT_VERIFY_FILE_CHECKSUM_SHA1_FAILED];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_VERIFY_PERMISSIONS:
      child_buf[0] = &print_params[STAT_VERIFY_PERMISSIONS_OK];
      child_buf[1] = &print_params[STAT_VERIFY_PERMISSIONS_FAILED];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_VERIFY_CREATION_TIME:
      child_buf[0] = &print_params[STAT_VERIFY_CREATION_TIME_OK];
      child_buf[1] = &print_params[STAT_VERIFY_CREATION_TIME_FAILED];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_VERIFY_MODIFICATION_TIME:
      child_buf[0] = &print_params[STAT_VERIFY_MODIFICATION_TIME_OK];
      child_buf[1] = &print_params[STAT_VERIFY_MODIFICATION_TIME_FAILED];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_VERIFY_USER:
      child_buf[0] = &print_params[STAT_VERIFY_USER_OK];
      child_buf[1] = &print_params[STAT_VERIFY_USER_FAILED];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_VERIFY_GROUP:
      child_buf[0] = &print_params[STAT_VERIFY_GROUP_OK];
      child_buf[1] = &print_params[STAT_VERIFY_GROUP_FAILED];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_VERIFY_DIR:
      child_buf[0] = &print_params[STAT_VERIFY_DIR_OK];
      child_buf[1] = &print_params[STAT_VERIFY_DIR_FAILED];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_VERIFY_DIR_FAILED:
      child_buf[0] = &print_params[STAT_VERIFY_DIR_FAILED_NOT_FOUND];
      child_buf[1] = NULL;
      return child_buf;
    case STAT_VERIFY_LINK:
      child_buf[0] = &print_params[STAT_VERIFY_LINK_TARGET];
      child_buf[1] = NULL;
      return child_buf;
    case STAT_VERIFY_LINK_TARGET:
      child_buf[0] = &print_params[STAT_VERIFY_LINK_TARGET_OK];
      child_buf[1] = &print_params[STAT_VERIFY_LINK_TARGET_FAILED];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_CHECK:
      child_buf[0] = &print_params[STAT_CHECK_FILE];
      child_buf[1] = NULL;
      return child_buf;
    case STAT_CHECK_FILE:
      child_buf[0] = &print_params[STAT_CHECK_FILE_OK];
      child_buf[1] = &print_params[STAT_CHECK_FILE_FAILED];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_EXTRACT:
      child_buf[0] = &print_params[STAT_EXTRACT_FILE];
      child_buf[1] = &print_params[STAT_EXTRACT_DIR];
      child_buf[2] = &print_params[STAT_EXTRACT_LINK];
      child_buf[3] = NULL;
      return child_buf;
    case STAT_EXTRACT_FILE:
      child_buf[0] = &print_params[STAT_EXTRACT_FILE_OK];
      child_buf[1] = &print_params[STAT_EXTRACT_FILE_EXCLUDED];
      child_buf[2] = &print_params[STAT_EXTRACT_FILE_FAILED];
      child_buf[3] = NULL;
      return child_buf;
    case STAT_EXTRACT_DIR:
      child_buf[0] = &print_params[STAT_EXTRACT_DIR_OK];
      child_buf[1] = &print_params[STAT_EXTRACT_DIR_FAILED];
      child_buf[2] = NULL;
      return child_buf;
    case STAT_EXTRACT_LINK:
      child_buf[0] = &print_params[STAT_EXTRACT_LINK_OK];
      child_buf[1] = &print_params[STAT_EXTRACT_LINK_FAILED];
      child_buf[2] = &print_params[STAT_EXTRACT_LINK_EXCLUDED];
      child_buf[3] = NULL;
      return child_buf;
    case STAT_OUTPUT_ERROR:
      child_buf[0] = &print_params[STAT_OUTPUT_ERROR_WRITING_FILE];
      child_buf[1] = &print_params[STAT_OUTPUT_ERROR_SETTING_FILE_TIME];
      child_buf[2] = &print_params[STAT_OUTPUT_ERROR_SETTING_FILE_OWNER];
      child_buf[3] = &print_params[STAT_OUTPUT_ERROR_SETTING_FILE_GROUP];
      child_buf[4] = NULL;
      return child_buf;
    default: return NULL;
    }
}

static int
get_return (status_t stat)
{
  switch (stat)
    {
    case STAT_ARCHIVE_OUTPUT_ERROR:
    case STAT_ARCHIVE_INPUT_ERROR_PROCESSING_STOPPED_EARLY:
    case STAT_INTERNAL_ERROR:
      return 100;
    default: return 0;
    }
}

static void
init_pp()
{
  for (int i = 0; i < STATUS_COUNT; i++)
    {
      print_params[i].pp_type             = i;
      print_params[i].pp_empty_format     = get_empty_format (i);
      print_params[i].pp_direct_format    = get_direct_format (i);
      print_params[i].pp_indirect_format  = get_indirect_format (i);
      print_params[i].pp_both_format      = get_both_format (i);
      print_params[i].pp_children         = get_children (i, pp_children_buffer[i]);
      print_params[i].pp_return_code      = get_return  (i);
    }
}

static print_params_t
get_default_pp()
{
  if (!print_params_inited)
    init_pp();

  return &print_params[STAT_ROOT];
}

void
init_filestore_stats (filestore_stats_t * stats)
{
  assert (stats);
  // stats->fsstat_success = 0;
  // stats->fsstat_warning = 0;
  // stats->fsstat_error = 0;

  for (int i = 0; i < STATUS_COUNT; i++)
    stats->fsstat_outcomes[i] = 0;

}

#define INDENT_SIZE 2

unsigned
get_indirect_count (filestore_stats_t *stats, print_params_t pp)
{
  unsigned r = 0;

  if (pp->pp_children)
    for (int i = 0; pp->pp_children[i]; i++)
      {
        r += stats->fsstat_outcomes[pp->pp_children[i]->pp_type];
        r += get_indirect_count (stats, pp->pp_children[i]);
      }

  return r;
}

#ifndef NDEBUG

static void
do_validate_print_params (print_params_t pp, bool covered[STATUS_COUNT])
{
  covered[pp->pp_type] = true;
  if (!(pp->pp_empty_format || pp->pp_direct_format || pp->pp_indirect_format || pp->pp_both_format))
    {
      LOGF (DEBUG, "Status not printable: %d", pp->pp_type);
    }

  if (pp->pp_children)
    for (int i = 0; pp->pp_children[i]; i++)
      {
        do_validate_print_params (pp->pp_children[i], covered);
      }
}

static void
validate_print_params (print_params_t pp)
{
  bool covered[STATUS_COUNT];
  for (int i = 0; i < STATUS_COUNT; i++)
    covered[i] = false;

  do_validate_print_params (pp, covered);

  for (int i = 0; i < STATUS_COUNT; i++)
    {
      if (!covered[i])
        {
          LOGF (DEBUG, "Status not reachable: %d", i);
        }
    }
}
#endif

static void 
do_print_filestore_stats (filestore_stats_t *stats, print_params_t pp, unsigned indent)
{
  unsigned direct_count = stats->fsstat_outcomes[pp->pp_type];
  unsigned indirect_count = get_indirect_count(stats, pp);

  const char *format;

  if (0 == direct_count)
    if (0 == indirect_count)
      format = pp->pp_empty_format;
    else
      if (pp->pp_indirect_format)
        format = pp->pp_indirect_format;
      else
        format = pp->pp_both_format;
  else
    if (0 == indirect_count)
      if (pp->pp_direct_format)
        format = pp->pp_direct_format;
      else
        format = pp->pp_both_format;
    else
      format = pp->pp_both_format;

  if (format)
    {
      for (int i = 0; i < indent; i++)
        for (int j = 0; j < INDENT_SIZE; j++)
          fputc(' ', stderr);

      fprintf(stderr, format, direct_count, indirect_count);
      fputc('\n', stderr);
    }

  if (indirect_count > 0)
    {
      for (int i = 0; pp->pp_children[i] != NULL; i++)
        do_print_filestore_stats(stats, pp->pp_children[i], indent+1);
    }
}

static int
do_compute_return_code (filestore_stats_t *stats, print_params_t pp, int max_so_far)
{
  if (!print_params_inited)
    init_pp();

  unsigned direct_count = stats->fsstat_outcomes[pp->pp_type];
  unsigned indirect_count = get_indirect_count(stats, pp);

  if (direct_count > 0 || indirect_count > 0)
    if (pp->pp_return_code > max_so_far)
      max_so_far = pp->pp_return_code;


  if (pp->pp_children)
    for (int i = 0; pp->pp_children[i] != NULL; i++)
      max_so_far = do_compute_return_code(stats, pp->pp_children[i], max_so_far);

  return max_so_far;
}

int compute_return_code (filestore_stats_t *stats)
{
  return do_compute_return_code (stats, get_default_pp(), 0);
}

void
print_filestore_stats (filestore_stats_t *stats)
{
  print_params_t pp = get_default_pp();
#ifndef NDEBUG
  validate_print_params (pp);
#endif
  fflush (stderr);

  do_print_filestore_stats (stats, pp, 0);

  fflush (stderr);
}
