/**
 * @file sysutils.h
 * Headers for the common functions for sysutils
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifndef _SYSUTILS_H_
#define _SYSUTILS_H_

#include <grp.h>
#include <pwd.h>
#include <argp.h>
#include <shadow.h>
#include <gshadow.h>
#include <sys/types.h>
#ifndef _GNU_SOURCE
#define _GNU_SOURCE
#endif /* _GNU_SOURCE */
#include "getopt.h"
#if STDC_HEADERS
#include <string.h>
#else
#if !HAVE_STRCHR
/** Redefinition of strchr for platforms that only have index */
#define strchr			index
/** Redefinition of strrchr for platforms that only have rindex */
#define strrchr			rindex
#endif /* !HAVE_STRCHR */
#if !HAVE_MEMCPY
/** Redefinition of memcpy for platforms that only have bcopy */
#define memcpy(dest, src, n)	bcopy((src), (dest), (n))
/** Redefinition of memmove for platforms that only have bcopy */
#define memmove(dest, src, n)	bcopy((src), (dest), (n))
#endif /* !HAVE_MEMCPY */
#endif /* STDC_HEADERS */
#include <sys/uio.h>

#ifndef _POSIX_SAVED_IDS
/** Redefinition of seteuid for platforms that only have setreuid */
#define seteuid(_uid)	setreuid(geteuid(), getuid())
/** Redefinition of setegid for platforms that only have setregid */
#define setegid(_gid)	setregid(getegid(), getgid())
#endif /* _POSIX_SAVED_IDS */

#define CH_FNAME	0x1		/**< Permission flag for full name */
#define CH_ROOM		0x2		/**< Permission flag for roomnumber */
#define CH_WPHONE	0x4		/**< Permission flag for workphone */
#define CH_HPHONE	0x8		/**< Permission flag for homephone */
#define CH_OTHER	0x10		/**< Permission flag for other info */
/** Combination of all permission flags */
#define CH_ALL		CH_FNAME | CH_ROOM | CH_WPHONE | CH_HPHONE

#define F_DEFAULT	(-1)		/**< Default file */
#define F_GROUP		0		/**< Group file */
#define F_PASSWD	1		/**< Password file */
#define F_NORMAL	0		/**< Normal file */
#define F_SHADOW	1		/**< Shadow file */

#define DF_EXISTSOK	1	/**< An existing home is regarded as ok */
#define DF_GROUPHOMES	2	/**<
				 * Create the home-directory inside a dir
				 * named like the user's primary group
				 */
#define DF_LETTERHOMES	4	/**<
				 * Create the home-directory inside a dir
				 * named by the first letter of the username
				 */
#define DF_USERGROUPS	8	/**<
				 * Each user has a group of their own; their
				 * home-directory is g+s
				 */

/** Path to passwd file */
#define PASSWD_FILE	SYSCONFDIR "/passwd"
/** Path to shadow file */
#define SHADOW_FILE	SYSCONFDIR "/shadow"
/** Path to group file */
#define GROUP_FILE	SYSCONFDIR "/group"

/* GSHADOW_FILE defined in gshadow.h */

/** Path to shells file */
#define SHELLS_FILE	SYSCONFDIR "/shells"

/** Path to lastlog file */
#define LASTLOG_FILE	"/var/log/lastlog"

/** Last Group ID.  If you expect more groups than this, bump this value */
#define LASTGID			59999
/** Last User ID.  If you expect more users than this, bump this value */
#define LASTUID			59999
/** Default path to home directory */
#define DEFAULT_HOME		"/home"
/** Default location of skeleton directory */
#define DEFAULT_SKEL		SYSCONFDIR "/skel"
/** Default shell for normal users */
#define DEFAULT_USER_SHELL	"/bin/bash"
/** Default shell for system users */
#define DEFAULT_SYS_SHELL	"/bin/false"

/** Default umask to use when creating home directories */
#define DEFAULT_UMASK		022
/** Default for usergroups; 1 - use 1 group/user, 0 - use default group */
#define DEFAULT_USERGROUPS	1
/** Default for letterhomes; 1 - /home/first letter/username */
#define DEFAULT_LETTERHOMES	0
/** Default for grouphomes; 1 - create homedirectories based on pgrp */
#define DEFAULT_GROUPHOMES	0
/** Default group to use for system users */
#define DEFAULT_SYSTEM_GROUP	"nobody"
/** Default group to use for normal users (if usergroups is set to false) */
#define DEFAULT_GROUP		"users"

/** Default first system User ID */
#define DEFAULT_FIRSTSUID	100
/** Default last system User ID */
#define DEFAULT_LASTSUID	999
/** Default first User ID */
#define DEFAULT_FIRSTUID	1000
/** Default last User ID */
#define DEFAULT_LASTUID		59999

/** Default first system User ID */
#define DEFAULT_FIRSTSGID	100
/** Default last system User ID */
#define DEFAULT_LASTSGID	999
/** Default first User ID */
#define DEFAULT_FIRSTGID	1000
/** Default last User ID */
#define DEFAULT_LASTGID		59999

/** Extension for backup files */
#define BACKUP_EXT	"-"
/** Extension for copies */
#define COPY_EXT	".copy"
/** Extension used while editing a file */
#define EDIT_EXT	".edit"
/** Extension used while writing a file */
#define WRITE_EXT	".write"

/**
 * Default editor to use; if /usr/bin/editor doesn't exist,
 * and VISUAL/EDITOR is unset */
#define DEFAULT_EDITOR	"/bin/vi"			/* of course! */
/** Fallback editor */
#define EDITOR		"/usr/bin/editor"

#define A_NONE		0x00000000		/**< No attributes */
#define A_NAME		0x00000001		/**< User/group name */
#define A_ID		0x00000002		/**< User/Group ID */
#define A_PGRP		0x00000004		/**< Primary group */
#define A_GROUPS	0x00000008		/**< Auxiliary groups */
#define A_HOME		0x00000010		/**< Home directory */
#define A_SHELL		0x00000020		/**< Login shell */
#define A_GECOS		0x00000040		/**< GECOS information */
#define A_USERS		0x00000080		/**< Group members */

#define A_ADMS		0x00800000		/**< Group admins */
#define A_LSTCHG	0x01000000		/**< Last password change */
#define A_MIN		0x02000000		/**< Min time before change */
#define A_MAX		0x04000000		/**< Max time before change */
#define A_WARN		0x08000000		/**< Time before warning */
#define A_INACT		0x10000000		/**< Time before inactivation */
#define A_EXPIRE	0x20000000		/**< Date of expiration */
#define A_LOCKED	0x40000000		/**< Password locked */
#define A_ADMGROUPS	0x80000000		/**< Groups user administers */
#define A_ALL		0xffffffff		/**< All attributes */

#define AS_ID		"id"		/**< Attr string; ID */
#define AS_GID		"gid"		/**< Alias for id */
#define AS_UID		"uid"		/**< Alias for id */
#define AS_PGRP		"pgrp"		/**< Attr string; primary group */
#define AS_GROUPS	"groups"	/**< Attr string; aux groups */
#define AS_HOME		"home"		/**< Attr string; home directory */
#define AS_SHELL	"shell"		/**< Attr string; login shell */
#define AS_GECOS	"gecos"		/**< Attr string; GECOS info */
#define AS_USERS	"users"		/**< Attr string; group members */
#define AS_USERNAME	"username"	/**< Attr string; username */
#define AS_GROUPNAME	"groupname"	/**< Attr string; groupname */
#define AS_ADMS		"adms"		/**< Attr string; group adms */
#define AS_LSTCHG	"lstchg"	/**< Attr string; last change */
#define AS_MIN		"min"		/**< Attr string; min change time */
#define AS_MAX		"max"		/**< Attr string; max change time */
#define AS_WARN		"warn"		/**< Attr string; warn time */
#define AS_INACT	"inact"		/**< Attr string; inact time */
#define AS_EXPIRE	"expire"	/**< Attr string; expiration date */
#define AS_LOCKED	"locked"	/**< Attr string; password lock */
#define AS_A_LOCKED	"account_locked"	/**< Alias for AS_LOCKED */
#define AS_G_LOCKED	"group_locked"		/**< Alias for AS_LOCKED */
#define AS_ADMGROUPS	"admgroups"	/**< Attr string; groups user admins */

#define G_NONE		0x00	/**< GECOS fields; none */
#define G_FNAME		0x01	/**< GECOS field; full user name */
#define G_ROOM		0x02	/**< GECOS field; room name */
#define G_WPHONE	0x04	/**< GECOS field; work phone */
#define G_HPHONE	0x08	/**< GECOS field; home phone */
#define G_OTHER		0x10	/**< GECOS field; other information */
#define G_ALL		0xff	/**< GECOS fields; all */

#define GE_FNAME	0	/**< GECOS field number; full user name */
#define GE_ROOM		1	/**< GECOS field number; room number */
#define GE_WPHONE	2	/**< GECOS field number; work phone */
#define GE_HPHONE	3	/**< GECOS field number; home phone */
#define GE_OTHER	4	/**< GECOS field number; other information */

#define GS_FNAME	"fname"		/**< GECOS attr; full user name */
#define GS_FULLNAME	"fullname"	/**< Alias for fname */
#define GS_ROOM		"room"		/**< GECOS attr; room number */
#define GS_WPHONE	"wphone"	/**< GECOS attr; work phone */
#define GS_HPHONE	"hphone"	/**< GECOS attr; home phone */
#define GS_OTHER	"other"		/**< GECOS attr; other information */

/** Password expiration information */
struct expirydata {
	long lstchg;			/**< Date of last change */
	long min;			/**< Min days before change allowed */
	long max;			/**< Max days before change required */
	long warn;			/**< Days before expiry to warn */
	long inact;			/**< Days after expiry to lock */
	long expire;			/**< Date of account expiration */
};

/** attribute=value parser/validator struct */
struct attr {
	const char *attribute;				/**< Attribute */
	void **value;					/**< Value */
	int (*validator)(const char *);			/**< Validator to use */
	int (*converter)(const char *, void **);	/**< Converter to use */
};

extern const char *progname;	/**< Used to store the name of the program */

/* miscellaneous functions */
void version(FILE *file, struct argp_state *state);
void set_author_information(const char *string);
error_t init_locales(const char *name);
size_t get_max_namelen(void);
long get_current_date(void);
char *input_string(const int echo);
char *input_password(const char *salt, const char *check, int *match);
error_t tty_write(const struct iovec *iov, const char *dst);

/* input parsing, type conversion and validation */
error_t string_to_bool(const char *string, void **value);
error_t string_to_uid_t(const char *string, void **value);
error_t string_to_gid_t(const char *string, void **value);
error_t string_to_long(const char *string, void **value);
error_t string_to_date(const char *string, void **value);
error_t string_to_string(const char *string, void **value);
char *date_to_string(const long days);
char *long_to_radix64(const long value);
error_t is_bool(const char *string);
error_t is_decimal(const char *string);
error_t is_uid_t(const char *string);
error_t is_gid_t(const char *string);
error_t is_long(const char *string);
error_t is_valid_date(const char *string);
error_t is_valid_path(const char *string);
error_t is_valid_filepath(const char *string);
error_t is_valid_shell(const char *string);
error_t is_listed_shell(const char *string);
error_t is_valid_string(const char *string, const char *invalid);
error_t is_valid_name(const char *string);
error_t is_valid_namelist_empty(const char *string);
error_t is_valid_namelist(const char *string);
error_t is_valid_gecos_field(const char *string);
error_t is_valid_gecos_other(const char *string);
error_t is_valid_gecos(const char *string);
error_t parse_key_pairs(char *string, struct attr attributes[]);

/* copy/backup/create/delete */
char *create_filename(const char *filename, const char *extension);
FILE *open_file(const char *filename, const char *mode);
error_t close_file(FILE **fp);
error_t backup_file(const char *source, const char *target);
error_t copy_file(const char *source, const char *target);
error_t copy_file_modes(const char *source, const char *target);
error_t set_file_perms(const char *file, const char *user,
		       const char *group, const mode_t mode);
error_t replace_file(const char *source, const char *target);
error_t unlink_file(const char *filename, error_t status);
char *create_home_directory(const struct passwd *pw, const int dirflags,
			    const mode_t homeumask, const char *homepath);

/* passwd/group/shadow/gshadow I/O */
int fputpwent(const struct passwd *pwd, FILE *target);
int fputspent(const struct spwd *spwd, FILE *target);
int fputgrent(const struct group *grp, FILE *target);
int fputsgent(const struct sgrp *sgrp, FILE *target);

/* file locking/unlocking */
error_t lock_files(void);
error_t unlock_files(error_t status);

/* passwd/shadow and group/gshadow related */
char *get_username(const uid_t uid);
char *get_groupname(const gid_t gid);
char *get_all_users(void);
uid_t *get_all_uids(uid_t *users);
char *get_all_groups(void);
char *get_groups(const char *username);
char *get_admgroups(const char *username);
char *get_group_primary_members(const struct group *gr);
char *get_group_members(const struct group *gr);
char *get_group_admins(const char *groupname);
uid_t get_free_uid(uid_t firstuid, uid_t lastuid);
gid_t get_free_gid(gid_t firstgid, gid_t lastgid);
char *get_host_name(void);
int is_password_locked(const char *password);
int is_user_locked(const char *username);
int is_group_locked(const char *groupname);
error_t is_root(void);
error_t is_useradmin(void);
error_t is_groupadmin(const char *groupname);
int is_caller(const char *name);
error_t is_free_uid(const uid_t uid);
error_t is_free_gid(const gid_t gid);
error_t is_free_username(const char *name);
error_t is_free_groupname(const char *name);

/* utility-functions for postprocessing passwd/shadow or group/gshadow */
char **split_gecos(const char *gecos);
char *join_gecos(const char *fname, const char *room, const char *wphone,
		 const char *hphone, const char *other);
int is_in_list(const char *list, const char *entry);
char *add_to_list(const char *list, const char *entry);
char *uniq_add_to_list(const char *list, const char *entry);
char *remove_from_list(const char *list, const char *entry);
int is_uniq_list(const char *list);
char *make_uniq_list(const char *list);
int is_in_array(char **array, const char *entry);
int is_distinct_array(char **array1, char **array2);
char **array_union(char **array1, char **array2);
char **array_cut(char **array1, char **array2);

#endif /* _SYSUTILS_H_ */
