/**
 * @file remove-shell.c
 * Remove a shell to the list of valid shells
 *
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#define _GNU_SOURCE		/**< getline() */
#include <argp.h>
#include <errno.h>
#include <stdio.h>
#include <limits.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "remove-shell"	/**< Name shown in logfiles */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("SHELL");

/** Program synopsis */
static char doc[] =
	N_("Remove a shell from the list of valid shells.");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "verbose", 'v', 0, 0,
	  N_("Warn if the shell is not listed as a valid shell"), -2 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	const char *shell;		/**< Shell to remove */
	int verbose;			/**< Warn if shell is not listed */
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'v':
		args->verbose = 1;
		break;

	case ARGP_KEY_INIT:
		args->shell = NULL;
		args->verbose = 0;
		break;

	case ARGP_KEY_ARG:
		if (args->shell)
			argp_usage(state);

		args->shell = arg;
		break;

	case ARGP_KEY_NO_ARGS:
		argp_usage(state);

	default:
		status = ARGP_ERR_UNKNOWN;
		break;
	}

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	FILE *rfp = NULL;
	FILE *wfp = NULL;

	error_t status = 0;

	size_t len = 0;

	char *shell = NULL;
	char *wname = NULL;
	char *bname = NULL;

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
	};

	struct arguments args;

	argp_program_version_hook = version;
	argp_err_exit_status = EINVAL;

	errno = 0;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Make sure the caller has root privileges */
	if ((status = is_root())) {
		if (status == EPERM) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "You must be root to %s.\n"),
				progname,
				_("remove a shell from "
				  "the list of valid shells"));
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_root", strerror(errno));
		}

		goto EXIT;
	}

	if (is_listed_shell(args.shell)) {
		if (args.verbose) {
			fprintf(stderr,
				_("%s: warning: `%s' does not exist "
				  "in the list of valid shells\n"),
				progname, argv[optind]);
		}

		goto EXIT;
	}

	/* Create filename /etc/shells.write */
	if (!(wname = create_filename(SHELLS_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/shells- */
	if (!(bname = create_filename(SHELLS_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Open /etc/shells */
	if (!(rfp = open_file(SHELLS_FILE, "r"))) {
		status = errno;
		goto EXIT;
	}

	/* Backup /etc/shells to /etc/shells- */
	if ((status = backup_file(SHELLS_FILE, bname)))
		goto EXIT2;

	/* Copy permissions from /etc/shells to /etc/shells- */
	if ((status = copy_file_modes(SHELLS_FILE, bname)))
		goto EXIT2;

	/* Open /etc/shells.write */
	if (!(wfp = open_file(wname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Perform changes on /etc/shells */
	while ((getline(&shell, &len, rfp) != -1)) {
		len = 0;

		if (!strncmp(shell, args.shell, strlen(args.shell))) {
			free(shell);
			shell = NULL;
			continue;
		}

		/* Write the entry */
		if ((status = fprintf(wfp, "%s", shell)) < 0) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fprintf()", strerror(errno));
			free(shell);
			goto EXIT2;
		}

		free(shell);
		shell = NULL;
	}

	/* Close /etc/shells.write */
	if ((status = close_file(&wfp)))
		goto EXIT2;

	/* Close /etc/shells */
	if ((status = close_file(&rfp)))
		goto EXIT2;

	/* Finally, move the new file in place */
	if ((status = replace_file(wname, SHELLS_FILE)))
		goto EXIT2;

	/* Set file permissions properly */
	if ((status = copy_file_modes(bname, SHELLS_FILE)))
		goto EXIT2;

EXIT2:
	/* These files might not exist, but that's ok */
	status = unlink_file(wname, status);

EXIT:
	free(wname);
	free(bname);

	return status;
}
