/**
 * @file misc.c
 * Memory and string management helper-functions
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>

#include "misc.h"

/**
 * Copy a string and return a pointer to its end
 *
 * @param dst A pointer to the destination
 * @param src A pointer to the source
 * @return A pointer to the end of the destination string
 */
static char *internal_stpcpy(char *dst, const char *src)
{
	if (!dst || !src)
		return NULL;

	while ((*dst++ = *src++) != '\0')
		/* nothing */;

	return dst - 1;
}

/**
 * Join a variable amount of strings,
 * with an optional delimiter between each,
 * including empty strings
 *
 * @param delimiter The string delimiter
 * @param ... The strings to join
 * @return A newly allocated string with the result
 */
char *strjoin(const char *delimiter, ...)
{
	va_list args;
	size_t dlen;
	size_t len;
	char *str;
	char *tmp;
	char *ptr = NULL;

	if (!delimiter)
		delimiter = "";

	dlen = strlen(delimiter);

	va_start(args, delimiter);

	if ((tmp = va_arg(args, char *))) {
		len = strlen(tmp) + 1;

		while ((tmp = va_arg(args, char *)))
			len += dlen + strlen(tmp);

		va_end(args);

		if (len < 2) {
			str = strdup("");
			goto EXIT;
		}

		if (!(str = ptr = calloc(len, sizeof (char))))
			goto EXIT;

		va_start(args, delimiter);

		while ((tmp = va_arg(args, char *))) {
			if (strlen(str))
				ptr = internal_stpcpy(ptr, delimiter);

			ptr = internal_stpcpy(ptr, tmp);
		}
	} else {
		str = strdup("");
	}

	va_end(args);

EXIT:
	return str;
}

/**
 * Join a variable amount of strings,
 * with an optional delimiter between each,
 * excluding empty strings
 *
 * @param delimiter The string delimiter
 * @param ... The strings to join
 * @return A newly allocated string with the result
 */
char *strjoine(const char *delimiter, ...)
{
	va_list args;
	size_t dlen;
	size_t len;
	char *str;
	char *tmp;
	char *ptr = NULL;

	if (!delimiter)
		delimiter = "";

	dlen = strlen(delimiter);

	va_start(args, delimiter);

	if ((tmp = va_arg(args, char *))) {
		len = strlen(tmp) + 1;

		while ((tmp = va_arg(args, char *))) {
			if (!strlen(tmp))
				continue;

			len += dlen + strlen(tmp);
		}

		va_end(args);

		if (len < 2) {
			str = strdup("");
			goto EXIT;
		}

		if (!(str = ptr = calloc(len, sizeof (char))))
			goto EXIT;

		va_start(args, delimiter);

		while ((tmp = va_arg(args, char *))) {
			if (!strlen(tmp))
				continue;

			if (strlen(str))
				ptr = internal_stpcpy(ptr, delimiter);

			ptr = internal_stpcpy(ptr, tmp);
		}
	} else {
		str = strdup("");
	}

	va_end(args);

EXIT:
	return str;
}

/**
 * Join a NULL-terminated array of strings
 * with an optional delimiter between each,
 * optionally excluding empty strings
 *
 * @param empty 0 to exclude empty strings,
 *              non-zero to include empty strings
 * @param delimiter The string delimiter
 * @param array The strings to join
 * @return A newly allocated string with the result
 */
static char *internal_strjoinv(const int empty, const char *delimiter,
			       char **array)
{
	size_t dlen;
	size_t len;
	char *str;
	char *tmp;
	char *ptr;
	int i;

	if (!delimiter)
		delimiter = "";

	dlen = strlen(delimiter);

	i = 0;

	if ((tmp = array[i++])) {
		len = strlen(tmp) + 1;

		while ((tmp = array[i++])) {
			if (!strlen(tmp) && !empty)
				continue;

			len += dlen + strlen(tmp);
		}

		if (len < 2) {
			str = strdup("");
			goto EXIT;
		}

		if (!(str = calloc(len, sizeof (char))))
			goto EXIT;

		i = 0;

		tmp = array[i++];
		ptr = internal_stpcpy(str, tmp);

		while ((tmp = array[i++])) {
			if (!strlen(tmp) && !empty)
				continue;

			if (strlen(str))
				ptr = internal_stpcpy(ptr, delimiter);

			ptr = internal_stpcpy(ptr, tmp);
		}
	} else {
		str = strdup("");
	}

EXIT:
	return str;
}

/**
 * Join a NULL-terminated array of strings
 * with an optional delimiter between each
 * including empty strings
 *
 * @param delimiter The string delimiter
 * @param array The strings to join
 * @return A newly allocated string with the result
 */
char *strjoinv(const char *delimiter, char **array)
{
	return internal_strjoinv(1, delimiter, array);
}

/**
 * Join a NULL-terminated array of strings
 * with an optional delimiter between each
 * excluding empty strings
 *
 * @param delimiter The string delimiter
 * @param array The strings to join
 * @return A newly allocated string with the result
 */
char *strjoinve(const char *delimiter, char **array)
{
	return internal_strjoinv(0, delimiter, array);
}

/**
 * Concatenate a variable amount of strings
 *
 * @param str1 The first string
 * @param ... The rest of the strings
 * @return A newly allocated string with the result
 */
char *strconcat(const char *str1, ...)
{
	va_list args;
	size_t len;
	char *str;
	char *tmp;
	char *ptr;

	if (!str1)
		return strdup("");

	len = strlen(str1) + 1;

	va_start(args, str1);

	if ((tmp = va_arg(args, char *))) {
		do {
			len += strlen(tmp);
		} while ((tmp = va_arg(args, char *)));

		va_end(args);

		if (!(str = calloc(len, sizeof (char))))
			goto EXIT;

		va_start(args, str1);

		ptr = internal_stpcpy(str, str1);

		while ((tmp = va_arg(args, char *)))
			ptr = internal_stpcpy(ptr, tmp);
	} else {
		str = strdup(str1);
	}

	va_end(args);

EXIT:
	return str;
}

/**
 * Split a string into an array, using the specified delimiter
 * to tell where to make the splits
 *
 * @param str A pointer to the string to split
 * @param delimiter A pointer to the delimiter
 * @param max_tokens The maximum number of tokens to split the string into
 *                   into; 0 means that the string should be fully split
 * @return A NULL-terminated, newly allocated array of strings
 */
char **strsplit(const char *str, const char *delimiter, const int max_tokens)
{
	int asize = 1;
	size_t dlen;
	const char *tmp2;
	char **array = NULL;
	char *tmp;

	if (!str || !delimiter || !delimiter[0])
		return NULL;

	dlen = strlen(delimiter);

	tmp2 = str;

	while ((tmp = strstr(tmp2, delimiter)) &&
	       (!max_tokens || asize <= max_tokens)) {
		size_t len;
		char *new;

		len = tmp - tmp2;

		if (!(new = calloc(len + 1, sizeof (char))))
			return NULL;

		strncpy(new, tmp2, len);
		new[len] = '\0';

		asize++;

		if (!(array = realloc(array, sizeof (char *) * asize)))
			return NULL;

		array[asize - 2] = new;

		tmp2 = tmp + dlen;
	}

	/* add the remaining string */
	asize++;

	if (!(array = realloc(array, sizeof (char *) * asize)))
		return NULL;

	if (!tmp2 || !(array[asize - 2] = strdup(tmp2)))
		return NULL;

	/* NULL terminate the array */
	array[asize - 1] = NULL;

	return array;
}

/**
 * Free a NULL-terminated array of strings, and the array itself
 *
 * @param[in,out] array The array to free
 */
void strfreev(char **array)
{
	int i;

	if (!array)
		return;

	for (i = 0; array[i]; i++)
		free(array[i]);

	free(array);
}
