/**
 * @file lsgroup.c
 * List attributes for the specified groups
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <grp.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "lsgroup"	/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

static int gshadowread;		/**< Can the user read /etc/gshadow */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("[ATTRIBUTE=VALUE...] GROUPS\n"
	   "-l");

/** Program synopsis */
static char doc[] =
	N_("List attributes for the specified groups that matches the\n"
	   "specified attributes.\n"
	   "\n"
	   "GROUPS should be a comma-separated list of groups, "
	   "or the word ALL to list attributes for all groups.\n"
	   "\n"
	   "Valid attributes:\n"
	   "\n"
	   "adms                           List of group administrators\n"
	   "group_locked | locked          Locked/unlocked group\n"
	   "groupname                      The groupname\n"
	   "id | gid                       The group's id\n"
	   "users                          List of group members\n");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "attr", 'a', N_("ATTRS"), OPTION_ARG_OPTIONAL,
	  N_("Display only the attributes "
	     "specified by a comma-separated list; "
	     "valid attributes are: "
	     "adms, group_locked | locked, id | gid, and users"), 0 },
	{ "colon", 'c', 0, 0,
	  N_("Display the attributes as colon-separated records"), 0 },
	{ "stanza", 'f', 0, 0,
	  N_("Display the attributes as stanzas"), 0 },
	{ "list", 'l', 0, OPTION_NO_USAGE,
	  N_("List the names of all groups, separated by commas"), 0 },
	{ "verbose", 'v', 0, 0,
	  N_("Warn if the specified groups do not exist"), -2 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	char *const *args;	/**< Arguments */
	const char *attrs;	/**< attribute=value pairs */
	int nargs;		/**< Number of arguments */
	int colon;		/**< Display as colon separated records */
	int list;		/**< Show comma separated list of all groups */
	int stanza;		/**< Display information as stanzas */
	int normal;		/**< Use normal output format */
	int verbose;		/**< Warn about non-existing groups */
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'a':
		args->attrs = arg ? arg : "";
		break;

	case 'c':
		args->colon = 1;
		args->normal = 0;
		break;

	case 'f':
		args->stanza = 1;
		args->normal = 0;
		break;

	case 'l':
		args->list = 1;
		break;

	case 'v':
		args->verbose = 1;
		break;

	case ARGP_KEY_INIT:
		args->args = NULL;
		args->nargs = 0;
		args->attrs = NULL;
		args->colon = 0;
		args->list = 0;
		args->normal = 1;
		args->stanza = 0;
		args->verbose = 0;
		break;

	case ARGP_KEY_ARGS:
		args->args = state->argv + state->next;
		args->nargs = state->argc - state->next;
		break;

	case ARGP_KEY_NO_ARGS:
		if (!args->list)
			argp_usage(state);

		break;

	case ARGP_KEY_END:
		if ((args->args) &&
		    (strchr(args->args[args->nargs - 1], '=')))
			argp_usage(state);

		if ((args->list) && (args->args))
			argp_usage(state);

		break;

	default:
		status = ARGP_ERR_UNKNOWN;
		break;
	}

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	error_t status = 0;

	unsigned int attr = A_ALL;

	char **grparray = NULL;

	gid_t user_gid = getgid();
	gid_t priv_gid = getegid();

	gid_t i; /* We're scanning <= LASTGID, hence gid_t */

	int j;

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
	};

	int lslocked = -1;
	gid_t lsid = 65535;
	char *lsname = NULL;
	char *lsadms = NULL;
	char *lsusers = NULL;

	struct attr attributes[] = {
		{
			.attribute	= AS_ADMS,
			.value		= (void **)&lsadms,
			.validator	= is_valid_namelist,
			.converter	= string_to_string
		}, {
			.attribute	= AS_G_LOCKED,
			.value		= (void **)&lslocked,
			.validator	= is_bool,
			.converter	= string_to_bool
		}, {
			.attribute	= AS_LOCKED,
			.value		= (void **)&lslocked,
			.validator	= is_bool,
			.converter	= string_to_bool
		}, {
			.attribute	= AS_GID,
			.value		= (void **)&lsid,
			.validator	= is_gid_t,
			.converter	= string_to_gid_t
		}, {
			.attribute	= AS_GROUPNAME,
			.value		= (void **)&lsname,
			.validator	= is_valid_name,
			.converter	= string_to_string
		}, {
			.attribute	= AS_ID,
			.value		= (void **)&lsid,
			.validator	= is_gid_t,
			.converter	= string_to_gid_t
		}, {
			.attribute	= AS_USERS,
			.value		= (void **)&lsusers,
			.validator	= is_valid_namelist,
			.converter	= string_to_string
		}, {
			.attribute	= NULL,
			.value		= NULL,
			.validator	= NULL,
			.converter	= NULL
		}
	};

	struct arguments args;

	/* Drop sgid */
	setegid(user_gid);
	errno = 0;

	argp_program_version_hook = version;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Find out if the shadow files can be read or not */

	/* Regain sgid */
	setegid(priv_gid);
	errno = 0;

	/* Attempt to get an entry from the group shadow password database */
	(void)getsgent();

	/* Did something fail? */
	if (errno) {
		if (errno == EACCES || errno == ENOENT) {
			errno = 0;
			attr &= ~A_LOCKED;
			attr &= ~A_ADMS;
		} else {
			status = errno;

			/* Drop sgid */
			setegid(user_gid);
			errno = status;

			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "getsgent()", strerror(status));

			goto EXIT;
		}
	} else {
		gshadowread = 1;
	}

	/* Drop sgid */
	setegid(user_gid);
	errno = 0;

	/* Parse the attributes if --attr is used */
	if (args.attrs) {
		char **attrarray = NULL;

		attr = A_NAME;

		if (!(attrarray = strsplit(args.attrs, ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
			goto EXIT;
		}

		for (i = 0; attrarray[i]; i++) {
			if (!strlen(attrarray[i])) {
				continue;
			} else if (!strcmp(attrarray[i], AS_ID)) {
				attr |= A_ID;
			} else if (!strcmp(attrarray[i], AS_GID)) {
				attr |= A_ID;
			} else if (!strcmp(attrarray[i], AS_USERS)) {
				attr |= A_USERS;
			} else if (gshadowread &&
					!strcmp(attrarray[i], AS_LOCKED)) {
				attr |= A_LOCKED;
			} else if (gshadowread &&
					!strcmp(attrarray[i], AS_G_LOCKED)) {
				attr |= A_LOCKED;
			} else if (gshadowread &&
					!strcmp(attrarray[i], AS_ADMS)) {
				attr |= A_ADMS;
			} else {
				fprintf(stderr,
					_("%s: invalid attributes "
					  "for '--attr'\n"
					  "Valid attributes are:\n"
					  "%s%sid | gid, and users\n"),
					progname,
					gshadowread ? "adms, " : "",
					gshadowread ? "group_locked | "
						      "locked, " : "");
				status = EINVAL;
				strfreev(attrarray);
				goto EXIT;
			}
		}

		strfreev(attrarray);
	}

	if (args.colon && args.stanza) {
		fprintf(stderr,
			_("%s: `-c' and `-f' cannot be combined\n"
			  "Try `%s --help' for more information.\n"),
			progname, progname);
		status = EINVAL;
		goto EXIT;
	}

	/* There are two alternatives here, neither of which are really
	 * pretty; either to read the entire group file once to get
	 * all groupnames, then use them for the ALL list, or to
	 * have separate code for the ALL case and the case of separate
	 * group-entries.  Since the latter is probably the most common,
	 * the latter has been chosen.
	 */
	if (args.list || !strcmp(args.args[args.nargs - 1], "ALL")) {
		char *tmp = NULL;

		if (!(tmp = get_all_groups())) {
			status = errno;
		} else if (!strlen(tmp)) {
			fprintf(stderr,
				_("%s: could not find any %s; the %s "
				  "might be corrupt\n"),
				progname, _("groups"), _("group database"));
			status = ENOENT;
		} else if (args.list) {
			fprintf(stdout, "%s\n", tmp);
		} else if (!(grparray = strsplit(tmp, ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
		}

		free(tmp);

		if (status || args.list)
			goto EXIT;
	} else {
		if (!(grparray = strsplit(args.args[args.nargs - 1], ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	/* Parse the attribute=value pairs */
	for (j = 0; j < (args.nargs - 1); j++) {
		if ((status = parse_key_pairs(args.args[j], attributes))) {
			if (status == ENOENT) {
				fprintf(stderr,
					_("%s: invalid attributes\n"
					"Valid attributes are:\n"
					"- `%s' | `%s'\n"
					"- `%s'\n"
					"- `%s' | `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"- `%s' | `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"- `%s'\n"
					"Try `%s --help' for more "
					"information.\n"),
					progname,
					AS_A_LOCKED, AS_LOCKED,
					AS_ADMGROUPS,
					GS_FNAME, GS_FULLNAME,
					AS_GECOS,
					AS_GROUPS,
					AS_HOME,
					GS_HPHONE,
					AS_ID, AS_GID,
					GS_OTHER,
					AS_PGRP,
					GS_ROOM,
					AS_SHELL,
					AS_USERNAME,
					GS_WPHONE,
					progname);
				status = EINVAL;
			}

			goto EXIT;
		}
	}


	/* Ok, now the only thing left is to list all specified groups */
	for (i = 0; grparray[i]; i++) {
		char *admlist = NULL;
		char *memlist = NULL;
		int isgrpadmin = 0;
		struct group *gr;
		int locked = 0;

		/* Get the next group entry */
		if (!(gr = getgrnam(grparray[i])) && errno) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "getgrnam()", strerror(errno));
			status = errno;
			goto EXIT;
		}

		/* Skip non-existing entries */
		if (!gr) {
			if (args.verbose)
				fprintf(stderr,
					_("%s: warning: %s `%s' "
					  "does not exist ... skipping\n"),
					progname, _("group"), grparray[i]);
			continue;
		}

		/* Try to find out if the group is locked; only works
		 * if we have access to reading /etc/gshadow
		 * Do an is_groupadmin() at the same time
		 */
		if (gshadowread && attr & A_LOCKED) {
			/* Regain sgid */
			setegid(priv_gid);
			errno = 0;

			isgrpadmin = !is_groupadmin(gr->gr_name);
			errno = 0;

			if ((locked = is_group_locked(gr->gr_name)) == -1) {
				if (errno != ENOENT) {
					status = errno;

					/* Drop sgid */
					setegid(user_gid);
					errno = 0;

					goto EXIT;
				}

				/* Drop sgid */
				setegid(user_gid);
				errno = 0;

				/* Either we don't have /etc/gshadow,
				 * or there was no such group
				 */
				if (args.verbose)
					fprintf(stderr,
						_("%s: warning: `%s' "
						  "does not exist in the "
						  "group shadow database\n"),
						progname, gr->gr_name);
				locked = (strlen(gr->gr_passwd) == 1);
			} else {
				/* Drop sgid */
				setegid(user_gid);
				errno = 0;
			}
		}

		/* Find out the list of group administrators; only works
		 * if we have access to reading /etc/gshadow
		 */
		if (gshadowread && attr & A_ADMS) {
			/* Regain sgid */
			setegid(priv_gid);
			errno = 0;

			if (!(admlist = get_group_admins(gr->gr_name))) {
				if (errno != ENOENT) {
					status = errno;

					/* Drop sgid */
					setegid(user_gid);
					errno = status;

					goto EXIT;
				}

				/* Either we don't have /etc/gshadow,
				 * or there was no such group
				 */
				admlist = NULL;
			}

			/* Drop sgid */
			setegid(user_gid);
			errno = 0;
		}

		/* Get the list of group members */
		if (!(memlist = get_group_members(gr))) {
			free(admlist);
			status = errno;
			goto EXIT;
		}

		/* Filter based on attributes */
		if ((lslocked > -1) && (lslocked != locked))
			continue;

		if ((lsid != 65535) && (lsid != gr->gr_gid))
			continue;

		if ((lsusers) && (strcmp(lsusers, memlist)))
			continue;

		if ((lsadms) && (strcmp(lsadms, nstr(admlist))))
			continue;

		/* Output all information */
		if (args.normal) {
			fprintf(stdout, "%s", gr->gr_name);

			if ((attr & A_LOCKED) && isgrpadmin)
				fprintf(stdout,
					" %s=%s",
					_("locked"),
					locked ? _("yes") : _("no"));

			if (attr & A_ID)
				fprintf(stdout,
					" %s=%d",
					_("id"), gr->gr_gid);

			if (attr & A_USERS)
				fprintf(stdout,
					" %s=%s",
					_("users"), memlist);

			if ((attr & A_ADMS) && isgrpadmin)
				fprintf(stdout,
					" %s=%s",
					_("adms"), nstr(admlist));
		} else if (args.colon) {
			fprintf(stdout,
				"#%s%s%s%s%s%s%s%s%s",
				_("name"),
				((attr & A_LOCKED) &&
				 isgrpadmin) ? ":" : "",
				((attr & A_LOCKED) &&
				 isgrpadmin) ? _("locked") : "",
				(attr & A_ID) ? ":" : "",
				(attr & A_ID) ? _("id") : "",
				(attr & A_USERS) ? ":" : "",
				(attr & A_USERS) ? _("users") : "",
				((attr & A_ADMS) &&
				 isgrpadmin) ? ":" : "",
				((attr & A_ADMS) &&
				 isgrpadmin) ? _("adms") : "");

			fprintf(stdout, "\n%s", gr->gr_name);

			if ((attr & A_LOCKED) && isgrpadmin)
				fprintf(stdout,
					":%s",
					locked ? _("yes") : _("no"));

			if (attr & A_ID)
				fprintf(stdout, ":%d", (int)gr->gr_gid);

			if (attr & A_USERS)
				fprintf(stdout, ":%s", memlist);

			if ((attr & A_ADMS) && isgrpadmin)
				fprintf(stdout, ":%s", nstr(admlist));
		} else {
			fprintf(stdout, "%s:", gr->gr_name);

			if (attr & A_ID)
				fprintf(stdout,
					"\n\t%s=%d",
					_("id"), (int)gr->gr_gid);

			if ((attr & A_LOCKED) && isgrpadmin)
				fprintf(stdout,
					"\n\t%s=%s",
					_("locked"),
					locked ? _("yes") : _("no"));

			if (attr & A_USERS)
				fprintf(stdout,
					"\n\t%s=%s",
					_("users"), memlist);

			if ((attr & A_ADMS) && isgrpadmin)
				fprintf(stdout,
					"\n\t%s=%s",
					_("adms"), nstr(admlist));
		}

		fprintf(stdout, "\n");

		if (args.stanza)
			fprintf(stdout, "\n");

		free(admlist);
		free(memlist);
	}

EXIT:
	/* Free all allocated memory */
	strfreev(grparray);

	return status;
}
