/**
 * @file lsage.c
 * List password expiry information the specified users
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <shadow.h>
#include <unistd.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "lsage"	/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("[USERS]");

/** Program synopsis */
static char doc[] =
	N_("List password expiry information for the specified users\n"
	   "that matches the specified attributes.\n"
	   "\n"
	   "USERS should be a comma-separated list of users, "
	   "or the word ALL to show expiry information for all users. "
	   "If no username is specified, "
	   "the information for the own account is shown.\n"
	   "\n"
	   "Only user administrators can view information about other "
	   "users' accounts.\n"
	   "\n"
	   "Valid attributes:\n"
	   "\n"
	   "account_locked | locked        Locked/unlocked account\n"
	   "expire                         The date of expiration for the "
	   "account (YYYY-MM-DD)\n"
	   "inact                          The maximum number of days of "
	   "inactivity after\n"
	   "                                 password expiry before the "
	   "account is locked\n"
	   "lstchg                         The date of the last password "
	   "change\n"
	   "                                 (YYYY-MM-DD)\n"
	   "max                            The maximum number of days before "
	   "the password\n"
	   "                                 must be changed\n"
	   "min                            The minimum number of days before "
	   "the password\n"
	   "                                 can be changed\n"
	   "warn                           The number of days before "
	   "password expiration\n"
	   "                                 to warn\n");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "attr", 'a', N_("ATTRS"), OPTION_ARG_OPTIONAL,
	  N_("Display only the attributes specified "
	     "by a comma-separated list; valid attributes are: "
	     "min, max, warn, inact, lstchg, expire, and locked"), 0 },
	{ "colon", 'c', 0, OPTION_ARG_OPTIONAL,
	  N_("Display the attributes as colon-separated records"), 0 },
	{ "stanza", 'f', 0, OPTION_ARG_OPTIONAL,
	  N_("Display the attributes as stanzas"), 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	char *const *args;	/**< Arguments */
	const char *attrs;	/**< attribute=value pairs */
	const char *users;	/**< Users to list information for */
	int nargs;		/**< Number of arguments */
	int colon;		/**< Display as colon separated records */
	int stanza;		/**< Display information as stanzas */
	int normal;		/**< Use normal output format */
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'a':
		args->attrs = arg ? arg : "";
		break;

	case 'c':
		args->colon = 1;
		args->normal = 0;
		break;

	case 'f':
		args->stanza = 1;
		args->normal = 0;
		break;

	case ARGP_KEY_INIT:
		args->args = NULL;
		args->nargs = 0;
		args->attrs = NULL;
		args->normal = 1;
		args->colon = 0;
		args->stanza = 0;
		break;

	case ARGP_KEY_ARGS:
		args->args = state->argv + state->next;
		args->nargs = state->argc - state->next;
		break;

	case ARGP_KEY_NO_ARGS:
		/* get_username prints a message on error */
		if (!(args->users = get_username(getuid()))) {
			status = errno;
			goto EXIT;
		}

		break;

	case ARGP_KEY_END:
		if ((args->args) &&
		    (strchr(args->args[args->nargs - 1], '='))) {
			/* get_username prints a message on error */
			if (!(args->users = get_username(getuid()))) {
				status = errno;
				goto EXIT;
			}
		} else if (args->nargs > 0)
			args->users = args->args[args->nargs - 1];
		break;

	default:
		status = ARGP_ERR_UNKNOWN;
		break;
	}

EXIT:

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	error_t status = 0;

	int isadmin = 0;

	unsigned int attr = A_ALL;

	char **usrarray = NULL;
	char **attrarray = NULL;

	char *expires = NULL;
	char *date = NULL;

	gid_t user_gid = getgid();
	gid_t priv_gid = getegid();

	uid_t i; /* We're scanning <= max(nrofusers), hence uid_t */

	int j;

	int lsmax = -1;
	int lsmin = -1;
	int lswarn = -1;
	int lsinact = -1;
	int lslocked = -1;
	char *lslstchg = NULL;
	char *lsexpire = NULL;

	struct attr attributes[] = {
		{
			.attribute	= AS_A_LOCKED,
			.value		= (void **)&lslocked,
			.validator	= is_bool,
			.converter	= string_to_bool
		}, {
			.attribute	= AS_MAX,
			.value		= (void **)&lsmax,
			.validator	= is_long,
			.converter	= string_to_long
		}, {
			.attribute	= AS_INACT,
			.value		= (void **)&lsinact,
			.validator	= is_long,
			.converter	= string_to_long
		}, {
			.attribute	= AS_WARN,
			.value		= (void **)&lswarn,
			.validator	= is_long,
			.converter	= string_to_long
		}, {
			.attribute	= AS_MIN,
			.value		= (void **)&lsmin,
			.validator	= is_long,
			.converter	= string_to_long
		}, {
			.attribute	= AS_LOCKED,
			.value		= (void **)&lslocked,
			.validator	= is_bool,
			.converter	= string_to_bool
		}, {
			.attribute	= AS_LSTCHG,
			.value		= (void **)&lslstchg,
			.validator	= is_valid_date,
			.converter	= string_to_date
		}, {
			.attribute	= AS_EXPIRE,
			.value		= (void **)&lsexpire,
			.validator	= is_valid_date,
			.converter	= string_to_date
		}, {
			.attribute	= NULL,
			.value		= NULL,
			.validator	= NULL,
			.converter	= NULL
		}
	};

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
	};

	struct arguments args;

	/* Drop sgid */
	setegid(user_gid);
	errno = 0;

	argp_program_version_hook = version;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Parse the attributes if --attr is used */
	if (args.attrs) {
		attr = A_NAME;

		if (!(attrarray = strsplit(args.attrs, ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
			goto EXIT;
		}

		for (i = 0; attrarray[i]; i++) {
			if (!strlen(attrarray[i])) {
				continue;
			} else if (!strcmp(attrarray[i], AS_MIN)) {
				attr |= A_MIN;
			} else if (!strcmp(attrarray[i], AS_MAX)) {
				attr |= A_MAX;
			} else if (!strcmp(attrarray[i], AS_WARN)) {
				attr |= A_WARN;
			} else if (!strcmp(attrarray[i], AS_INACT)) {
				attr |= A_INACT;
			} else if (!strcmp(attrarray[i], AS_LSTCHG)) {
				attr |= A_LSTCHG;
			} else if (!strcmp(attrarray[i], AS_EXPIRE)) {
				attr |= A_EXPIRE;
			} else if (!strcmp(attrarray[i], AS_LOCKED)) {
				attr |= A_LOCKED;
			} else if (!strcmp(attrarray[i], AS_A_LOCKED)) {
				attr |= A_LOCKED;
			} else {
				fprintf(stderr,
					_("%s: invalid attributes "
					  "for `--attr'\n"
					  "Valid attributes are:\n"
					  "- `%s' | `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "Try `%s --help' for more "
					  "information.\n"),
					progname,
					AS_A_LOCKED, AS_LOCKED,
					AS_EXPIRE, AS_INACT, AS_LSTCHG,
					AS_MAX, AS_MIN, AS_WARN,
					progname);
				status = EINVAL;
				strfreev(attrarray);
				goto EXIT;
			}
		}

		strfreev(attrarray);
	}

	if (args.colon && args.stanza) {
		fprintf(stderr,
			_("%s: `-c' and `-f' cannot be combined\n"
			  "Try `%s --help' for more information.\n"),
			progname, progname);
		status = EINVAL;
		goto EXIT;
	}

	/* Check if the user is a user-administrator */
	if ((status = is_useradmin())) {
		if (status == EPERM) {
			status = 0;
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_useradmin", strerror(errno));
			goto EXIT;
		}
	} else {
		isadmin = 1;
	}

	/* Unless the user requests information about himself,
	 * or is a user-administrator, deny the request
	 */
	if (args.users && !isadmin) {
		int retval = is_caller(args.users);

		if (retval == -1) {
			status = errno;
			goto EXIT;
		} else if (!retval) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "You must be a user-administrator to "
				  "%s %s\n"
				  "for other users than yourself.\n"),
				progname, _("view"),
				_("password aging information"));
			status = EPERM;
			goto EXIT;
		}
	}

	/* There are two alternatives here, neither of which are really
	 * pretty; either to read the entire passwd file once to get
	 * all usernames, then use them for the ALL list, or to
	 * have separate code for the ALL case and the case of separate
	 * user-entries.  Since the latter is probably the most common,
	 * the latter has been chosen.
	 */
	if (args.users && !strcmp(args.users, "ALL")) {
		char *tmp = NULL;

		if (!(tmp = get_all_users())) {
			status = errno;
		} else if (!strlen(tmp)) {
			fprintf(stderr,
				_("%s: could not find any %s; the %s "
				  "might be corrupt\n"),
				progname, _("users"), _("user database"));
			status = ENOENT;
		} else if (!(usrarray = strsplit(tmp, ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
		}

		free(tmp);

		if (status)
			goto EXIT;
	} else {
		if ((status = is_valid_namelist(args.users))) {
			if (status == EINVAL) {
				fprintf(stderr,
					_("%s: the specified list of %s "
					  "contains invalid characters\n"),
					progname, _("users"));
			} else {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "is_valid_namelist",
					strerror(errno));
			}

			goto EXIT;
		}

		if (!(usrarray = strsplit(args.users, ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	/* Parse the attribute=value pairs */
	for (j = 0; j < (args.nargs - 1); j++) {
		if ((status = parse_key_pairs(args.args[j], attributes))) {
			if (status == ENOENT) {
				fprintf(stderr,
					_("%s: invalid attributes\n"
					  "Valid attributes are:\n"
					  "- `%s' | `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "- `%s'\n"
					  "Try `%s --help' for more "
					  "information.\n"),
					progname,
					AS_A_LOCKED, AS_LOCKED,
					AS_EXPIRE, AS_INACT, AS_LSTCHG,
					AS_MAX, AS_MIN, AS_WARN,
					progname);
				status = EINVAL;
			}

			goto EXIT;
		}
	}

	/* View aging infomation for all specified users */
	for (i = 0; usrarray[i]; i++) {
		struct spwd *sp;
		int locked = 0;

		/* Regain sgid */
		setegid(priv_gid);
		errno = 0;

		/* Get the next shadow entry */
		if (!(sp = getspnam(usrarray[i])) && errno) {
			status = errno;
			/* Drop sgid */
			setegid(user_gid);
			errno = status;

			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "getspnam()", strerror(status));
			goto EXIT;
		}

		/* Drop sgid */
		setegid(user_gid);
		errno = 0;

		/* Skip non-existing entries */
		if (!sp)
			continue;

		/* Clear the password field; except the first character;
		 * we need the first character to check if the password
		 * is locked, but anything more than that is just an
		 * unnecessary risk of leaking information
		 */
		if (sp->sp_pwdp && sp->sp_pwdp[0]) {
			j = 1;

			while (sp->sp_pwdp[j])
				sp->sp_pwdp[j++] = '!';
		}

		/* Convert number of days since 1970-01-01 to a
		 * printable date in ISO-8601 format (YYYY-MM-DD)
		 */
		if (!(date = date_to_string(sp->sp_lstchg))) {
			status = errno;
			goto EXIT;
		}

		/* Convert number of days since 1970-01-01 to a
		 * printable date in ISO-8601 format (YYYY-MM-DD)
		 */
		if (!(expires = date_to_string(sp->sp_expire))) {
			status = errno;
			goto EXIT;
		}

		/* Check whether the user is locked */
		if ((locked = is_password_locked(sp->sp_pwdp)) == -1) {
			status = errno;
			goto EXIT;
		}

		/* Filter based on attributes */
		if ((lsmin > -1) && (lsmin != sp->sp_min))
			continue;

		if ((lsmax > -1) && (lsmax != sp->sp_max))
			continue;

		if ((lswarn > -1) && (lswarn != sp->sp_warn))
			continue;

		if ((lsinact > -1) && (lsinact != sp->sp_inact))
			continue;

		if ((lslstchg) && (lslstchg != date))
			continue;

		if ((lsexpire) && (lsexpire != expires))
			continue;

		if ((lslocked > -1) && (lslocked != locked))
			continue;

		/* Output all information */
		if (args.normal) {
			fprintf(stdout, "%s", sp->sp_namp);

			if (attr & A_MIN)
				fprintf(stdout,
					" %s=%ld",
					_("min"), sp->sp_min);

			if (attr & A_MAX)
				fprintf(stdout,
					" %s=%ld",
					_("max"), sp->sp_max);

			if (attr & A_WARN)
				fprintf(stdout,
					" %s=%ld",
					_("warn"), sp->sp_warn);

			if (attr & A_INACT)
				fprintf(stdout,
					" %s=%ld",
					_("inact"), sp->sp_inact);

			if (attr & A_LSTCHG)
				fprintf(stdout,
					" %s=%s",
					_("lstchg"), date);

			if (attr & A_EXPIRE)
				fprintf(stdout,
					" %s=%s",
					_("expire"), expires);

			if (attr & A_LOCKED)
				fprintf(stdout,
					" %s=%s",
					_("locked"),
					locked ? _("yes") : _("no"));
		} else if (args.colon) {
			fprintf(stdout,
				"#%s%s%s%s%s%s%s%s%s%s%s%s%s%s%s\n",
				_("name"),
				(attr & A_MIN) ? ":" : "",
				(attr & A_MIN) ? _("min") : "",
				(attr & A_MAX) ? ":" : "",
				(attr & A_MAX) ? _("max") : "",
				(attr & A_WARN) ? ":" : "",
				(attr & A_WARN) ? _("warn") : "",
				(attr & A_INACT) ? ":" : "",
				(attr & A_INACT) ? _("inact") : "",
				(attr & A_LSTCHG) ? ":" : "",
				(attr & A_LSTCHG) ? _("lstchg") : "",
				(attr & A_EXPIRE) ? ":" : "",
				(attr & A_EXPIRE) ? _("expire") : "",
				(attr & A_LOCKED) ? ":" : "",
				(attr & A_LOCKED) ? _("locked") : "");
			fprintf(stdout, "%s", sp->sp_namp);

			if (attr & A_MIN)
				fprintf(stdout, ":%ld", sp->sp_min);

			if (attr & A_MAX)
				fprintf(stdout, ":%ld", sp->sp_max);

			if (attr & A_WARN)
				fprintf(stdout, ":%ld", sp->sp_warn);

			if (attr & A_INACT)
				fprintf(stdout, ":%ld", sp->sp_inact);

			if (attr & A_LSTCHG)
				fprintf(stdout, ":%s", date);

			if (attr & A_EXPIRE)
				fprintf(stdout, ":%s", expires);

			if (attr & A_LOCKED)
				fprintf(stdout, ":%s",
					locked ? _("yes") : _("no"));
		} else {
			fprintf(stdout, "%s:", sp->sp_namp);

			if (attr & A_MIN)
				fprintf(stdout,
					"\n\t%s=%ld",
					_("min"), sp->sp_min);

			if (attr & A_MAX)
				fprintf(stdout,
					"\n\t%s=%ld",
					_("max"), sp->sp_max);

			if (attr & A_WARN)
				fprintf(stdout,
					"\n\t%s=%ld",
					_("warn"), sp->sp_warn);

			if (attr & A_INACT)
				fprintf(stdout,
					"\n\t%s=%ld",
					_("inact"), sp->sp_inact);

			if (attr & A_LSTCHG)
				fprintf(stdout,
					"\n\t%s=%s",
					_("lstchg"), date);

			if (attr & A_EXPIRE)
				fprintf(stdout,
					"\n\t%s=%s",
					_("expire"), expires);

			if (attr & A_LOCKED)
				fprintf(stdout,
					"\n\t%s=%s",
					_("locked"),
					locked ? _("yes") : _("no"));
		}

		fprintf(stdout, "\n");

		if (args.stanza)
			fprintf(stdout, "\n");
	}

EXIT:
	/* Free all allocated memory */
	strfreev(usrarray);
	free(expires);
	free(date);

	return status;
}
