/**
 * @file gpasswd.c
 * Change password for the specified group
 *
 * Copyright (C) 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <errno.h>
#include <stdio.h>

#include "misc.h"
#include "sysutils.h"
#include "libpasswd.h"

#define PRG_NAME "gpasswd"	/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("GROUP\n"
	   "-s");

/** Program synopsis */
static char doc[] =
	N_("Change the password of the specified group.");

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct setpasswd_options_t *args = state->input;

	arg = NULL;

	switch (key) {
	case ARGP_KEY_INIT:
		state->child_inputs[0] = args;
		break;

		if ((args->name == NULL) && (args->showhash == 0))
			argp_usage(state);
		break;

	default:
		return ARGP_ERR_UNKNOWN;
	}

	return 0;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	error_t status = 0;

	/* argp parser */
	static const struct argp_child child_parsers[] = {
		{ &setpasswd_argp, 0, 0, 0 },
		{ 0, 0, 0, 0 }
	};

	struct argp argp = {
		.options	= NULL,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
		.children	= child_parsers,
	};

	struct setpasswd_options_t args;

	argp_program_version_hook = version;

	errno = 0;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		return status;

	set_author_information(_("Written by David Weinehall, "
				 "Barry deFreese, "
				 "and Ben Asselstine.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		return status;
	}

	return setpasswd(args.name, 1, args.removepass, args.showhash);
}
