/**
 * @file chsh.c
 * Change login shell for the specified user
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <pwd.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "chsh"		/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("[USER]");

/** Program synopsis */
static char doc[] =
	N_("Change the login shell for the specified user.\n"
	   "\n"
	   "If no username is specified, "
	   "the information for the current user is changed.");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "shell", 's', N_("SHELL"), 0,
	  N_("The user's login shell"), 0 },
	{ "list-shells", 'l', 0, 0,
	  N_("List valid login shells"), 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	const char *user;		/**< User to edit */
	const char *shell;		/**< Shell to change to */
	int list;			/**< Set to 1 to list shells */
};

/**
 * Used to keep track of whether we're batch-editing
 * or doing interactive changes
 * 0 - interactive
 * 1 - batch
 */
static int batch = 0;

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 's':
		args->shell = arg;

		if (is_valid_shell(args->shell))
			argp_error(state, "invalid value supplied\n");

		batch = 1;
		break;

	case 'l':
		args->list = 1;
		break;

	case ARGP_KEY_INIT:
		args->user = NULL;
		args->shell = NULL;
		args->list = 0;
		break;

	case ARGP_KEY_ARG:
		if (args->user)
			argp_usage(state);

		args->user = arg;
		break;

	default:
		status = ARGP_ERR_UNKNOWN;
		break;
	}

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	FILE *pwrfp = NULL;
	FILE *pwwfp = NULL;
	struct passwd *pw;

	int empty = 1;

	error_t status = 0;

	int isadmin = 0;

	char *pwwname = NULL;
	char *pwbname = NULL;
	char *shell = NULL;
	char *username = NULL;

	mode_t oldmask;

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
	};

	struct arguments args;

	argp_program_version_hook = version;
	argp_err_exit_status = EINVAL;

	errno = 0;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Check if the caller has root privileges */
	if ((status = is_useradmin())) {
		if (status == EPERM) {
			status = 0;
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_useradmin", strerror(errno));
			goto EXIT;
		}
	} else {
		isadmin = 1;
	}

	/* If --list-shell is supplied, just list the available shells
	 * and exit
	 */
	if (args.list) {
		fprintf(stdout, _("\nValid login shells:\n\n"));
		setusershell();

		while ((shell = getusershell()))
			fprintf(stdout, "%s\n", shell);

		endusershell();
		goto EXIT;
	}

	/* If an argument is supplied, it is the name
	 * of the user to change the login shell for
	 */
	if (args.user) {
		if (!(username = strdup(args.user))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	/* If a shell was entered as an arg, make a copy
         * for consistency with interactive mode
         */
	if (args.shell) {
		if (!(shell = strdup(args.shell))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	/* Unless the user requests to change information about himself,
	 * or calls the program as root, deny the request
	 */
	if (username && !isadmin) {
		int retval = is_caller(username);

		if (retval == -1) {
			status = errno;
			goto EXIT;
		} else if (!retval) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "You must be a user-administrator to "
				  "%s %s\n"
				  "for other users than yourself.\n"),
				progname, _("change"), _("login shell"));
			status = EPERM;
			goto EXIT;
		}
	}

	/* If no username was provided, retrieve the name
	 * of the caller of the program
	 */
	if (!username) {
		if (!(username = get_username(getuid()))) {
			status = errno;
			goto EXIT;
		}
	}

	/* Verify that the user exists */
	if (!(pw = getpwnam(username)) && errno) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "getpwnam()", strerror(errno));
		status = errno;
		goto EXIT;
	}

	/* Abort on non-existing users */
	if (!pw) {
		fprintf(stderr,
			_("%s: the specified %s does not exist\n"),
			progname, _("user"));
		status = ENOENT;
		goto EXIT;
	}

	/* Make sure that the user is allowed to change his shell */
	if (!isadmin && is_listed_shell(pw->pw_shell)) {
		fprintf(stderr,
			_("%s: your current login shell is a restricted "
			  "shell; changing shell is not allowed\n"),
			progname);
		status = EPERM;
		goto EXIT;
	}

	/* Note: from this point on we know that username is valid,
	 * since it existed in the user database, hence we can
	 * print it without fear
	 */

	/* If we have no options, enter interactive mode */
	if (!batch) {
		fprintf(stdout,
			_("Changing the login shell for `%s'\n"
			  "Enter the new value, or press enter to keep "
			  "your old shell\n"),
			username);

		/* Note: From now on we assume that the old information
		 * is untainted; if not some other program is to blame
		 */
		while (!shell) {
			fprintf(stdout,
				_("\tLogin shell [%s]: "),
				pw->pw_shell);
			(void)fflush(stdout);

			if (!(shell = input_string(1))) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "input_string",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					break;
				}
			}

			if (!is_valid_shell(shell) && !is_listed_shell(shell))
				break;

			if (errno == ENOMEM || errno == EFAULT) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "is_valid_shell",
					strerror(errno));
				status = errno;
				goto EXIT;
			}

			/* If there was no error but the shell was invalid,
			 * we only allow the new shell if it was specified
			 * by a user-administrator
			 */
			if (isadmin)
				break;

			fprintf(stderr, _("\nInvalid shell specified\n\n"));

			free(shell);
			shell = NULL;
		}
	}

	/* No new shell specified, or the same as the old one;
	 * exit without changes
	 */
	if (!shell || !strcmp(shell, pw->pw_shell))
		goto EXIT;

	/* Create filename /etc/passwd.write */
	if (!(pwwname = create_filename(PASSWD_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/passwd- */
	if (!(pwbname = create_filename(PASSWD_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Change umask */
	oldmask = umask(0077);

	/* Acquire file locks */
	if ((status = lock_files()))
		goto EXIT;

	/* Open /etc/passwd */
	if (!(pwrfp = open_file(PASSWD_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/passwd to /etc/passwd- */
	if ((status = backup_file(PASSWD_FILE, pwbname)))
		goto EXIT2;

	/* Copy permissions from /etc/passwd to /etc/passwd- */
	if ((status = copy_file_modes(PASSWD_FILE, pwbname)))
		goto EXIT2;

	/* Open /etc/passwd.write */
	if (!(pwwfp = open_file(pwwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Perform changes on /etc/passwd */
	while ((pw = fgetpwent(pwrfp))) {
		static struct passwd pw2;

		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		pw2.pw_name = pw->pw_name;
		pw2.pw_passwd = pw->pw_passwd;
		pw2.pw_uid = pw->pw_uid;
		pw2.pw_gid = pw->pw_gid;
		pw2.pw_gecos = pw->pw_gecos;
		pw2.pw_dir = pw->pw_dir;

		/* If the entry is the user to edit, perform changes */
		pw2.pw_shell = strcmp(username, pw->pw_name) ? pw->pw_shell :
							       shell;

		/* Write the entry */
		if ((status = fputpwent(&pw2, pwwfp))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputpwent", strerror(errno));
			goto EXIT2;
		}
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetpwent()",
			strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/passwd.write */
	if ((status = close_file(&pwwfp)))
		goto EXIT2;

	/* Close /etc/passwd */
	if ((status = close_file(&pwrfp)))
		goto EXIT2;

	/* Everything is in order, move the new file in place */
	if ((status = replace_file(pwwname, PASSWD_FILE)))
		goto EXIT2;

	/* Set file permissions properly */
	if ((status = copy_file_modes(pwbname, PASSWD_FILE)))
		goto EXIT2;

EXIT2:
	/* Restore umask */
	umask(oldmask);

	/* These files might not exist, but that's ok */
	status = unlink_file(pwwname, status);

	/* Release file locks */
	status = unlock_files(status);

EXIT:
	free(pwwname);
	free(pwbname);
	free(username);
	free(shell);

	return status;
}
