/**
 * @file chgrpmem.c
 * Change administrators or members for the specified groups
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <grp.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gshadow.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "chgrpmem"	/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("GROUPS");

/** Program synopsis */
static char doc[] =
	N_("Change the administrators or members for the specified groups.\n"
	   "\n"
	   "GROUPS should be a comma-separated list of groups "
	   "or the word ALL to list all groups.\n"
	   "\n"
	   "Changes are done by specifying an action (+/-/=)"
	   "and a comma-separated list of users.");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "adms", 'a', N_("(+/-/=)[ADMS]"), 0,
	  N_("Add/remove/set group administrators"), 0 },
	{ "users", 'm', N_("(+/-/=)[USERS]"), 0,
	  N_("Add/remove/set group members"), 0 },
	{ "verbose", 'v', 0, 0,
	  N_("Warn if the specified groups does not exist"), -2 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	const char *groups;	/**< Comma-separated list of groups to modify */
	const char *addadmlist;	/**< List of group admins to add */
	const char *remadmlist;	/**< List of group admins to remove */
	const char *setadmlist;	/**< Exact list of group admins */
	const char *addmemlist;	/**< List of group members to add */
	const char *remmemlist;	/**< List of group members to remove */
	const char *setmemlist;	/**< Exact list of group members */
	int verbose;		/**< Warn about non-existing groups */
};

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'a':
		switch (arg[0]) {
		case '+':
			if (args->addadmlist)
				argp_error(state,
					   _("you can only specify the list "
					     "of admins to add once"));

			args->addadmlist = arg + 1;
			break;

		case '-':
			if (args->remadmlist)
				argp_error(state,
					   _("you can only specify the list "
					     "of admins to remove once"));

			args->remadmlist = arg + 1;
			break;

		case '=':
			if (args->setadmlist)
				argp_error(state,
					   _("you can only specify the list "
					     "of admins once"));

			args->setadmlist = arg + 1;
			break;

		default:
			argp_error(state,
				   _("the argument to `-a' must be either of "
				     "`+', `-', or `=', optionally followed "
				     "followed by a comma-"
				     "separated list of users"));
		}

		if ((args->addadmlist || args->remadmlist) &&
		    args->setadmlist)
			argp_error(state,
				   _("you cannot combine either of "
				     "`+' or `-' with `='"));

		if ((status = is_valid_namelist_empty(arg + 1))) {
			if (status == EINVAL)
				argp_error(state,
					   _("list contains malformed "
					     "usernames"));

			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_valid_namelist",
				strerror(status));
			goto EXIT;
		}

		break;

	case 'm':
		switch (arg[0]) {
		case '+':
			if (args->addmemlist)
				argp_error(state,
					   _("you can only specify the list "
					     "of members to add once"));

			args->addmemlist = arg + 1;
			break;

		case '-':
			if (args->remmemlist)
				argp_error(state,
					   _("you can only specify the list "
					     "of members to remove once"));

			args->remmemlist = arg + 1;
			break;

		case '=':
			if (args->setmemlist)
				argp_error(state,
					   _("you can only specify the list "
					     "of members once"));

			args->setmemlist = arg + 1;
			break;

		default:
			argp_error(state,
				   _("the argument to `-m' must be either of "
				     "`+', `-', or `=', optionally followed "
				     "followed by a comma-"
				     "separated list of users"));
		}

		if ((args->addmemlist || args->remmemlist) &&
		    args->setmemlist)
			argp_error(state,
				   _("you cannot combine either of "
				     "`+' or `-' with `='"));

		if ((status = is_valid_namelist_empty(arg + 1))) {
			if (status == EINVAL)
				argp_error(state,
					   _("list contains malformed "
					     "usernames"));

			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_valid_namelist",
				strerror(status));
			goto EXIT;
		}

		break;

	case 'v':
		args->verbose = 1;
		break;

	case ARGP_KEY_INIT:
		args->groups = NULL;
		args->addadmlist = NULL;
		args->remadmlist = NULL;
		args->setadmlist = NULL;
		args->addmemlist = NULL;
		args->remmemlist = NULL;
		args->setmemlist = NULL;
		args->verbose = 0;
		break;

	case ARGP_KEY_ARG:
		if (args->groups)
			argp_usage(state);

		args->groups = arg;
		break;

	case ARGP_KEY_FINI:
		if (!args->addadmlist && !args->addmemlist &&
		    !args->remadmlist && !args->remmemlist &&
		    !args->setadmlist && !args->setmemlist)
			argp_usage(state);
		break;

	case ARGP_KEY_NO_ARGS:
		argp_usage(state);
		break;

	default:
		status = ARGP_ERR_UNKNOWN;
		break;
	}

EXIT:

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	FILE *grrfp = NULL;
	FILE *grwfp = NULL;
	struct group *gr;

	FILE *sgrfp = NULL;
	FILE *sgwfp = NULL;
	struct sgrp *sg;

	int empty = 1;
	int changed = 0;

	error_t status = 0;

	char *grwname = NULL;
	char *grbname = NULL;
	char *sgwname = NULL;
	char *sgbname = NULL;

	char **grparray = NULL;
	char **addadms = NULL;
	char **remadms = NULL;
	char **setadms = NULL;
	char **addmems = NULL;
	char **remmems = NULL;
	char **setmems = NULL;

	gid_t i; /* We're scanning <= LASTGID, hence gid_t */

	mode_t oldmask;

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
	};

	struct arguments args;

	argp_program_version_hook = version;
	argp_err_exit_status = EINVAL;

	errno = 0;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	if (args.addadmlist && !(addadms = strsplit(args.addadmlist, ",", 0))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strsplit", strerror(errno));
		status = errno;
		goto EXIT;
	}

	if (args.remadmlist && !(remadms = strsplit(args.remadmlist, ",", 0))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strsplit", strerror(errno));
		status = errno;
		goto EXIT;
	}

	if (args.setmemlist && !(setmems = strsplit(args.setmemlist, ",", 0))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strsplit", strerror(errno));
		status = errno;
		goto EXIT;
	}

	if (args.addmemlist && !(addmems = strsplit(args.addmemlist, ",", 0))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strsplit", strerror(errno));
		status = errno;
		goto EXIT;
	}

	if (args.remmemlist && !(remmems = strsplit(args.remmemlist, ",", 0))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strsplit", strerror(errno));
		status = errno;
		goto EXIT;
	}

	if (args.setadmlist && !(setadms = strsplit(args.setadmlist, ",", 0))) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "strsplit", strerror(errno));
		status = errno;
		goto EXIT;
	}

	if (!is_distinct_array(addadms, remadms) ||
	    !is_distinct_array(addmems, remmems)) {
		fprintf(stderr,
			_("%s: the list of members or administrators "
			  "to add must be\n"
			  "distinct from the list of members "
			  "or administrators to remove\n"),
			progname);
		status = EINVAL;

		goto EXIT;
	}

	/* There are two alternatives here, neither of which are really
	 * pretty; either to read the entire group file once to get
	 * all groupnames, then use them for the ALL list, or to
	 * have separate code for the ALL case and the case of separate
	 * group-entries.  Since the latter is probably the most common,
	 * the latter has been chosen.
	 */
	if (!strcmp(argv[argc - 1], "ALL")) {
		char *tmp = NULL;

		if (!(tmp = get_all_groups())) {
			status = errno;
		} else if (!strlen(tmp)) {
			fprintf(stderr,
				_("%s: could not find any %s; the %s "
				  "might be corrupt\n"),
				progname, _("groups"), _("group database"));
			status = ENOENT;
		} else if (!(grparray = strsplit(tmp, ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;
		}

		free(tmp);

		if (status)
			goto EXIT;
	} else {
		char *tmp = NULL;
		i = 0;

		if (!(grparray = strsplit(argv[argc - 1], ",", 0))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strsplit", strerror(errno));
			status = errno;

			goto EXIT;
		}

		/* If verbose mode has been requested,
		 * warn about all non-existing groups
		 */
		while (args.verbose && (tmp = grparray[i++])) {
			if (!getgrnam(tmp)) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "getgrnam()",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					fprintf(stderr,
						_("%s: warning: %s `%s' "
						  "does not exist\n"),
						progname, _("group"),
						tmp);
				}
			}
		}
	}

	/* Only group-administrators are allowed to change the lists
	 * of members and administrators for a group
	 */
	for (i = 0; grparray[i]; i++) {
		if ((status = is_groupadmin(grparray[i])) == EPERM) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "Only group-administrators may %s %s.\n"),
				progname, _("change"),
				_("the group member list"));

			goto EXIT;
		} else if (status) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_groupadmin", strerror(errno));

			goto EXIT;
		}
	}

	/* Verify that all group members exist */
	if (addmems) {
		uid_t j;

		for (j = 0; addmems[j]; j++) {
			if (!(status = is_free_username(addmems[j]))) {
				fprintf(stderr,
					_("%s: one or several specified "
					  "%s does not exist\n"),
					progname, _("group members"));
				status = ENOENT;
				goto EXIT;
			} else if (errno) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "is_free_username",
					strerror(errno));
				status = errno;
				goto EXIT;
			}
		}
	}

	if (setmems && setmems[0][0] != '\0') {
		uid_t j;

		for (j = 0; setmems[j]; j++) {
			if (!(status = is_free_username(setmems[j]))) {
				fprintf(stderr,
					_("%s: one or several specified "
					  "%s does not exist\n"),
					progname, _("group members"));
				status = ENOENT;

				goto EXIT;
			} else if (errno) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "is_free_username",
					strerror(errno));
				status = errno;

				goto EXIT;
			}
		}
	}

	/* Verify that all group administrators exist */
	if (addadms) {
		uid_t j;

		for (j = 0; addadms[j]; j++) {
			if (!(status = is_free_username(addadms[j]))) {
				fprintf(stderr,
					_("%s: one or several specified "
					  "%s does not exist\n"),
					progname, _("group administrators"));
				status = ENOENT;

				goto EXIT;
			} else if (errno) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "is_free_username",
					strerror(errno));
				status = errno;

				goto EXIT;
			}
		}
	}

	if (setmems && setmems[0][0] != '\0') {
		uid_t j;

		for (j = 0; setadms[j]; j++) {
			if (!(status = is_free_username(setadms[j]))) {
				fprintf(stderr,
					_("%s: one or several specified "
					  "%s does not exist\n"),
					progname, _("group administrators"));
				status = ENOENT;

				goto EXIT;
			} else if (errno) {
				fprintf(stderr,
					_("%s: `%s' failed; %s\n"),
					progname, "is_free_username",
					strerror(errno));
				status = errno;

				goto EXIT;
			}
		}
	}


	/* Create filename /etc/group.write */
	if (!(grwname = create_filename(GROUP_FILE, WRITE_EXT))) {
		status = errno;

		goto EXIT;
	}

	/* Create filename /etc/group- */
	if (!(grbname = create_filename(GROUP_FILE, BACKUP_EXT))) {
		status = errno;

		goto EXIT;
	}

	/* Create filename /etc/gshadow.write */
	if (!(sgwname = create_filename(GSHADOW_FILE, WRITE_EXT))) {
		status = errno;

		goto EXIT;
	}

	/* Create filename /etc/gshadow- */
	if (!(sgbname = create_filename(GSHADOW_FILE, BACKUP_EXT))) {
		status = errno;

		goto EXIT;
	}

	/* Acquire file locks */
	if ((status = lock_files()))
		goto EXIT;

	/* Change umask */
	oldmask = umask(0077);

	/* Open /etc/group */
	if (!(grrfp = open_file(GROUP_FILE, "r"))) {
		status = errno;

		goto EXIT2;
	}

	/* Backup /etc/group to /etc/group- */
	if ((status = backup_file(GROUP_FILE, grbname)))
		goto EXIT2;

	/* Copy permissions from /etc/group to /etc/group- */
	if ((status = copy_file_modes(GROUP_FILE, grbname)))
		goto EXIT2;

	/* Open /etc/group.write */
	if (!(grwfp = open_file(grwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Open /etc/gshadow */
	if (!(sgrfp = open_file(GSHADOW_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/gshadow to /etc/gshadow- */
	if ((status = backup_file(GSHADOW_FILE, sgbname)))
		goto EXIT2;

	/* Copy permissions from /etc/gshadow to /etc/gshadow- */
	if ((status = copy_file_modes(GSHADOW_FILE, sgbname)))
		goto EXIT2;

	/* Open /etc/gshadow.write */
	if (!(sgwfp = open_file(sgwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Perform changes on /etc/group */
	while ((gr = fgetgrent(grrfp))) {
		static struct group gr2;
		char **newmems = NULL;

		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* Copy the old entry */
		gr2.gr_name = gr->gr_name;
		gr2.gr_passwd = gr->gr_passwd;
		gr2.gr_gid = gr->gr_gid;

		/* If the entry is part of the array of groups to modify,
		 * modify its member-list; if not, copy the old values
		 */
		if (is_in_array(grparray, gr->gr_name)) {
			char **tmp = NULL;

			if (addmems || remmems) {
				tmp = array_union(gr->gr_mem, addmems);
				newmems = array_cut(tmp, remmems);
				strfreev(tmp);

				/* error-check for array_union and array_cut */
				if ((!tmp || !newmems) && errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname,
						!tmp ? "array_union" :
						       "array_cut",
						strerror(errno));
					status = errno;
					goto EXIT;
				}

				gr2.gr_mem = newmems;
				changed = 1;
			} else if (setmems) {
				gr2.gr_mem = setmems;
				changed = 1;
			} else {
				gr2.gr_mem = gr->gr_mem;
			}
		} else {
			gr2.gr_mem = gr->gr_mem;
		}

		/* Write the entry */
		status = fputgrent(&gr2, grwfp);

		strfreev(newmems);

		if (status) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputgrent", strerror(errno));
			goto EXIT2;
		}
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetgrent()",
			strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/group.write */
	if ((status = close_file(&grwfp)))
		goto EXIT2;

	/* Close /etc/group */
	if ((status = close_file(&grrfp)))
		goto EXIT2;

	/* Perform changes on /etc/gshadow */
	while ((sg = fgetsgent(sgrfp))) {
		static struct sgrp sg2;
		char **newadms = NULL;
		char **newmems = NULL;

		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		/* Copy the old entry */
		sg2.sg_name = sg->sg_name;
		sg2.sg_passwd = sg->sg_passwd;

		/* If the entry is part of the array of groups to modify,
		 * modify its member- and admin-list; if not,
		 * copy the old values
		 */
		if (is_in_array(grparray, sg->sg_name)) {
			char **tmp = NULL;

			if (addadms || remadms) {
				tmp = array_union(sg->sg_adm, addadms);
				newadms = array_cut(tmp, remadms);
				strfreev(tmp);

				/* error-check for array_union and array_cut */
				if ((!tmp || !newadms) && errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname,
						!tmp ? "array_union" :
						       "array_cut",
						strerror(errno));
					status = errno;
					goto EXIT;
				}

				sg2.sg_adm = newadms;
				changed = 1;
			} else if (setadms) {
				sg2.sg_adm = setadms;
				changed = 1;
			} else {
				sg2.sg_adm = sg->sg_adm;
			}

			if (addmems || remmems) {
				tmp = array_union(sg->sg_mem, addmems);
				newmems = array_cut(tmp, remmems);
				strfreev(tmp);

				/* error-check for array_union and array_cut */
				if ((!tmp || !newmems) && errno) {
					strfreev(newadms);
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname,
						!tmp ? "array_union" :
						       "array_cut",
						strerror(errno));
					status = errno;
					goto EXIT;
				}

				sg2.sg_mem = newmems;
				changed = 1;
			} else if (setmems) {
				sg2.sg_mem = setmems;
				changed = 1;
			} else {
				sg2.sg_mem = sg->sg_mem;
			}
		} else {
			sg2.sg_adm = sg->sg_adm;
			sg2.sg_mem = sg->sg_mem;
		}

		/* Write the entry */
		status = fputsgent(&sg2, sgwfp);

		strfreev(newadms);
		strfreev(newmems);

		if (status) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputsgent", strerror(errno));
			goto EXIT2;
		}
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetsgent",
			strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/gshadow.write */
	if ((status = close_file(&sgwfp)))
		goto EXIT2;

	/* Close /etc/gshadow */
	if ((status = close_file(&sgrfp)))
		goto EXIT2;

	/* If nothing has changed, don't replace old files */
	if (!changed)
		goto EXIT2;

	/* Everything is in order, move the new files in place */
	if ((status = replace_file(grwname, GROUP_FILE)))
		goto EXIT2;

	if ((status = replace_file(sgwname, GSHADOW_FILE)))
		goto EXIT2;

	/* Set file permissions properly */
	if ((status = copy_file_modes(grbname, GROUP_FILE)))
		goto EXIT2;

	if ((status = copy_file_modes(sgbname, GSHADOW_FILE)))
		goto EXIT2;

EXIT2:
	/* Restore umask */
	umask(oldmask);

	/* These files might not exist, but that's ok */
	status = unlink_file(grwname, status);
	status = unlink_file(sgwname, status);

	/* Release file locks */
	status = unlock_files(status);

EXIT:
	/* Free all allocated memory */
	strfreev(grparray);
	strfreev(addadms);
	strfreev(remadms);
	strfreev(setadms);
	strfreev(addmems);
	strfreev(remmems);
	strfreev(setmems);
	free(grwname);
	free(grbname);
	free(sgwname);
	free(sgbname);

	return status;
}
