/**
 * @file chfn.c
 * Change comments in the GECOS-field for the specified user
 *
 * Copyright (C) 2002, 2003, 2004 David Weinehall
 * Copyright (C) 2004, 2006 Free Software Foundation, Inc.
 *
 *  This file is part of GNU Sysutils
 *
 *  GNU Sysutils is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  GNU Sysutils is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <argp.h>
#include <pwd.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "misc.h"
#include "sysutils.h"

#define PRG_NAME "chfn"		/**< Name shown by --help etc */

extern const char *progname;	/**< Used to store the name of the program */

/** Address to send bug-reports to */
const char *argp_program_bug_address = PACKAGE_BUGREPORT;

/** Usage information */
static char args_doc[] =
	N_("[USER]");

/** Program synopsis */
static char doc[] =
	N_("Change full name and other information for a user.\n"
	   "\n"
	   "If no username is specified, "
	   "the information for the current user is changed.");

/** Structure with the available command line options */
static struct argp_option options[] = {
	{ "fname", 'f', N_("NAME"), 0,
	  N_("The user's full name"), 0 },
	{ "hphone", 'h', N_("PHONE#"), 0,
	  N_("The user's home phone"), 0 },
	{ "other", 'o', N_("OTHER"), 0,
	  N_("Other information"), 0 },
	{ "room", 'r', N_("ROOM"), 0,
	  N_("The user's room number"), 0 },
	{ "wphone", 'w', N_("PHONE#"), 0,
	  N_("The user's work phone"), 0 },
	{ 0, 0, 0, 0, 0, 0 }
};

/** Structure to hold output from argument parser */
struct arguments {
	const char *user;	/**< User to modify */
	const char *fname;	/**< New full name */
	const char *room;	/**< New room number */
	const char *wphone;	/**< New work phone */
	const char *hphone;	/**< New home phone */
	const char *other;	/**< New other information */
};

/** Keeps track of what values the user has permissions to change */
static unsigned int chperms = 0;

/**
 * Used to keep track of whether we're batch-editing
 * or doing interactive changes
 * 0 - interactive
 * 1 - batch
 */
static int batch = 0;

/**
 * Parse a single option
 *
 * @param key The option
 * @param arg The argument for the option
 * @param state The state of argp
 * @return 0 on success,
 *         ARGP_ERR_UNKNOWN on failure
 */
static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	struct arguments *args = state->input;
	error_t status = 0;

	switch (key) {
	case 'f':
		if (is_valid_gecos_field(arg))
			argp_error(state, "invalid value supplied");

		args->fname = arg;
		batch = 1;
		break;

	case 'h':
		if (is_valid_gecos_field(arg))
			argp_error(state, "invalid value supplied");

		args->hphone = arg;
		batch = 1;
		break;

	case 'o':
		if (is_valid_gecos_field(arg))
			argp_error(state, "invalid value supplied");

		args->other = arg;
		batch = 1;
		break;

	case 'r':
		if (is_valid_gecos_field(arg))
			argp_error(state, "invalid value supplied");

		args->room = arg;
		batch = 1;
		break;

	case 'w':
		if (is_valid_gecos_field(arg))
			argp_error(state, "invalid value supplied");

		args->wphone = arg;
		batch = 1;
		break;

	case ARGP_KEY_INIT:
		args->user = NULL;
		args->fname = NULL;
		args->hphone = NULL;
		args->other = NULL;
		args->room = NULL;
		args->wphone = NULL;
		break;

	case ARGP_KEY_ARG:
		if (args->user)
			argp_usage(state);

		args->user = arg;
		break;

	default:
		status = ARGP_ERR_UNKNOWN;
		break;
	}

	return status;
}

/**
 * The program's main-function
 *
 * @param argc The number of arguments
 * @param argv The arguments
 * @return 0 on success, errno on failure
 */
int main(int argc, char *argv[])
{
	FILE *pwrfp = NULL;
	FILE *pwwfp = NULL;
	struct passwd *pw;

	int empty = 1;

	error_t status = 0;
	int isadmin = 0;

	char *pwwname = NULL;
	char *pwbname = NULL;
	char *fname = NULL;
	char *room = NULL;
	char *wphone = NULL;
	char *hphone = NULL;
	char *other = NULL;
	char *newgecos = NULL;
	char *username = NULL;

	char **gecosv = NULL;

	mode_t oldmask;

	chperms = CH_ALL;

	/* argp parser */
	struct argp argp = {
		.options	= options,
		.parser		= parse_opt,
		.args_doc	= args_doc,
		.doc		= doc,
	};

	struct arguments args;

	argp_program_version_hook = version;
	argp_err_exit_status = EINVAL;

	errno = 0;

	/* Initialise support for locales, and set the program-name */
	if ((status = init_locales(PRG_NAME)))
		goto EXIT;

	set_author_information(_("Written by David Weinehall.\n"));

	/* Parse command line */
	if ((status = argp_parse(&argp, argc, argv, 0, 0, &args))) {
		if (status != EINVAL)
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "argp_parse()", strerror(status));

		goto EXIT;
	}

	/* Only user-admins are allowed to change the Other field */
	if ((status = is_useradmin())) {
		if (status == EPERM) {
			status = 0;
		} else {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_useradmin", strerror(errno));
			goto EXIT;
		}
	} else {
		isadmin = 1;
		chperms |= CH_OTHER;
	}

	/* If an argument is supplied, it is the name
	 * of the user to change GECOS field comments for
	 */
	if (args.user) {
		if (!(username = strdup(args.user))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	/* strdup all indata, since we do not want to create separate
	 * code-paths for interactively changed values and command-line
	 * supplied values
	 */
	if (args.fname) {
		if (!(fname = strdup(args.fname))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	if (args.hphone) {
		if (!(hphone = strdup(args.hphone))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	if (args.other) {
		if (!(other = strdup(args.other))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	if (args.room) {
		if (!(room = strdup(args.room))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	if (args.wphone) {
		if (!(wphone = strdup(args.wphone))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	/* Unless the user requests to change information about himself,
	 * or calls the program as root, deny the request
	 */
	if (username && !isadmin) {
		int retval = is_caller(username);

		if (retval == -1) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "is_caller", strerror(errno));
			status = errno;
			goto EXIT;
		} else if (!retval) {
			fprintf(stderr,
				_("%s: insufficient privileges\n"
				  "You must be a user-administrator to "
				  "%s %s\n"
				  "for other users than yourself.\n"),
				progname, _("change"), _("GECOS information"));
			status = EPERM;
			goto EXIT;
		}
	}

	/* If no username was provided, retrieve the name
	 * of the caller of the program
	 */
	if (!username) {
		if (!(username = get_username(getuid()))) {
			status = errno;
			goto EXIT;
		}
	}

	/* Make sure the user is allowed to perform the requested
	 * changes
	 */
	if (!chperms ||
	    (!(chperms & CH_FNAME) && args.fname) ||
	    (!(chperms & CH_ROOM) && args.room) ||
	    (!(chperms & CH_WPHONE) && args.wphone) ||
	    (!(chperms & CH_HPHONE) && args.hphone) ||
	    (!(chperms & CH_OTHER) && args.other)) {
		fprintf(stderr,
			_("%s: insufficient privileges to change "
			  "GECOS information\n"),
			progname);
		status = EPERM;
		goto EXIT;
	}

	/* Verify that the user exists */
	if (!(pw = getpwnam(username)) && errno) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "getpwnam()", strerror(errno));
		status = errno;
		goto EXIT;
	}

	/* Abort on non-existing users */
	if (!pw) {
		fprintf(stderr,
			_("%s: the specified %s does not exist\n"),
			progname, _("user"));
		status = ENOENT;
		goto EXIT;
	}

	/* Note: from this point on we know that username is valid,
	 * since it existed in the user database, hence we can
	 * print it without fear
	 */

	/* Split the gecos-information into an array */
	if (!(gecosv = split_gecos(pw->pw_gecos))) {
		status = errno;
		goto EXIT;
	}

	/* If we have no options, enter interactive mode */
	if (!batch) {
		fprintf(stdout,
			_("Changing the user information for `%s'\n"
			  "Enter the new value, or press enter to keep "
			  "the old value\n"),
			username);

		/* Note: From now on we assume that the old information
		 * is untainted; if not some other program is to blame
		 */
		while (chperms & CH_FNAME) {
			fprintf(stdout,
				_("\tFull Name [%s]: "),
				gecosv[GE_FNAME]);
			(void)fflush(stdout);

			if (!(fname = input_string(1))) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "input_string",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					break;
				}
			}

			if (!is_valid_gecos_field(fname))
				break;

			fprintf(stderr, _("\nInvalid input\n\n"));

			free(fname);
			fname = NULL;
		}

		while (chperms & CH_ROOM) {
			fprintf(stdout,
				_("\tRoom Number [%s]: "),
				gecosv[GE_ROOM]);
			(void)fflush(stdout);

			if (!(room = input_string(1))) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "input_string",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					break;
				}
			}

			if (!is_valid_gecos_field(room))
				break;

			fprintf(stderr, _("\nInvalid input\n\n"));

			free(room);
			room = NULL;
		}

		while (chperms & CH_WPHONE) {
			fprintf(stdout,
				_("\tWork Phone [%s]: "),
				gecosv[GE_WPHONE]);
			(void)fflush(stdout);

			if (!(wphone = input_string(1))) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "input_string",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					break;
				}
			}

			if (!is_valid_gecos_field(wphone))
				break;

			fprintf(stderr, _("\nInvalid input\n\n"));

			free(wphone);
			wphone = NULL;
		}

		while (chperms & CH_HPHONE) {
			fprintf(stdout,
				_("\tHome Phone [%s]: "),
				gecosv[GE_HPHONE]);
			(void)fflush(stdout);

			if (!(hphone = input_string(1))) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "input_string",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					break;
				}
			}

			if (!is_valid_gecos_field(hphone))
				break;

			fprintf(stderr, _("\nInvalid input\n\n"));

			free(hphone);
			hphone = NULL;
		}

		while (chperms & CH_OTHER) {
			fprintf(stdout,
				_("\tOther [%s]: "),
				gecosv[GE_OTHER]);
			(void)fflush(stdout);

			if (!(other = input_string(1))) {
				if (errno) {
					fprintf(stderr,
						_("%s: `%s' failed; %s\n"),
						progname, "input_string",
						strerror(errno));
					status = errno;
					goto EXIT;
				} else {
					break;
				}
			}

			if (!is_valid_gecos_other(other))
				break;

			fprintf(stderr, _("\nInvalid input\n\n"));

			free(other);
			other = NULL;
		}
	}

	/* No new information specified, or the same as the old one;
	 * exit without changes
	 */
	if ((!fname || !strcmp(fname, gecosv[GE_FNAME])) &&
	    (!room || !strcmp(room, gecosv[GE_ROOM])) &&
	    (!wphone || !strcmp(wphone, gecosv[GE_WPHONE])) &&
	    (!hphone || !strcmp(hphone, gecosv[GE_HPHONE])) &&
	    (!other || !strcmp(other, gecosv[GE_OTHER])))
		goto EXIT;

	/* If fname is missing, copy the old string */
	if (!fname) {
		if (!(fname = strdup(gecosv[GE_FNAME]))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	/* If room is missing, copy the old string */
	if (!room) {
		if (!(room = strdup(gecosv[GE_ROOM]))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	/* If wphone is missing, copy the old string */
	if (!wphone) {
		if (!(wphone = strdup(gecosv[GE_WPHONE]))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	/* If hphone is missing, copy the old string */
	if (!hphone) {
		if (!(hphone = strdup(gecosv[GE_HPHONE]))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	/* If other is missing, copy the old string */
	if (!other) {
		if (!(other = strdup(gecosv[GE_OTHER]))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "strdup()", strerror(errno));
			status = errno;
			goto EXIT;
		}
	}

	/* Join all gecos strings */
	if (!(newgecos = join_gecos(fname, room, wphone, hphone, other))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/passwd.write */
	if (!(pwwname = create_filename(PASSWD_FILE, WRITE_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Create filename /etc/passwd- */
	if (!(pwbname = create_filename(PASSWD_FILE, BACKUP_EXT))) {
		status = errno;
		goto EXIT;
	}

	/* Acquire file locks */
	if ((status = lock_files()))
		goto EXIT;

	/* Change umask */
	oldmask = umask(0077);

	/* Open /etc/passwd */
	if (!(pwrfp = open_file(PASSWD_FILE, "r"))) {
		status = errno;
		goto EXIT2;
	}

	/* Backup /etc/passwd to /etc/passwd- */
	if ((status = backup_file(PASSWD_FILE, pwbname)))
		goto EXIT2;

	/* Copy permissions from /etc/passwd to /etc/passwd- */
	if ((status = copy_file_modes(PASSWD_FILE, pwbname)))
		goto EXIT2;

	/* Open /etc/passwd.write */
	if (!(pwwfp = open_file(pwwname, "w"))) {
		status = errno;
		goto EXIT2;
	}

	/* Perform changes on /etc/passwd */
	while ((pw = fgetpwent(pwrfp))) {
		static struct passwd pw2;

		/* Set as an indication that the file has at least 1 entry */
		empty = 0;

		pw2.pw_name = pw->pw_name;
		pw2.pw_passwd = pw->pw_passwd;
		pw2.pw_uid = pw->pw_uid;
		pw2.pw_gid = pw->pw_gid;

		/* If the entry is the user to edit, perform changes */
		pw2.pw_gecos = strcmp(username, pw->pw_name) ? pw->pw_gecos :
							       newgecos;

		pw2.pw_dir = pw->pw_dir;
		pw2.pw_shell = pw->pw_shell;

		/* Write the entry */
		if ((status = fputpwent(&pw2, pwwfp))) {
			fprintf(stderr,
				_("%s: `%s' failed; %s\n"),
				progname, "fputpwent", strerror(errno));
			goto EXIT2;
		}
	}

	/* Make sure no errors occured */
	if (errno && (errno != ENOENT || empty)) {
		fprintf(stderr,
			_("%s: `%s' failed; %s\n"),
			progname, "fgetpwent()",
			strerror(errno));
		status = errno;
		goto EXIT2;
	}

	if (errno == ENOENT)
		errno = 0;

	/* Close /etc/passwd.write */
	if ((status = close_file(&pwwfp)))
		goto EXIT2;

	/* Close /etc/passwd */
	if ((status = close_file(&pwrfp)))
		goto EXIT2;

	/* Everything is in order, move the new file in place */
	if ((status = replace_file(pwwname, PASSWD_FILE)))
		goto EXIT2;

	/* Set file permissions properly */
	if ((status = copy_file_modes(pwbname, PASSWD_FILE)))
		goto EXIT2;

EXIT2:
	/* Restore umask */
	umask(oldmask);

	/* This file might not exist, but that's ok */
	status = unlink_file(pwwname, status);

	/* Release file locks */
	status = unlock_files(status);

EXIT:
	strfreev(gecosv);
	free(pwwname);
	free(pwbname);
	free(fname);
	free(room);
	free(wphone);
	free(hphone);
	free(other);
	free(newgecos);
	free(username);

	return status;
}
