/*  Synapse 0.2
 *  Copyright (C) 2006/2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "synapse.h"

/**
	Libera dalla memoria un descrittore di tab, questa funzione e' da
	invocare quando una tab viene chiusa

	@param	the_tab	Riferimento alla struttura di tipo PageDesc da
			liberare dalla memoria
	@param	data	Non usato
*/
void free_page_desc_for_tab ( gpointer the_tab, gpointer data ) {
	PageDesc *tab;

	tab = ( PageDesc* ) the_tab;

	if ( tab->query )
		g_free ( tab->query );

	if ( tab->status )
		g_free ( tab->status );

	gtk_tree_store_clear ( tab->icons );
	g_object_unref ( G_OBJECT ( tab->icons ) );
	g_free ( tab );
}

/**
	Permette di rimuovere la tab descritta dalla struttura qui passata.
	La funzione non effettua l'operazione se nell'applicazione c'e' una
	sola tab aperta

	@param	tab	Riferimento alla tab da chiudere
*/
void remove_tab ( PageDesc *tab ) {
	if ( g_list_length ( Graph.tabs ) == 1 )
		return;

	gtk_notebook_remove_page ( GTK_NOTEBOOK ( Graph.notebook ), g_list_index ( Graph.tabs, tab ) );
	Graph.tabs = g_list_remove ( Graph.tabs, tab );
	free_page_desc_for_tab ( ( gpointer ) tab, NULL );
}

/**
	Wrapper alla funzione remove_tab(), da usare come callback per il
	pulsante che, nel menu contestuale delle tab, permette di chiudere la
	tab selezionata

	@param	item	Riferimento al pulsante del menu contestuale la cui
			attivazione invoca questa callback
	@param	data	Riferimento alla tab da rimuovere

	@return		FALSE
*/
gboolean wrapper_menu_remove_tab ( GtkMenuItem *item, gpointer data ) {
	remove_tab ( ( PageDesc* ) data );
	return FALSE;
}

/**
	Permette di rinominare una tab

	@param	item	Riferimento al pulsante del menu contestuale la cui
			attivazione invoca questa callback
	@param	data	Riferimento alla tab da rinominare

	@return		FALSE
*/
gboolean rename_tab ( GtkMenuItem *item, gpointer data ) {
	const gchar *text;
	GtkWidget *dialog;
	GtkWidget *name;
	PageDesc *tab;

	tab = ( PageDesc* ) data;

	dialog = gtk_dialog_new_with_buttons ( _( "Tab name" ), GTK_WINDOW ( Graph.main_win ), GTK_DIALOG_MODAL,
	                                       GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT, GTK_STOCK_OK, GTK_RESPONSE_ACCEPT, NULL );

	name = gtk_entry_new_with_max_length ( MAX_TAB_NAME );
	gtk_entry_set_text ( GTK_ENTRY ( name ), tab->label );
	gtk_box_pack_start ( GTK_BOX ( GTK_DIALOG ( dialog )->vbox ), name, FALSE, FALSE, 5 );
	gtk_widget_show_all ( dialog );

	if ( gtk_dialog_run ( GTK_DIALOG ( dialog ) ) == GTK_RESPONSE_ACCEPT ) {
		text = gtk_entry_get_text ( GTK_ENTRY ( name ) );
		if ( !strlen ( text ) )
			return FALSE;

		( void ) snprintf ( tab->label, MAX_TAB_NAME, "%s", text );
		gtk_button_set_label ( GTK_BUTTON ( tab->button ), tab->label );
	}

	gtk_widget_destroy ( dialog );
	return FALSE;
}

/**
	Apre il menu contestuale che appare cliccando sulla linguetta che
	identifica una tab

	@param	button	Riferimento al pulsante che ha prodotto l'invocazione
			della callback
	@param	ev	Rappresentazione dell'evento verificatosi
	@param	data	Riferimento alla tab per cui il menu contestuale e'
			stato aperto

	@return		FALSE
*/
gboolean tab_context_menu ( GtkWidget *button, GdkEventButton *ev, gpointer data ) {
	int pos;
	GtkWidget *tab_menu;
	GtkWidget *item;

	if ( ev->button == 1 ) {
		pos = g_list_index ( Graph.tabs, data );
		if ( gtk_notebook_get_current_page ( GTK_NOTEBOOK ( Graph.notebook ) ) != pos )
			gtk_notebook_set_current_page ( GTK_NOTEBOOK ( Graph.notebook ), pos );
	}

	if ( ev->button == 3 ) {
		tab_menu = gtk_menu_new ();

		item = gtk_menu_item_new_with_label ( _( "Remove Tab" ) );
		g_signal_connect ( G_OBJECT ( item ), "activate", G_CALLBACK ( wrapper_menu_remove_tab ), data );
		gtk_menu_shell_append ( GTK_MENU_SHELL ( tab_menu ), item );

		item = gtk_menu_item_new_with_label ( _( "Rename Tab" ) );
		g_signal_connect ( G_OBJECT ( item ), "activate", G_CALLBACK ( rename_tab ), data );
		gtk_menu_shell_append ( GTK_MENU_SHELL ( tab_menu ), item );

		gtk_widget_show_all ( tab_menu );
		gtk_menu_popup ( GTK_MENU ( tab_menu ), NULL, NULL, NULL, NULL, 0, ev->time );
	}

	return FALSE;
}

/**
	Funzione usata per l'ordinamento degli elementi nelle
	rappresentazioni ad icone

	@param	model	Riferimento al modello da ordinare
	@param	a	Riferimento ad un elemento che gia' si trova
			all'interno del modello
	@param	b	Riferimento all'elemento da inserire ordinatamente
			nel modello
	@param	user_data	Non usato

	@return		0 se i due elementi sono uguali, <= 1 se il primo e'
			minore, >= 1 se e' minore il secondo
*/
static gint sort_func ( GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b, gpointer user_data ) {
	gchar *name_a, *name_b;
	int ret;

	gtk_tree_model_get ( model, a, ICON_VIEW_NAME_COL, &name_a, -1 );
	gtk_tree_model_get ( model, b, ICON_VIEW_NAME_COL, &name_b, -1 );
	ret = g_utf8_collate ( name_a, name_b );
	g_free ( name_a );
	g_free ( name_b );
	return ret;
}

/**
	Callback invocata ogni volta che si cambia la pagina attiva nel
	notebook primario dell'applicazione, provvede a modificare
	l'indicazione della query corrente nell'apposita barra (le pagine nel
	notebook sono molte, ma la barra e' sempre una e va aggiornata)

	@param	notebook	Riferimento al notebook primario
				dell'applicazione
	@param	page	Riferimento alla nuova pagina attivata
	@param	pos	Posizione della nuova pagina attivata
	@param	data	Non usato

	@return		FALSE
*/
gboolean notebook_page_switched ( GtkNotebook *notebook, GtkNotebookPage *page, guint pos, gpointer data ) {
	PageDesc *tab;

	tab = ( PageDesc* ) g_list_nth_data ( Graph.tabs, pos );
	Graph.running = tab;

	if ( tab->query )
		gtk_entry_set_text ( query_label_reference (), tab->query );
	else
		gtk_entry_set_text ( query_label_reference (), "" );

	return FALSE;
}

/**
	Aggiorna il riferimento alla query corrente nella struttura
	descrittiva della tab attiva

	@param	combo	La combo box che ospita l'ultima query eseguita, cfr.
			GraphicTemplate::query_text
	@param	data	Non usato

	@return		FALSE
*/
gboolean update_tab_query ( GtkComboBox *combo, gpointer data ) {
	if ( Graph.running->query )
		g_free ( Graph.running->query );

	Graph.running->query = g_strdup ( gtk_entry_get_text ( query_label_reference () ) );
	return FALSE;
}

/**
*/
void oms_text_action_performed ( GtkWidget *widget, gpointer attach ) {
	UINT64 fileid;
	gchar *tmp;
	GList *list;
	GList *iter;

	list = kiazma_icon_stacked_view_get_selected_items ( KIAZMA_ICON_STACKED_VIEW ( widget ) );
	tmp = g_string_free ( ( GString* ) attach, FALSE );

	for ( iter = g_list_first ( list ); iter; iter = g_list_next ( iter ) ) {
		fileid_by_path ( iter->data, &fileid );
		hyppo_vfs_setxattr ( fileid, META_ANNOTATIONS, tmp, 0 );
	}

	g_free ( tmp );

	g_list_foreach ( list, gtk_tree_path_free, NULL );
	g_list_free ( list );
}

/**
	Alloca ed inizializza una nuova tab nel sistema, aggiungendola
	automaticamente alla schermata
*/
void create_tab () {
	PageDesc *nuovo_tab;

	nuovo_tab = g_new ( PageDesc, 1 );

	if ( Graph.tabs )
		Graph.tabs = g_list_append ( Graph.tabs, nuovo_tab );

	else {
		Graph.tabs = g_list_alloc ();
		Graph.tabs->data = nuovo_tab;
	}

	nuovo_tab->query = NULL;
	nuovo_tab->status = NULL;

	nuovo_tab->icons = gtk_tree_store_new ( 3, G_TYPE_STRING, GDK_TYPE_PIXBUF, G_TYPE_UINT64 );
	gtk_tree_sortable_set_default_sort_func ( GTK_TREE_SORTABLE ( nuovo_tab->icons ), sort_func, NULL, NULL );
	gtk_tree_sortable_set_sort_column_id ( GTK_TREE_SORTABLE ( nuovo_tab->icons ),
	                                       GTK_TREE_SORTABLE_DEFAULT_SORT_COLUMN_ID, GTK_SORT_ASCENDING );

	nuovo_tab->view = kiazma_icon_stacked_view_new ();
	kiazma_icon_stacked_view_set_reorderable ( KIAZMA_ICON_STACKED_VIEW ( nuovo_tab->view ), FALSE );
	kiazma_icon_stacked_view_set_model ( KIAZMA_ICON_STACKED_VIEW ( nuovo_tab->view ), GTK_TREE_MODEL ( nuovo_tab->icons ) );
	kiazma_icon_stacked_view_set_selection_mode ( KIAZMA_ICON_STACKED_VIEW ( nuovo_tab->view ), GTK_SELECTION_MULTIPLE );
	gtk_widget_set_size_request ( nuovo_tab->view, MAIN_WINDOW_WIDTH, MAIN_WINDOW_HEIGHT - 50 );
	kiazma_icon_stacked_view_set_text_column ( KIAZMA_ICON_STACKED_VIEW ( nuovo_tab->view ), ICON_VIEW_NAME_COL );
	kiazma_icon_stacked_view_set_pixbuf_column ( KIAZMA_ICON_STACKED_VIEW ( nuovo_tab->view ), ICON_VIEW_ICON_COL );
	kiazma_icon_stacked_view_set_id_column ( KIAZMA_ICON_STACKED_VIEW ( nuovo_tab->view ), ICON_VIEW_ID_COL );
	g_signal_connect ( nuovo_tab->view, "button-press-event", G_CALLBACK ( button_press_event ), NULL );
	g_signal_connect ( nuovo_tab->view, "item-activated", G_CALLBACK ( wrapper_view_open_file ), NULL );
	g_signal_connect ( nuovo_tab->view, "oms_selection_text_closed", G_CALLBACK ( oms_text_action_performed ), NULL );

	( void ) snprintf ( nuovo_tab->label, MAX_TAB_NAME, "%d", g_list_length ( Graph.tabs ) );
	nuovo_tab->button = gtk_button_new_with_label ( nuovo_tab->label );
	g_signal_connect ( G_OBJECT ( nuovo_tab->button ), "button-press-event",
	                   G_CALLBACK ( tab_context_menu ), ( gpointer ) nuovo_tab );

	gtk_notebook_append_page ( GTK_NOTEBOOK ( Graph.notebook ), nuovo_tab->view, nuovo_tab->button );
	gtk_widget_show_all ( Graph.notebook );
}
