/*  Synapse 0.1
 *  Copyright (C) 2006 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "synapse.h"

GraphicTemplate		Graph;												/**< Istanza dell'applicazione principale, qui si trovano tutte le informazioni relative a Synapse */
gchar			ReverseTranslations [ METADATA_END ] [ MAX_LENGHT_FOR_METADATA_NAME ];				/**< Matrice di stringhe usata per le traduzioni inverse. @see init_reverse_translations() */
char			*MountedPath									= NULL;

/**
	Libera dalla memoria gli elementi dinamicamente allocati
	dell'applicazione, ed esce
*/
void destroy_all () {
	gtk_widget_destroy ( Graph.main_win );
	g_list_foreach ( Graph.tabs, free_page_desc_for_tab, NULL );
	g_list_free ( Graph.tabs );

	g_bookmark_file_free ( Bookmarks );
	if ( Uris )
		g_strfreev ( Uris );

	if ( MountedPath )
		g_free ( MountedPath );

	close_actions ();
	exit ( 0 );
}

/**
	Utility to update the string to show in the status bar

	@param	string		New string to view in the bar
*/
inline void update_status_bar ( gchar *string ) {
	guint id;

	id = gtk_statusbar_get_context_id ( GTK_STATUSBAR ( Graph.status ), "query" );
	gtk_statusbar_push ( GTK_STATUSBAR ( Graph.status ), id, string );
}

/**
	Update the progression bar on bottom of the window

	@param 	parts		Number of all parts to rappresent
	@param	completed	Number of completed parts
*/
inline void update_status_progress ( int parts, int completed ) {
	gdouble tot;

	if ( !parts )
		gtk_progress_bar_set_fraction ( GTK_PROGRESS_BAR ( Graph.progress ), 0 );

	else {
		tot = 1 / parts;
		gtk_progress_bar_set_fraction ( GTK_PROGRESS_BAR ( Graph.progress ), tot * ( ( gdouble ) completed ) );
	}
}

/**
	Utility to refresh the icons list and pass directly to the execution
	of the new query
*/
inline void update_main_window () {
	gtk_tree_store_clear ( Graph.running->icons );
	run_query ( gtk_combo_box_get_active_text ( GTK_COMBO_BOX ( Graph.query_text ) ) );
}

/**
	Usata per prelevare la query editata con l'editor ed immetterla nella
	apposita barra, pronta per essere eseguita
*/
void convert_query () {
	gchar *query;

	query = kiazma_query_editor_get_query ( KIAZMA_QUERY_EDITOR ( Graph.composer ) );
	if ( query ) {
		gtk_entry_set_text ( query_label_reference (), query );
		g_free ( query );
	}
}

/**
	View the query composer over the main window
*/
void show_composer () {
	kiazma_query_editor_reset ( KIAZMA_QUERY_EDITOR ( Graph.composer ) );
	gtk_widget_show_all ( Graph.composer );

	switch ( gtk_dialog_run ( GTK_DIALOG ( Graph.composer ) ) ) {
		case GTK_RESPONSE_OK:
			gtk_entry_set_text ( query_label_reference (), "" );
			convert_query ();
			break;

		default:
			break;
	}

	gtk_widget_hide ( Graph.composer );
}

/**
*/
gboolean wrapper_menu_show_composer ( GtkMenuItem *item, gpointer data ) {
	show_composer ();
	return FALSE;
}

/**
	Aggiunge un nuovo file all'interno del filesystem Hyppocampus,
	permettendo all'utente di selezionare un file nel suo filesystem reale
	e di indicizzarlo nel filesystem relazionale virtuale. Qui viene
	gestito il pannello di selezione del file originario, l'operazione
	di creazione del nuovo elemento in Hyppocampus, la copia del contenuto
	e l'aggiornamento della vista ad icone
*/
void add_file_into_hyppocampus () {
	char *content;
	gchar *name;
	gchar *path;
	FILE *original;
	struct stat status;
	GtkWidget *chooser;

	chooser = gtk_file_chooser_dialog_new ( _( "Choose Files" ), GTK_WINDOW ( Graph.main_win ), GTK_FILE_CHOOSER_ACTION_OPEN,
	                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL );
	gtk_file_chooser_set_current_folder ( GTK_FILE_CHOOSER ( chooser ), getenv ( "HOME" ) );
	gtk_file_chooser_set_local_only ( GTK_FILE_CHOOSER ( chooser ), TRUE );

	if ( gtk_dialog_run ( GTK_DIALOG ( chooser ) ) == GTK_RESPONSE_ACCEPT ) {
		path = gtk_file_chooser_get_filename ( GTK_FILE_CHOOSER ( chooser ) );

		stat ( path, &status );
		if ( ( original = fopen ( path, "r" ) ) == NULL ) {
			xfce_err ( _( "Unable to open file %s" ), path );
			g_free ( path );
			return;
		}

		if ( ( content = calloc ( status.st_size, sizeof ( char ) ) ) == NULL )
			g_free ( path );
			return;

		fread ( content, sizeof ( char ), status.st_size, original );
		fclose ( original );

		name = g_path_get_basename ( path );
		g_free ( path );
		original = hyppo_vfs_create ( name );

		if ( original == NULL ) {
			xfce_err ( _( "Unable to fill file for %s" ), name );
			g_free ( name );
			return;
		}

		fwrite ( content, sizeof ( char ), status.st_size, original );

		if ( ferror ( original ) )
			xfce_warn ( _( "Unable to write contents for file %s\nFile in Hyppocampus will be empty" ), name );

		fclose ( original );
		free ( content );
		g_free ( name );

		update_main_window ();
	}

	gtk_widget_destroy ( chooser );
}

/**
	Wrapper alla funzione add_file_into_hyppocampus(), da usare come
	callback per il pulsante che, nel menu contestuale principale,
	permette di aggiungere un nuovo file nello spazio relazionale di
	Hyppocampus

	@param	item	Pulsante nel menu contestuale generico cui e'
			assegnata la callback
	@param	data	Non usato

	@return		FALSE
*/
gboolean wrapper_menu_add_file_into_hyppocampus ( GtkMenuItem *item, gpointer data ) {
	add_file_into_hyppocampus ();
	return FALSE;
}

/**
*/
gboolean wrapper_button_start_query ( GtkButton *button, gpointer data  ) {
	start_query ();
	return FALSE;
}

/**
*/
gboolean wrapper_menu_show_preferences ( GtkMenuItem *item, gpointer data ) {
	show_preferences ();
	return FALSE;
}

/**
	Callback assegnata al pulsante che, nel menu contestuale principale,
	permette di accedere alla pagina di informazioni sull'applicazione

	@param	item	Riferimento al pulsante nel menu contestuale generico
			cui e' assegnata la callback
	@param	data	Non usato

	@return		FALSE
*/
gboolean show_about ( GtkMenuItem *item, gpointer data ) {
	const gchar *me []	= { "Roberto -MadBob- Guido", NULL };
	FILE *license;
	gchar *GPL;
	struct stat license_file;
	GtkWidget *about;
	GError *error		= NULL;

	about = gtk_about_dialog_new ();
	gtk_about_dialog_set_name ( GTK_ABOUT_DIALOG ( about ), "Synapse" );
	gtk_about_dialog_set_website ( GTK_ABOUT_DIALOG ( about ), "http://synapse.berlios.de" );
	gtk_about_dialog_set_logo ( GTK_ABOUT_DIALOG ( about ), gdk_pixbuf_new_from_file ( SYNAPSE_IMAGES_PATH "logo.png", &error ) );
	gtk_about_dialog_set_authors ( GTK_ABOUT_DIALOG ( about ), me );
	gtk_about_dialog_set_copyright ( GTK_ABOUT_DIALOG ( about ), "(C) 2006 Roberto Guido" );

	license = fopen ( SYNAPSE_IMAGES_PATH "COPYING", "r" );
	if ( license ) {
		stat ( SYNAPSE_IMAGES_PATH "COPYING", &license_file );
		GPL = g_new ( gchar, license_file.st_size );
		fread ( GPL, sizeof ( gchar ), license_file.st_size, license );
		gtk_about_dialog_set_license ( GTK_ABOUT_DIALOG ( about ), GPL );
		fclose ( license );
	}

	( void ) gtk_dialog_run ( GTK_DIALOG ( about ) );
	gtk_widget_destroy ( about );

	return FALSE;
}

/**
	Build the Synapse application menu, to attach to the main menu. It
	contains the reference to the preferences panel and the about panel

	@return		The submenu to attach
*/
GtkWidget* do_application_menu () {
	GtkWidget *first;
	GtkWidget *menu;
	GtkWidget *item;

	first = gtk_menu_item_new_with_label ( _( "Synapse" ) );
	gtk_widget_show ( first );

	menu = gtk_menu_new ();

	item = gtk_menu_item_new_with_label ( _( "Preferences" ) );
	g_signal_connect ( G_OBJECT ( item ), "activate", G_CALLBACK ( wrapper_menu_show_preferences ), NULL );
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	item = gtk_menu_item_new_with_label ( _( "About Synapse" ) );
	g_signal_connect ( G_OBJECT ( item ), "activate", G_CALLBACK ( show_about ), NULL );
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	gtk_menu_item_set_submenu ( GTK_MENU_ITEM ( first ), menu );
	return first;
}

/**
	Costruisce il menu contestuale principale dell'applicazione, quello
	che compare premendo il tasto destro del mouse in un punto qualunque
	della finestra lontano dalle icone dei files

	@return		Puntatore al menu creato
*/
GtkWidget* do_main_menu () {
	GtkWidget *menu;
	GtkWidget *item;

	menu = gtk_menu_new ();

	item = gtk_menu_item_new_with_label ( _( "Add files" ) );
	g_signal_connect ( G_OBJECT ( item ), "activate", G_CALLBACK ( wrapper_menu_add_file_into_hyppocampus ), NULL );
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	item = gtk_menu_item_new_with_label ( _( "Query Editor" ) );
	g_signal_connect ( G_OBJECT ( item ), "activate", G_CALLBACK ( wrapper_menu_show_composer ), NULL );
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	item = do_bookmarks_menu ();
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	item = do_application_menu ();
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	return menu;
}

/**
	Callback che permette di aprire un file in base al suo tipo MIME

	@param	item	Riferimento al pulsante del menu contestuale per i
			file cui e' associata questa callback
	@param	data	GtkTreePath che rappresenta il file su cui agire

	@return		FALSE

	@todo		Scrivere questa funzione dopo essersi studiati un
			meccanismo di gestione per le informazioni relative ai
			tipi MIME
*/
gboolean wrapper_menu_open_file ( GtkMenuItem *item, gpointer data ) {
	return FALSE;
}

/**
	Se non richiesto diversamente nella configurazione dell'applicazione,
	costruisce il pannello di conferma per la cancellazione del file / dei
	files indicati

	@param	path	Riferimento (relativo al modello di elementi corrente)
			all'elemento da eliminare, o NULL se si vogliono
			eliminare piu' files insieme (in tal caso, "paths"
			deve essere != NULL)
	@param	paths	Lista di GtkTreePath's che si riferiscono agli
			elementi da rimuovere tutti insieme con una sola
			conferma. Affinche' questo parametro venga
			contemplato, "path" deve essere == NULL

	@return		TRUE se la cancellazione viene confermata dall'utente
			o se, per configurazione, non viene richiesta alcuna
			conferma, FALSE altrimenti
*/
gboolean dialog_ask_remove_confirmation ( GtkTreePath *path, GList *paths ) {
	gchar *fs_name;
	GString *string;
	gboolean confirm;
	GtkWidget *dialog;
	GtkTreeIter iter;
	GList *paths_iter;

	confirm = TRUE;

	if ( Confs.ask_remove_confirm ) {
		if ( path == NULL && paths == NULL )
			return FALSE;

		if ( path ) {
			string = g_string_new ( _( "Are you sure to remove file:\n" ) );

			gtk_tree_model_get_iter ( GTK_TREE_MODEL ( Graph.running->icons ), &iter, path );
			gtk_tree_model_get ( GTK_TREE_MODEL ( Graph.running->icons ), &iter, ICON_VIEW_NAME_COL, &fs_name, -1 );
			g_string_append_printf ( string, "%s\n", fs_name );
			g_free ( fs_name );
		}

		else if ( paths ) {
			string = g_string_new ( _( "Are you sure to remove files:\n" ) );
			paths_iter = paths;

			while ( paths_iter ) {
				gtk_tree_model_get_iter ( GTK_TREE_MODEL ( Graph.running->icons ), &iter, paths_iter->data );
				gtk_tree_model_get ( GTK_TREE_MODEL ( Graph.running->icons ), &iter,
							ICON_VIEW_NAME_COL, &fs_name, -1 );
				g_string_append_printf ( string, "%s\n", fs_name );
				paths_iter = g_list_next ( paths_iter );
				g_free ( fs_name );
			}
		}

		else
			return FALSE;

		fs_name = g_string_free ( string, FALSE );
		dialog = gtk_message_dialog_new ( GTK_WINDOW ( Graph.main_win ), GTK_DIALOG_MODAL,
							GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, fs_name );
		gtk_widget_show_all ( dialog );

		if ( gtk_dialog_run ( GTK_DIALOG ( dialog ) ) == GTK_RESPONSE_NO )
			confirm = FALSE;

		gtk_widget_destroy ( dialog );
	}

	return confirm;
}

/**
	Rimuove un file

	@param	path	Riferimento all'elemento da eliminare
	@param	ask_confirm	TRUE se si desidera l'invocazione del pannello
				di conferma costruito in
				dialog_ask_remove_confirmation() prima di
				effettuare la cancellazione
*/
void remove_file ( GtkTreePath *path, gboolean ask_confirm ) {
	gboolean confirm;
	UINT64 fileid;
	GtkTreeIter iter;

	if ( !gtk_tree_model_get_iter ( GTK_TREE_MODEL ( Graph.running->icons ), &iter, path ) ) {
		xfce_err ( _( "Unable to get file name" ) );
		return;
	}

	confirm = TRUE;

	if ( ask_confirm )
		confirm = dialog_ask_remove_confirmation ( path, NULL );

	if ( confirm ) {
		gtk_tree_model_get ( GTK_TREE_MODEL ( Graph.running->icons ), &iter, ICON_VIEW_ID_COL, &fileid, -1 );
		hyppo_vfs_remove ( fileid );
		update_main_window ();
	}

	return;
}

/**
	Wrapper della funzione remove_file(), da usare come callback per il
	pulsante nel menu contestuale dedicato ai singoli files che permette
	di rimuovere il file selezionato

	@param	item	Riferimento al pulsante del menu contestuale cui e'
			assegnata la callback
	@param	data	Riferimento all'elemento da rimuovere, espresso come
			GtkTreePath

	@return		FALSE
*/
gboolean wrapper_menu_remove_file ( GtkMenuItem *item, gpointer data ) {
	remove_file ( ( GtkTreePath* ) data, TRUE );
	return FALSE;
}

/**
	Costruisce il pannello delle proprieta' per un file. All'interno della
	funzione si provvede gia' ad aggiornare i valori assegnati ai metadati
	qualora venissero modificati dall'utente e confermati

	@param	path	GtkTreePath riferita all'elemento di cui mostrare le
			proprieta'
*/
void show_file_properties ( GtkTreePath *path ) {
	UINT64 id;
	GtkWidget *panel;
	GtkTreeIter iter;

	if ( !gtk_tree_model_get_iter ( GTK_TREE_MODEL ( Graph.running->icons ), &iter, path ) ) {
		xfce_err ( _( "Unable to get file ID" ) );
		return;
	}

	gtk_tree_model_get ( GTK_TREE_MODEL ( Graph.running->icons ), &iter, ICON_VIEW_ID_COL, &id, -1 );
	panel = synapse_file_panel_new ( id );
	gtk_widget_show_all ( panel );
	gtk_dialog_run ( GTK_DIALOG ( panel ) );
	synapse_file_panel_destroy ( SYNAPSE_FILE_PANEL ( panel ) );
}

/**
	Wrapper della funzione show_file_properties(), da assegnare come
	callback al pulsante del menu contestuale dedicato ai files che
	permette di visualizzare il pannello di informazioni sul file
	selezionato

	@param	item	Riferimento al pulsante del menu contestuale cui e'
			assegnata questa callback
	@param	data	Riferimento all'elemento di cui visualizzare le
			informazioni, in forma di GtkTreePath

	@return		FALSE
*/
gboolean wrapper_menu_show_file_properties ( GtkMenuItem *item, gpointer data ) {
	show_file_properties ( ( GtkTreePath* ) data );
	return FALSE;
}

/**
	Costruisce il menu contestuale per il file su cui e' stato premuto il
	tasto destro del mouse

	@param	path	Riferimento al file selezionato

	@return		Il menu contestuale
*/
GtkWidget* do_element_menu ( GtkTreePath *path ) {
	UINT64 metaid;
	UINT64 fileid;
	GtkWidget *menu;
	GtkWidget *item;
	GtkWidget *submenu;
	GtkWidget *subitem;
	GList *attrs;
	GList *iter;

	menu = gtk_menu_new ();

	item = gtk_menu_item_new_with_label ( _( "Open" ) );
	gtk_widget_set_sensitive ( item, FALSE );
	g_signal_connect ( G_OBJECT ( item ), "activate", G_CALLBACK ( wrapper_menu_open_file ), ( gpointer ) path );
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	item = gtk_menu_item_new_with_label ( _( "Select by metadata" ) );
	submenu = gtk_menu_new ();
	fileid_by_path ( path, &fileid );

	attrs = hyppo_vfs_listxattr ( fileid );

	/**
		Sarebbe interessante non limitare la ricerca per metadato
		uguale ad un solo valore ma a valori multipli: organizzando
		questo menu con GtkCheckMenuItem e mettendo un pulsante di
		conferma al fondo, si potrebbero selezionare piu' voci che
		devono matchare nella ricerca e dare poi la conferma per
		l'esecuzione della query
	*/
	for ( iter = g_list_first ( attrs ); iter; iter = g_list_next ( iter ) ) {
		metaid = *( ( UINT64* ) iter->data );
		subitem = gtk_menu_item_new_with_label ( hyppo_metadata_translate_in_string ( metaid ) );
		g_signal_connect ( G_OBJECT ( subitem ), "activate", G_CALLBACK ( do_collective_query ), ( gpointer ) path );
		gtk_widget_show ( subitem );
		gtk_menu_shell_append ( GTK_MENU_SHELL ( submenu ), subitem );
	}

	gtk_menu_item_set_submenu ( GTK_MENU_ITEM ( item ), submenu );
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	item = gtk_menu_item_new_with_label ( _( "Delete" ) );
	g_signal_connect ( G_OBJECT ( item ), "activate", G_CALLBACK ( wrapper_menu_remove_file ), ( gpointer ) path );
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	if ( Confs.enable_actions )
		add_menu_actions ( menu, fileid, attrs );

	item = gtk_menu_item_new_with_label ( _( "Properties" ) );
	g_signal_connect ( G_OBJECT ( item ), "activate", G_CALLBACK ( wrapper_menu_show_file_properties ), ( gpointer ) path );
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	hyppo_vfs_free_xattrs ( attrs );
	Graph.current_file = fileid;
	return menu;
}

/**
	Handler per gli eventi di pressione dei tasti del mouse

	@param	icon_view	Riferimento alla vista ad icone su cui e'
				stato premuto il tasto del mouse
	@param	ev		Struttura riassuntiva dell'evento prodotto
	@param	data		Non usato

	@return			FALSE
*/
gboolean button_press_event ( GtkWidget *icon_view, GdkEventButton *ev, gpointer data ) {
	GtkTreePath *path;

	if ( ev->button == 1 ) {
	}

	else if ( ev->button == 3 ) {
		if ( ( path = kiazma_icon_stacked_view_get_path_at_pos ( KIAZMA_ICON_STACKED_VIEW ( icon_view ), ev->x, ev->y ) ) ) {
			kiazma_icon_stacked_view_unselect_all ( KIAZMA_ICON_STACKED_VIEW ( icon_view ) );
			kiazma_icon_stacked_view_select_path ( KIAZMA_ICON_STACKED_VIEW ( icon_view ), path );
			Graph.element_menu = do_element_menu ( path );
			gtk_menu_popup ( GTK_MENU ( Graph.element_menu ), NULL, NULL, NULL, NULL, 0, ev->time );
		}

		else {
			kiazma_icon_stacked_view_unselect_all ( KIAZMA_ICON_STACKED_VIEW ( icon_view ) );
			Graph.main_menu = do_main_menu ();
			gtk_menu_popup ( GTK_MENU ( Graph.main_menu ), NULL, NULL, NULL, NULL, 0, ev->time );
		}
	}

	return FALSE;
}

/**
	Wrapper della funzione remove_file() che viene richiamata quando viene
	eseguita la shortcut per la rimozione dei files selezionati

	@param	item	Riferimento all'elemento da rimuovere, in forma di
			GtkTreePath
	@param	data	Non usato
*/
void wrapper_shortcut_remove_file ( gpointer item, gpointer data ) {
	remove_file ( ( GtkTreePath* ) item, FALSE );
}

/**
	Handler degli eventi di pressione tasti, qui vengono intercettate le
	shortcut da tastiera

	@param	icon_view	Riferimento alla vista ad icone attiva mentre
				veniva eseguita la shortcut
	@param	ev		Struttura riassuntiva dell'evento prodotto
	@param	data		Non usato

	@return			FALSE
*/
gboolean key_press_event ( GtkWidget *icon_view, GdkEventKey *ev, gpointer data ) {
	int i;
	int index;
	gboolean did;
	GList *files;

	for ( i = 0, did = FALSE; i < S_MAX_SHORTCUTS && !did; i++ ) {
		if ( Confs.shortcuts [ i ].modifier == ev->state && Confs.shortcuts [ i ].key == ev->keyval ) {
			switch ( i ) {
				case S_ADD_FILE:
					add_file_into_hyppocampus ();
					did = TRUE;
					break;

				case S_REMOVE_FILE:
					did = TRUE;
					files = kiazma_icon_stacked_view_get_selected_items ( KIAZMA_ICON_STACKED_VIEW ( Graph.running->view ) );
					if ( !files )
						break;

					/**
						@todo	Rivedere l'intrico di funzioni e wrapper dedicati alla
							rimozione dei files che non l'ho capito manco io...
					*/

					if ( dialog_ask_remove_confirmation ( NULL, files ) )
						g_list_foreach ( files, wrapper_shortcut_remove_file, NULL );

					g_list_foreach ( files, gtk_tree_path_free, NULL );
					g_list_free ( files );
					break;

				case S_FILE_INFO:
					files = kiazma_icon_stacked_view_get_selected_items ( KIAZMA_ICON_STACKED_VIEW ( Graph.running->view ) );
					if ( !files )
						break;

					show_file_properties ( ( GtkTreePath* ) g_list_nth_data ( files, 0 ) );
					g_list_foreach ( files, gtk_tree_path_free, NULL );
					g_list_free ( files );
					did = TRUE;
					break;

				case S_SELECT_ALL:
					kiazma_icon_stacked_view_select_all ( KIAZMA_ICON_STACKED_VIEW ( Graph.running->view ) );
					did = TRUE;
					break;

				case S_OPEN_NEW_TAB:
					create_tab ();
					did = TRUE;
					break;

				case S_CLOSE_CURRENT_TAB:
					remove_tab ( ( PageDesc* ) g_list_nth_data ( Graph.tabs,
					             gtk_notebook_get_current_page ( GTK_NOTEBOOK ( Graph.notebook ) ) ) );
					did = TRUE;
					break;

				case S_PREV_TAB:
					gtk_notebook_set_current_page ( GTK_NOTEBOOK ( Graph.notebook ),
					                    gtk_notebook_get_current_page ( GTK_NOTEBOOK ( Graph.notebook ) ) - 1 );
					did = TRUE;
					break;

				case S_NEXT_TAB:
					index = gtk_notebook_get_current_page ( GTK_NOTEBOOK ( Graph.notebook ) );

					if ( index == gtk_notebook_get_n_pages ( GTK_NOTEBOOK ( Graph.notebook ) ) - 1 )
						gtk_notebook_set_current_page ( GTK_NOTEBOOK ( Graph.notebook ), 0 );
					else
						gtk_notebook_set_current_page ( GTK_NOTEBOOK ( Graph.notebook ), index + 1 );

					did = TRUE;
					break;

				case S_SWITCH_ADDRESS_BAR:
					gtk_widget_grab_focus ( Graph.query_text );
					break;

				case S_SWITCH_KEYWORD_SEARCH:
					gtk_widget_grab_focus ( Graph.keywords_text );
					break;

				case S_HISTORY:
					gtk_combo_box_popup ( GTK_COMBO_BOX ( Graph.query_text ) );
					gtk_widget_grab_focus ( Graph.query_text );
					break;

				case S_BOOKMARK_THIS:
					add_current_to_bookmarks ();
					break;

				case S_PREFERENCES_PANEL:
					show_preferences ();
					break;

				case S_QUERY_EDITOR:
					show_composer ();
					break;

				default:
					break;
			}
		}
	}

	return FALSE;
}

/**
	Wrapper della funzione open_file() usato come handler dell'attivazione
	di un elemento all'interno delle viste ad icone

	@param	icon_view	Riferimento alla vista ad icone da cui
				proviene l'evento
	@param	path		Riferimento all'elemento attivato
	@param	data		Non usato

	@return			FALSE

	@todo			Scrivere questa funzione quando si sara'
				messo a punto il sistema di gestione delle
				informazioni sui tipi MIME
*/
gboolean wrapper_view_open_file ( GtkWidget *icon_view, GtkTreePath *path, gpointer data ) {
	printf ( "item attivato\n" );
	return FALSE;
}

/**
*/
gboolean do_start_query ( GtkWidget *widget, GdkEventKey *event, gpointer useless ) {
	if ( event->keyval == GDK_Return )
		start_query ();

	return FALSE;
}

/**
	Inizializza la barra degli strumenti che appare nella parte alta
	della finestra dell'applicazione
*/
void init_toolbar () {
	GtkListStore *store;

	Graph.toolbar = gtk_hbox_new ( FALSE, 3 );
	gtk_box_pack_start ( GTK_BOX ( Graph.main_box ), Graph.toolbar, FALSE, FALSE, 0 );

	Graph.query_text = gtk_combo_box_entry_new_text ();
	set_tooltip ( Graph.query_text, _( "Write here you query" ), S_SWITCH_ADDRESS_BAR );
	g_signal_connect ( G_OBJECT ( Graph.query_text ), "changed", G_CALLBACK ( update_tab_query ), NULL );
	gtk_box_pack_start ( GTK_BOX ( Graph.toolbar ), Graph.query_text, TRUE, TRUE, 0 );

	/**
		@todo	The buttons have to be placed in a KiazmaToolbar
			(to implement)
	*/

	Graph.query_clear = kiazma_button_stock_new ( GTK_STOCK_CLEAR );
	g_signal_connect ( G_OBJECT ( Graph.query_clear ), "clicked", G_CALLBACK ( clear_query ), NULL );
	gtk_box_pack_start ( GTK_BOX ( Graph.toolbar ), Graph.query_clear, FALSE, FALSE, 0 );

	Graph.query_start = kiazma_button_stock_new ( GTK_STOCK_OK );
	g_signal_connect ( G_OBJECT ( Graph.query_start ), "clicked", G_CALLBACK ( start_query ), NULL );
	gtk_widget_add_accelerator ( Graph.query_start, "clicked", Graph.accel_group, GDK_Return, 0, 0 );
	gtk_box_pack_start ( GTK_BOX ( Graph.toolbar ), Graph.query_start, FALSE, FALSE, 0 );

	Graph.keywords_text = gtk_entry_new ();
	set_tooltip ( Graph.keywords_text, _( "Search files by keyword" ), S_SWITCH_KEYWORD_SEARCH );
	g_signal_connect ( G_OBJECT ( Graph.keywords_text ), "changed", G_CALLBACK ( incremental_keyword_search ), NULL );

	Graph.keywords_list = gtk_entry_completion_new ();
	gtk_entry_completion_set_minimum_key_length ( Graph.keywords_list, MIN_LENGHT_FOR_KEYWORD_SEARCH );
	g_signal_connect ( G_OBJECT ( Graph.keywords_list ), "match-selected", G_CALLBACK ( open_file_selected_by_keyword ), NULL );
	gtk_entry_set_completion ( GTK_ENTRY ( Graph.keywords_text ), Graph.keywords_list );
	gtk_entry_completion_set_popup_set_width ( Graph.keywords_list, TRUE );
	gtk_entry_completion_set_popup_completion ( Graph.keywords_list, TRUE );
	gtk_entry_completion_set_text_column ( Graph.keywords_list, 0 );
	store = gtk_list_store_new ( 1, G_TYPE_STRING );
	gtk_entry_completion_set_model ( Graph.keywords_list, ( GtkTreeModel* ) store );
	gtk_box_pack_start ( GTK_BOX ( Graph.toolbar ), Graph.keywords_text, FALSE, FALSE, 0 );
}

/**
	Inizializza l'editor grafico per la costruzione delle query
*/
void init_query_composer () {
	Graph.composer = kiazma_query_editor_new ( KIAZME_QUERY_EDITOR_EASY );
}

/**
	Inizializza la finestra principale dell'applicazione, immettendovi la
	prima tab
*/
void init_main_window () {
	Graph.notebook = gtk_notebook_new ();
	gtk_widget_set_size_request ( Graph.notebook, MAIN_WINDOW_WIDTH, MAIN_WINDOW_HEIGHT - 50 );
	gtk_notebook_set_scrollable ( GTK_NOTEBOOK ( Graph.notebook ), TRUE );
	g_signal_connect ( G_OBJECT ( Graph.notebook ), "switch-page", G_CALLBACK ( notebook_page_switched ), NULL );

	Graph.tabs = NULL;
	create_tab ();
	gtk_box_pack_start ( GTK_BOX ( Graph.main_box ), Graph.notebook, TRUE, TRUE, 0 );
}

/**
	Inizializza la barra di stato che appare nella parte bassa della
	finestra dell'applicazione
*/
void init_statusbar () {
	GtkWidget *box;

	box = gtk_hbox_new ( 1, TRUE );
	gtk_box_pack_start ( GTK_BOX ( Graph.main_box ), box, FALSE, FALSE, 0 );

	Graph.progress = gtk_progress_bar_new ();
	gtk_widget_set_size_request ( Graph.progress, MAIN_WINDOW_WIDTH / 3, 25 );
	update_status_progress ( 0, 0 );
	gtk_box_pack_start ( GTK_BOX ( box ), Graph.progress, TRUE, TRUE, 0 );

	Graph.status = gtk_statusbar_new ();
	gtk_widget_set_size_request ( Graph.status, ( MAIN_WINDOW_WIDTH / 3 ) * 2, 25 );
	update_status_bar _( "Waiting for queries" );
	gtk_box_pack_start ( GTK_BOX ( box ), Graph.status, TRUE, TRUE, 0 );
}

/**
*/
gboolean execute_first_query ( gpointer useless ) {
	gtk_button_clicked ( GTK_BUTTON ( Graph.query_start ) );
	return FALSE;
}

/**
	Funzione principale dell'applicazione
*/
int main ( int argc, char **argv ) {
	register int i;
	int n;
	gchar *first_query			= NULL;

	gtk_init ( &argc, &argv );

	for ( i = 1; i < argc; i++ ) {
		if ( hyppo_traslate_query ( argv [ i ], &first_query, &n ) != HYPPO_TRASLATION_OK ) {
			g_warning ( "Unrecognized parameter" );
			first_query = NULL;
		}
	}

	textdomain ( "Synapse" );
	bindtextdomain ( "Synapse", NULL );
	bind_textdomain_codeset ( "Synapse", "UTF-8" );

	gtk_set_locale ();
	g_set_application_name ( "Synapse" );

	load_config ();
	hyppo_vfs_init ();
	MountedPath = hyppo_vfs_get_mounted_path ();
	printf ( "Mounted = %s\n", MountedPath );

	Graph.main_win = gtk_window_new ( GTK_WINDOW_TOPLEVEL );
	gtk_container_set_border_width ( GTK_CONTAINER ( Graph.main_win ), 5 );
	gtk_window_set_title ( GTK_WINDOW ( Graph.main_win ), _( "Synapse" ) );
	gtk_widget_set_size_request ( Graph.main_win, MAIN_WINDOW_WIDTH, MAIN_WINDOW_HEIGHT );
	g_signal_connect ( G_OBJECT ( Graph.main_win ), "delete_event", G_CALLBACK ( destroy_all ), NULL );
	g_signal_connect ( G_OBJECT ( Graph.main_win ), "key-press-event", G_CALLBACK ( key_press_event ), NULL );
	g_signal_connect ( G_OBJECT ( Graph.main_win ), "destroy", G_CALLBACK ( destroy_all ), NULL );

	Graph.main_box = gtk_vbox_new ( FALSE, 3 );
	gtk_container_add ( GTK_CONTAINER ( Graph.main_win ), Graph.main_box );

	Graph.tooltips_group = gtk_tooltips_new ();

	Graph.accel_group = gtk_accel_group_new ();
	gtk_window_add_accel_group ( GTK_WINDOW ( Graph.main_win ), Graph.accel_group );

	/**
		The day will be a specific icon theme for Synapse we build a
		personalized icon theme to load...
	*/
	Graph.icons_theme = gtk_icon_theme_get_default ();

	init_toolbar ();
	init_query_composer ();
	init_main_window ();
	init_statusbar ();
	init_reverse_translations ();
	enable_tooltips ();
	load_actions ();
	gtk_widget_show_all ( Graph.main_win );

	load_history ();

	if ( first_query == NULL )
		first_query = history_last ();
	else
		save_in_history ( first_query );

	if ( first_query ) {
		gtk_combo_box_set_active ( GTK_COMBO_BOX ( Graph.query_text ), 0 );
		g_free ( first_query );
		g_idle_add_full ( G_PRIORITY_DEFAULT_IDLE, execute_first_query, NULL, NULL );
	}

	gtk_main ();
	destroy_all ();
	exit ( 0 );
}
