/*  Synapse 0.2
 *  Copyright (C) 2006/2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "synapse.h"

/**
	Name of the file where bookmarks are saved, into the configuration
	path of Synapse
*/
#define DEFAULT_NAME_FOR_BOOKMARKS_FILE		"bookmarks"

/**
	Reference structure of the panel used to manage bookmarks
*/
typedef struct {
	GtkWidget		*list;							/**< List of defined bookmarks */
	GtkTreeStore		*list_model;						/**< Data structure for all bookmarks */
	GtkTreeSelection	*list_selector;						/**< Iterator of the bookmarks list */
	GtkWidget		*name;							/**< Text entry used to define the name for the selected bookmark */
	GtkWidget		*query;							/**< Text entry used to define the query for the selected bookmark */
} BookmarksPanel;

GBookmarkFile		*Bookmarks;							/**< Global reference to the bookmarks file */
gchar			**Uris;								/**< Reference to the strings rappresenting the queries bookmarked. This array is extracted once, each time the entire structure into "Bookmarks" is changed */

/**
	Parse the file in DEFAULT_NAME_FOR_BOOKMARKS_FILE and fill the
	structure "Bookmarks"
*/
void read_bookmarks () {
	gchar *conf_path;

	conf_path = get_system_path ( DEFAULT_NAME_FOR_BOOKMARKS_FILE );

	Bookmarks = g_bookmark_file_new ();
	if ( g_bookmark_file_load_from_file ( Bookmarks, conf_path, NULL ) == FALSE )
		g_bookmark_file_to_file ( Bookmarks, conf_path, NULL );

	if ( g_bookmark_file_get_size ( Bookmarks ) )
		Uris = g_bookmark_file_get_uris ( Bookmarks, NULL );
	else
		Uris = NULL;

	g_free ( conf_path );
}

/**
	Callback binded to all buttons relative to the bookmarked queries,
	execute the selected query into the currently selected tab

	@param	item	Reference to the menu button selected
	@param	pos	Position of the bookmark selected, relative to the
			whole list in "Uris"

	@return		FALSE
*/
static gboolean open_bookmark ( GtkMenuItem *item, gpointer pos ) {
	int i;

	i = GPOINTER_TO_INT ( pos );
	gtk_entry_set_text ( query_label_reference (), Uris [ i ] );
	start_query ();
	return FALSE;
}

/**
	Build the panel to edit a bookmark

	@param	name	Widget dedicated to contain the name of the bookmark
	@param	uri	Widget dedicated to contain the URI of the bookmark

	@return		The panel for the editing
*/
static GtkWidget* edit_dialog ( GtkWidget **name, GtkWidget **uri ) {
	GtkWidget *dialog;
	GtkWidget *main_box;
	GtkWidget *box;
	GtkWidget *label;
	GtkSizeGroup *sg;

	dialog = gtk_dialog_new_with_buttons ( _( "Edit Item Properties" ), GTK_WINDOW ( Graph.main_win ),
	                                       GTK_DIALOG_NO_SEPARATOR | GTK_DIALOG_MODAL, GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
	                                       GTK_STOCK_OK, GTK_RESPONSE_OK, NULL );

	gtk_dialog_set_default_response ( GTK_DIALOG ( dialog ), GTK_RESPONSE_OK );
	main_box = GTK_DIALOG ( dialog )->vbox;

	sg = gtk_size_group_new ( GTK_SIZE_GROUP_HORIZONTAL );

	box = gtk_hbox_new ( FALSE, 1 );
	label = gtk_label_new ( _( "Name: " ) );
	gtk_size_group_add_widget ( sg, label );
	gtk_box_pack_start ( GTK_BOX ( box ), label, FALSE, FALSE, 0 );
	*name = gtk_entry_new ();
	gtk_box_pack_start ( GTK_BOX ( box ), *name, TRUE, TRUE, 0 );
	gtk_box_pack_start ( GTK_BOX ( main_box ), box, FALSE, FALSE, 0 );

	box = gtk_hbox_new ( FALSE, 1 );
	label = gtk_label_new ( _( "Query: " ) );
	gtk_size_group_add_widget ( sg, label );
	gtk_box_pack_start ( GTK_BOX ( box ), label, FALSE, FALSE, 0 );
	*uri = gtk_entry_new ();
	gtk_box_pack_start ( GTK_BOX ( box ), *uri, TRUE, TRUE, 0 );
	gtk_box_pack_start ( GTK_BOX ( main_box ), box, FALSE, FALSE, 0 );

	gtk_widget_show_all ( dialog );
	return dialog;
}

/**
	Callback for the change of selection in the bookmarks list

	@param	select	Selection for the new bookmark to focus
	@param	panel	Main panel for bookmarks
*/
static void edit_bookmark ( GtkTreeSelection *select, BookmarksPanel *panel ) {
	gchar *name;
	gchar *uri;
	GtkTreeIter iter;

	if ( gtk_tree_selection_get_selected ( select, ( GtkTreeModel** ) &( panel->list_model ), &iter ) ) {
		gtk_tree_model_get ( GTK_TREE_MODEL ( panel->list_model ), &iter, 0, &name, 1, &uri, -1 );

		gtk_entry_set_text ( GTK_ENTRY ( panel->name ), name );
		gtk_entry_set_text ( GTK_ENTRY ( panel->query ), uri );

		g_free ( name );
		g_free ( uri );
	}
}

/**
	Callback assigned to the button aiming the addiction of a new bookmark
	in the editing panel

	@param	button	Reference to the button associated to the action
	@param	panel	Reference to the whole editing panel
*/
static void add_bookmark_dialog ( GtkButton *button, BookmarksPanel *panel ) {
	GtkWidget *dialog;
	GtkWidget *wname;
	GtkWidget *wuri;
	GtkTreeIter child;

	dialog = edit_dialog ( &wname, &wuri );

	if ( gtk_dialog_run ( GTK_DIALOG ( dialog ) ) == GTK_RESPONSE_OK ) {
		gtk_tree_store_append ( panel->list_model, &child, NULL );
		gtk_tree_store_set ( panel->list_model, &child, 0, gtk_entry_get_text ( GTK_ENTRY ( wname ) ),
					1, gtk_entry_get_text ( GTK_ENTRY ( wuri ) ), -1 );
	}

	gtk_widget_destroy ( dialog );
}

/**
	Callback assigned to the button aimed to remove the bookmark selected
	in the edit panel

	@param	button	Reference to the button associated to the action
	@param	panel	Reference to the whole editing panel
*/
static void remove_bookmark ( GtkButton *button, BookmarksPanel *panel ) {
	GtkTreeIter child;

	if ( gtk_tree_selection_get_selected ( panel->list_selector, ( GtkTreeModel** ) &( panel->list_model ), &child ) )
		gtk_tree_store_remove ( panel->list_model, &child );
}

/**
	Build the main panel for the configuration of the bookmarks system,
	to add or remove elements
*/
static void manage_bookmarks () {
	int i;
	int size;
	gchar *title;
	gchar *uri;
	gchar *conf_path;
	GtkWidget *panel;
	GtkWidget *box;
	GtkWidget *container;
	GtkWidget *props_frame;
	GtkWidget *label;
	GtkTreeIter child;
	BookmarksPanel *this;

	panel = gtk_dialog_new_with_buttons ( _( "Synapse Bookmarks" ), GTK_WINDOW ( Graph.main_win ),
	                                       GTK_DIALOG_NO_SEPARATOR | GTK_DIALOG_MODAL, GTK_STOCK_CANCEL, GTK_RESPONSE_REJECT,
	                                       GTK_STOCK_OK, GTK_RESPONSE_OK, NULL );

	gtk_dialog_set_default_response ( GTK_DIALOG ( panel ), GTK_RESPONSE_OK );

	this = g_new ( BookmarksPanel, 1 );

	box = gtk_hpaned_new ();
	gtk_box_pack_start ( GTK_BOX ( GTK_DIALOG ( panel )->vbox ), box, FALSE, FALSE, 0 );

	/*** lista URIs ***/

	container = gtk_vbox_new ( FALSE, 3 );
	gtk_container_set_border_width ( GTK_CONTAINER ( container ), 2 );
	gtk_paned_pack1 ( GTK_PANED ( box ), container, FALSE, TRUE );

	this->list_model = gtk_tree_store_new ( 2, G_TYPE_STRING, G_TYPE_STRING );
	size = g_bookmark_file_get_size ( Bookmarks );

	for ( i = 0; i < size; i++ ) {
		title = g_bookmark_file_get_title ( Bookmarks, Uris [ i ], NULL );

		if ( title ) {
			gtk_tree_store_append ( this->list_model, &child, NULL );
			gtk_tree_store_set ( this->list_model, &child, 0, title, 1, Uris [ i ], -1 );
		}
	}

	this->list = gtk_tree_view_new_with_model ( GTK_TREE_MODEL ( this->list_model ) );
	gtk_widget_set_size_request ( this->list, 200, 300 );
	this->list_selector = gtk_tree_view_get_selection ( GTK_TREE_VIEW ( this->list ) );
	gtk_tree_selection_set_mode ( this->list_selector, GTK_SELECTION_BROWSE );
	gtk_tree_view_insert_column_with_attributes ( GTK_TREE_VIEW ( this->list ), -1, _( "Bookmarks" ),
							gtk_cell_renderer_text_new (), "text", 0, NULL );
	g_signal_connect ( G_OBJECT ( this->list_selector ), "changed", G_CALLBACK ( edit_bookmark ), this );
	gtk_box_pack_start ( GTK_BOX ( container ), this->list, TRUE, TRUE, 0 );

	label = gtk_button_new_from_stock ( GTK_STOCK_ADD );
	g_signal_connect ( G_OBJECT ( label ), "clicked", G_CALLBACK ( add_bookmark_dialog ), this );
	gtk_box_pack_start ( GTK_BOX ( container ), label, FALSE, FALSE, 0 );

	/*** pannello edit ***/

	container = gtk_vbox_new ( FALSE, 0 );
	gtk_paned_pack2 ( GTK_PANED ( box ), container, FALSE, TRUE );

	props_frame = kiazma_frame_new ( _( "Item Properties" ) );
	gtk_box_pack_start ( GTK_BOX ( container ), props_frame, FALSE, FALSE, 0 );

	this->name = gtk_entry_new ();
	kiazma_frame_add ( KIAZMA_FRAME ( props_frame ), _( "Name" ), this->name );

	this->query = gtk_entry_new ();
	kiazma_frame_add ( KIAZMA_FRAME ( props_frame ), _( "Query" ), this->query );

	label = kiazma_button_stock_new ( GTK_STOCK_DELETE );
	g_signal_connect ( G_OBJECT ( label ), "clicked", G_CALLBACK ( remove_bookmark ), this );
	kiazma_frame_add ( KIAZMA_FRAME ( props_frame ), _( "Delete this item" ), label );

	gtk_widget_show_all ( panel );

	if ( gtk_dialog_run ( GTK_DIALOG ( panel ) ) == GTK_RESPONSE_OK ) {
		g_bookmark_file_free ( Bookmarks );
		Bookmarks = g_bookmark_file_new ();

		if ( gtk_tree_model_get_iter_first ( GTK_TREE_MODEL ( this->list_model ), &child ) ) {
			do {
				gtk_tree_model_get ( GTK_TREE_MODEL ( this->list_model ), &child, 0, &title, 1, &uri, -1 );
				g_bookmark_file_set_mime_type ( Bookmarks, uri, "text/plain" );
				g_bookmark_file_add_application ( Bookmarks, uri, NULL, NULL );
				g_bookmark_file_set_title ( Bookmarks, uri, title );
				g_free ( title );
				g_free ( uri );
			} while ( gtk_tree_model_iter_next ( GTK_TREE_MODEL ( this->list_model ), &child ) );
		}

		conf_path = get_system_path ( DEFAULT_NAME_FOR_BOOKMARKS_FILE );
		if ( g_bookmark_file_to_file ( Bookmarks, conf_path, NULL ) == FALSE )
			xfce_err ( _( "Unable to save bookmarks file" ) );

		g_free ( conf_path );

		if ( Uris )
			g_strfreev ( Uris );

		Uris = g_bookmark_file_get_uris ( Bookmarks, NULL );
	}

	gtk_widget_destroy ( panel );
}

/**
	Wrapper for manage_bookmarks(), to bind to the relative button
	into the main contextual menu

	@param	item	Reference to the menu button selected
	@param	data	Unused

	@return		FALSE
*/
static gboolean wrapper_menu_manage_bookmarks ( GtkMenuItem *item, gpointer data ) {
	manage_bookmarks ();
	return FALSE;
}

/**
	Add the query currently displayed by the filemanager to the bookmarks
*/
void add_current_to_bookmarks () {
	gchar *query;
	gchar *conf_path;
	const gchar *query_write;
	GtkWidget *dialog;
	GtkWidget *name;
	GtkWidget *uri;

	query = history_last ();
	if ( !query )
		return;

	dialog = edit_dialog ( &name, &uri );
	gtk_entry_set_text ( GTK_ENTRY ( uri ), query );

	if ( gtk_dialog_run ( GTK_DIALOG ( dialog ) ) == GTK_RESPONSE_OK ) {
		conf_path = get_system_path ( DEFAULT_NAME_FOR_BOOKMARKS_FILE );

		if ( Uris )
			g_strfreev ( Uris );

		query_write = gtk_entry_get_text ( GTK_ENTRY ( uri ) );
		g_bookmark_file_set_mime_type ( Bookmarks, query_write, "text/plain" );
		g_bookmark_file_add_application ( Bookmarks, query_write, NULL, NULL );
		g_bookmark_file_set_title ( Bookmarks, query_write, gtk_entry_get_text ( GTK_ENTRY ( name ) ) );

		if ( g_bookmark_file_to_file ( Bookmarks, conf_path, NULL ) == FALSE )
			xfce_err ( _( "Unable to save bookmarks file" ) );

		g_free ( conf_path );
		Uris = g_bookmark_file_get_uris ( Bookmarks, NULL );
		update_status_bar ( "Query bookmarked" );
	}

	gtk_widget_destroy ( dialog );
	g_free ( query );
}

/**
	Wrapper for add_current_to_bookmarks(), to bind to the relative button
	into the main contextual menu

	@param	item	Reference to the menu button selected
	@param	data	Unused

	@return		FALSE
*/
static gboolean wrapper_menu_add_current_to_bookmarks ( GtkMenuItem *item, gpointer data ) {
	add_current_to_bookmarks ();
	return FALSE;
}

/**
	Build the bookmarks submenu for the global contextual menu, with the
	general buttons for configuration and the list of bookmarked queries
	ready to be selected and executed

	@return		The bookmarks menu to add to the global contextual menu
*/
GtkWidget* do_bookmarks_menu () {
	int i;
	int len;
	gchar *title;
	GtkWidget *first;
	GtkWidget *menu;
	GtkWidget *item;
	GtkWidget *list;
	GtkWidget *list_main;

	first = gtk_menu_item_new_with_label ( _( "Bookmarks" ) );
	gtk_widget_show ( first );

	menu = gtk_menu_new ();

	item = gtk_menu_item_new_with_label ( _( "Bookmark Current" ) );
	g_signal_connect ( G_OBJECT ( item ), "activate", G_CALLBACK ( wrapper_menu_add_current_to_bookmarks ), NULL );
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	item = gtk_menu_item_new_with_label ( _( "Show All" ) );
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	gtk_menu_item_set_submenu ( GTK_MENU_ITEM ( first ), menu );

	if ( ( len = g_bookmark_file_get_size ( Bookmarks ) ) == 0 )
		gtk_widget_set_sensitive ( item, FALSE );

	else {
		list_main = item;
		list = gtk_menu_new ();

		for ( i = 0; i < len; i++ ) {
			title = g_bookmark_file_get_title ( Bookmarks, Uris [ i ], NULL );

			if ( title ) {
				item = gtk_menu_item_new_with_label ( title );
				g_free ( title );
			}
			else
				item = gtk_menu_item_new_with_label ( Uris [ i ] );

			g_signal_connect ( G_OBJECT ( item ), "activate", G_CALLBACK ( open_bookmark ), GINT_TO_POINTER ( i ) );
			gtk_widget_show ( item );
			gtk_menu_shell_append ( GTK_MENU_SHELL ( list ), item );
		}

		gtk_menu_item_set_submenu ( GTK_MENU_ITEM ( list_main ), list );
	}

	item = gtk_menu_item_new_with_label ( _( "Manage" ) );
	g_signal_connect ( G_OBJECT ( item ), "activate", G_CALLBACK ( wrapper_menu_manage_bookmarks ), NULL );
	gtk_widget_show ( item );
	gtk_menu_shell_append ( GTK_MENU_SHELL ( menu ), item );

	return first;
}
