/*  LibKiazma 0.2
 *  Copyright (C) 2006/2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
	@file	kiazma_uidlist.c

	Convenience list of users installed in the system; usefull to select
	the owner of a file or other related stuffs
*/

#include "libkiazma.h"
#include "kiazma_uidlist.h"

#include <pwd.h>
#include <grp.h>

/**
	@internal

	Init the class for KiazmaUidList

	@param	klass	Instance of the class to init
*/
static void kiazma_uidlist_class_init ( KiazmaUidList *klass ) {
	return;
}

/**
	@internal

	Init the widgets which compose the KiazmaUidList

	@param	ttt	The KiazmaUidList to init
*/
static void kiazma_uidlist_init ( KiazmaUidList *ttt ) {
	gchar *identifier;
	GtkTreeIter child;
	GtkTreeStore *model;
	GtkCellRenderer *cell;
	struct passwd *item;

	model = gtk_tree_store_new ( 2, G_TYPE_STRING, G_TYPE_STRING );

	cell = gtk_cell_renderer_text_new ();
	gtk_cell_layout_pack_start ( GTK_CELL_LAYOUT ( ttt ), cell, TRUE );
	gtk_cell_layout_set_attributes ( GTK_CELL_LAYOUT ( ttt ), cell, "text", 0, NULL );

	item = getpwent ();
	while ( item ) {
		gtk_tree_store_append ( model, &child, NULL );
		identifier = g_strdup_printf ( "%s (%s)", item->pw_gecos, item->pw_name );
		gtk_tree_store_set ( model, &child, 0, identifier, 1, item->pw_name, -1 );
		g_free ( identifier );
		item = getpwent ();
	}

	endpwent ();
	gtk_combo_box_set_model ( GTK_COMBO_BOX ( ttt ), ( GtkTreeModel* ) model );
	kiazma_uidlist_clear ( ttt );
}

/**
	Register the GType for KiazmaUidList

	@return		The newly registered type for the widget
*/
GType kiazma_uidlist_get_type () {
	static GType ttt_type = 0;

	if ( !ttt_type ) {
		static const GTypeInfo ttt_info = {
			sizeof ( KiazmaUidListClass ),
			NULL,
			NULL,
			( GClassInitFunc ) kiazma_uidlist_class_init,
			NULL,
			NULL,
			sizeof ( KiazmaUidList ),
			0,
			( GInstanceInitFunc ) kiazma_uidlist_init,
		};

		ttt_type = g_type_register_static ( GTK_TYPE_COMBO_BOX, "KiazmaUidList", &ttt_info, 0 );
	}

	return ttt_type;
}

/**
	Retrive the UID of the user selected in the KiazmaUidList

	@param	lista	The KiazmaUidList from which get the information

	@return		UID of the selected user
*/
uid_t kiazma_uidlist_get_value ( KiazmaUidList *lista ) {
	gchar *username;
	struct passwd *user;
	uid_t to_ret;
	GtkTreeIter iter;
	GtkTreeStore *model;

	gtk_combo_box_get_active_iter ( GTK_COMBO_BOX ( lista ), &iter );
	model = ( GtkTreeStore* ) gtk_combo_box_get_model ( GTK_COMBO_BOX ( lista ) );

	gtk_tree_model_get ( GTK_TREE_MODEL ( model ), &iter, 1, &username, -1 );
	user = getpwnam ( ( const char* ) username );

	if ( user != NULL )
		to_ret = user->pw_uid;

	else {
		g_warning ( "Unable to retrive userID for %s : %s", username, strerror ( errno ) );
		to_ret = 0;
	}

	g_free ( username );
	return to_ret;
}

/**
	Permits to set a value in the KiazmaUidList

	@param	list	The KiazmaUidList to set
	@param	user	UID of the user to select into the widget
*/
void kiazma_uidlist_set_value ( KiazmaUidList *list, uid_t user ) {
	int i;
	struct passwd *item;

	i = 0;
	item = getpwent ();

	while ( item ) {
		if ( item->pw_uid == user )
			break;

		item = getpwent ();
		i++;
	}

	endpwent ();
	gtk_combo_box_set_active ( GTK_COMBO_BOX ( list ), i );
}

/**
	Reset the KiazmaUidList, selecting the current user which is running
	the invoking application (the user himself)

	@param	wid	The KiazmaUidList to reset
*/
void kiazma_uidlist_clear ( KiazmaUidList *wid ) {
	uid_t now;

	now = getuid ();
	kiazma_uidlist_set_value ( wid, now );
}

/**
	Builds a new KiazmaUidList

	@return		The new KiazmaUidList
*/
GtkWidget* kiazma_uidlist_new () {
	return g_object_new ( KIAZMA_UIDLIST_TYPE, NULL );
}
