/*  LibKiazma 0.2
 *  Copyright (C) 2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
	@file	kiazma_query_editor.c
*/

#include "libkiazma.h"
#include "kiazma_query_editor.h"

/**
	@internal

	Init the class for KiazmaQueryEditor

	@param	klass	Instance of the class to init
*/
static void kiazma_query_editor_class_init ( KiazmaQueryEditorClass *klass ) {
	return;
}

/**
	@internal
*/
static GtkWidget* do_selection_frame_for_easy ( gchar *title, GtkWidget *main_box ) {
	GtkWidget *box;
	GtkWidget *frame;

	frame = kiazma_frame_new_empty ( title );
	gtk_box_pack_start ( GTK_BOX ( main_box ), frame, FALSE, FALSE, 0 );

	box = kiazma_metadata_set_collector_new ();
	gtk_container_add ( GTK_CONTAINER ( frame ), box );

	return box;
}

/**
	@internal
*/
static gboolean exit_kiazma_query_editor ( GtkButton *but, KiazmaQueryEditor *ttt ) {
	gtk_dialog_response ( GTK_DIALOG ( ttt ), GTK_RESPONSE_OK );
	return FALSE;
}

/**
	@internal

	Init the internals of KiazmaQueryEditor

	@param	ttt	The KiazmaQueryEditor to init
*/
static void kiazma_query_editor_init ( KiazmaQueryEditor *ttt ) {
	GtkWidget *box;
	GtkWidget *header;
	GtkWidget *main_box;
	GtkWidget *response;

	response = gtk_button_new_from_stock ( GTK_STOCK_OK );
	g_signal_connect ( G_OBJECT ( response ), "clicked", G_CALLBACK ( exit_kiazma_query_editor ), ttt );
	gtk_dialog_add_action_widget ( GTK_DIALOG ( ttt ), response, GTK_RESPONSE_OK );

	response = gtk_button_new_from_stock ( GTK_STOCK_CANCEL );
	g_signal_connect ( G_OBJECT ( response ), "clicked", G_CALLBACK ( exit_kiazma_query_editor ), ttt );
	gtk_dialog_add_action_widget ( GTK_DIALOG ( ttt ), response, GTK_RESPONSE_CANCEL );

	gtk_window_set_title ( GTK_WINDOW ( ttt ), _( "Hyppocampus Query Editor" ) );

	main_box = GTK_DIALOG ( ttt )->vbox;

	ttt->modes_notebook = gtk_notebook_new ();
	gtk_container_add ( GTK_CONTAINER ( main_box ), ttt->modes_notebook );

	box = gtk_vbox_new ( FALSE, 3 );

	header = gtk_hbox_new ( FALSE, 6 );
	gtk_box_pack_start ( GTK_BOX ( header ), gtk_label_new ( _( "Attribute to Select " ) ), FALSE, FALSE, 0 );
	ttt->easy_mode_panel.selection = kiazma_metadata_selector_new_full ();
	gtk_box_pack_start ( GTK_BOX ( header ), ttt->easy_mode_panel.selection, FALSE, FALSE, 0 );
	gtk_box_pack_start ( GTK_BOX ( box ), header, FALSE, FALSE, 0 );

	ttt->easy_mode_panel.included = do_selection_frame_for_easy ( _( "Metadata to Include" ), box );
	ttt->easy_mode_panel.excluded = do_selection_frame_for_easy ( _( "Metadata to Exclude" ), box );
	ttt->easy_mode_panel.possible = do_selection_frame_for_easy ( _( "Metadata to Check" ), box );
	gtk_notebook_append_page ( GTK_NOTEBOOK ( ttt->modes_notebook ), box, gtk_label_new ( _( "Easy Mode" ) ) );
}

/**
	Register the GType for KiazmaQueryEditor

	@return		The newly registered type for the widget
*/
GType kiazma_query_editor_get_type () {
	static GType ttt_type = 0;

	if ( !ttt_type ) {
		static const GTypeInfo ttt_info = {
			sizeof ( KiazmaQueryEditorClass ),
			NULL,
			NULL,
			( GClassInitFunc ) kiazma_query_editor_class_init,
			NULL,
			NULL,
			sizeof ( KiazmaQueryEditor ),
			0,
			( GInstanceInitFunc ) kiazma_query_editor_init,
		};

		ttt_type = g_type_register_static ( GTK_TYPE_DIALOG, "KiazmaQueryEditor", &ttt_info, 0 );
	}

	return ttt_type;
}

/**
	Builds a new KiazmaQueryEditor

	@return		The new KiazmaQueryEditor
*/
GtkWidget* kiazma_query_editor_new ( KiazmaQueryEditorMode default_mode ) {
	GtkWidget *new_item;
	new_item = GTK_WIDGET ( g_object_new ( kiazma_query_editor_get_type (), NULL ) );
	return new_item;
}

/**
*/
void kiazma_query_editor_reset ( KiazmaQueryEditor *editor ) {
	kiazma_metadata_selector_set ( KIAZMA_METADATA_SELECTOR ( editor->easy_mode_panel.selection ), 0 );
	kiazma_metadata_set_collector_reset ( KIAZMA_METADATA_SET_COLLECTOR ( editor->easy_mode_panel.included ) );
	kiazma_metadata_set_collector_reset ( KIAZMA_METADATA_SET_COLLECTOR ( editor->easy_mode_panel.excluded ) );
	kiazma_metadata_set_collector_reset ( KIAZMA_METADATA_SET_COLLECTOR ( editor->easy_mode_panel.possible ) );
}

/**
*/
gchar* kiazma_query_editor_get_query ( KiazmaQueryEditor *editor ) {
	UINT64 meta;
	gchar *tmp;
	gchar *value;
	gchar *query				= NULL;
	HYPPO_COMPARISON_OPERATOR op;

	if ( gtk_notebook_get_current_page ( GTK_NOTEBOOK ( editor->modes_notebook ) ) == 0 ) {
		gchar *included			= NULL;
		gchar *excluded			= NULL;
		gchar *possible			= NULL;

		if ( kiazma_metadata_set_collector_get_first (
				KIAZMA_METADATA_SET_COLLECTOR ( editor->easy_mode_panel.included ),
				&meta, &op, &value ) ) {

			tmp = hyppo_compose_statement_string ( meta, op, value );
			if ( tmp ) {
				included = g_strdup_printf ( "( %s", tmp );
				g_free ( tmp );

				while ( kiazma_metadata_set_collector_get_next (
						KIAZMA_METADATA_SET_COLLECTOR ( editor->easy_mode_panel.included ),
						&meta, &op, &value ) ) {

					tmp = hyppo_compose_statement_string ( meta, op, value );
					if ( tmp ) {
						query = g_strdup_printf ( "%s AND %s", included, tmp );
						g_free ( included );
						included = query;
						g_free ( tmp );
					}
				}

				query = g_strdup_printf ( "%s )", included );
				g_free ( included );
				included = query;
			}
		}

		if ( kiazma_metadata_set_collector_get_first (
				KIAZMA_METADATA_SET_COLLECTOR ( editor->easy_mode_panel.excluded ),
				&meta, &op, &value ) ) {

			tmp = hyppo_compose_statement_string ( meta, op, value );
			excluded = g_strdup_printf ( "NOT ( %s", tmp );
			g_free ( tmp );

			while ( kiazma_metadata_set_collector_get_next (
					KIAZMA_METADATA_SET_COLLECTOR ( editor->easy_mode_panel.excluded ),
					&meta, &op, &value ) ) {

				tmp = hyppo_compose_statement_string ( meta, op, value );
				query = g_strdup_printf ( "%s AND %s", excluded, tmp );
				g_free ( excluded );
				excluded = query;
				g_free ( tmp );
			}

			query = g_strdup_printf ( "%s )", excluded );
			g_free ( excluded );
			excluded = query;
		}

		if ( kiazma_metadata_set_collector_get_first (
				KIAZMA_METADATA_SET_COLLECTOR ( editor->easy_mode_panel.possible ),
				&meta, &op, &value ) ) {

			tmp = hyppo_compose_statement_string ( meta, op, value );
			possible = g_strdup_printf ( "( %s", tmp );
			g_free ( tmp );

			while ( kiazma_metadata_set_collector_get_next (
					KIAZMA_METADATA_SET_COLLECTOR ( editor->easy_mode_panel.possible ),
					&meta, &op, &value ) ) {

				tmp = hyppo_compose_statement_string ( meta, op, value );
				query = g_strdup_printf ( "%s OR %s", excluded, tmp );
				g_free ( possible );
				possible = query;
				g_free ( tmp );
			}

			query = g_strdup_printf ( "%s )", possible );
			g_free ( possible );
			possible = query;
		}

		query = hyppo_strconcat ( " AND ", 3, included, excluded, possible );
	}

	return query;
}
