/*  LibKiazma 0.2
 *  Copyright (C) 2006/2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
	@file		kiazma_metadata_wrapper.c
	@brief		Wrapper to handle an unique widget for all metadata values

	The KiazmaMetadataWrapper is not a widget, but a container which containes the more
	appropriate widget to rappresent the value assigned to a particular metadata
*/

#include "libkiazma.h"
#include "kiazma_metadata_wrapper.h"

enum {
	KIAZMA_MW_CHANGED,
	KIAZMA_MW_END_SIGNALS
};

static guint kiazma_metadata_wrapper_signals [ KIAZMA_MW_END_SIGNALS ];

static void kiazma_metadata_wrapper_class_init ( KiazmaMetadataWrapperClass *klass ) {
	GObjectClass *gobject_class;

	gobject_class = ( GObjectClass* ) klass;

	kiazma_metadata_wrapper_signals [ KIAZMA_MW_CHANGED ] = g_signal_new ( g_intern_static_string ( "changed" ),
						G_TYPE_FROM_CLASS ( gobject_class ), G_SIGNAL_RUN_FIRST, 0, NULL, NULL,
						g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0 );
}

static void kiazma_metadata_wrapper_init ( KiazmaMetadataWrapper *ttt ) {
	gtk_frame_set_shadow_type ( GTK_FRAME ( ttt ), GTK_SHADOW_NONE );
	gtk_container_set_border_width ( GTK_CONTAINER ( ttt ), 0 );
}

static void kiazma_metadata_wrapper_value_changed ( GtkEditable *editable, KiazmaMetadataWrapper *main ) {
	g_signal_emit ( main, kiazma_metadata_wrapper_signals [ KIAZMA_MW_CHANGED ], 0 );
}

GType kiazma_metadata_wrapper_get_type () {
	static GType ttt_type = 0;

	if ( !ttt_type ) {
		static const GTypeInfo ttt_info = {
			sizeof ( KiazmaMetadataWrapperClass ),
			NULL,
			NULL,
			( GClassInitFunc ) kiazma_metadata_wrapper_class_init,
			NULL,
			NULL,
			sizeof ( KiazmaMetadataWrapper ),
			0,
			( GInstanceInitFunc ) kiazma_metadata_wrapper_init,
		};

		ttt_type = g_type_register_static ( GTK_TYPE_FRAME, "KiazmaMetadataWrapper", &ttt_info, 0 );
	}

	return ttt_type;
}

static GtkWidget* choose_appropriate_meta ( UINT64 meta ) {
	int metatype;
	GtkWidget *new			= NULL;

	metatype = AllMetadata [ meta ].type;

	if ( metatype & META_TYPE_INTERNAL )
		metatype = META_TYPE_LLONG | META_TYPE_UNSIGNED;

	if ( metatype & META_TYPE_SHORT ) {
		if ( metatype & META_TYPE_UNSIGNED )
			new = gtk_spin_button_new_with_range ( 0, USHRT_MAX, 1 );
		else
			new = gtk_spin_button_new_with_range ( SHRT_MIN, SHRT_MAX, 1 );
	}

	else if ( metatype & META_TYPE_INT ) {
		if ( metatype & META_TYPE_UNSIGNED )
			new = gtk_spin_button_new_with_range ( 0, UINT_MAX, 1 );
		else
			new = gtk_spin_button_new_with_range ( INT_MIN, INT_MAX, 1 );
	}

	else if ( metatype & META_TYPE_DOUBLE ) {
		if ( metatype & META_TYPE_UNSIGNED )
			new = gtk_spin_button_new_with_range ( 0, __DBL_MAX__, 1 );
		else
			new = gtk_spin_button_new_with_range ( __DBL_MIN__, __DBL_MAX__, 1 );
	}

	else if ( metatype & META_TYPE_FLOAT ) {
		if ( metatype & META_TYPE_UNSIGNED )
			new = gtk_spin_button_new_with_range ( 0, __FLT_MAX__, 1 );
		else
			new = gtk_spin_button_new_with_range ( __FLT_MIN__, __FLT_MAX__, 1 );
	}

	else if ( metatype & META_TYPE_LONG ) {
		if ( metatype & META_TYPE_UNSIGNED )
			new = gtk_spin_button_new_with_range ( 0, ULONG_MAX, 1 );
		else
			new = gtk_spin_button_new_with_range ( LONG_MIN, LONG_MAX, 1 );
	}

	else if ( metatype & META_TYPE_LLONG ) {
		if ( metatype & META_TYPE_UNSIGNED )
			new = gtk_spin_button_new_with_range ( 0, ULLONG_MAX, 1 );
		else
			new = gtk_spin_button_new_with_range ( LLONG_MIN, LLONG_MAX, 1 );
	}

	else if ( metatype & META_TYPE_ENUM )
		new = kiazma_enumeration_combo_new ( AllMetadata [ meta ].opt );

	else if ( metatype & META_TYPE_TIME ) {
		new = kiazma_time_new ();
		kiazma_time_clear ( KIAZMA_TIME ( new ) );
	}

	else if ( metatype & META_TYPE_UID )
		new = kiazma_uidlist_new ();

	else if ( metatype & META_TYPE_GID )
		new = kiazma_gidlist_new ();

	else if ( metatype & META_TYPE_MODE )
		new = kiazma_permissions_new ();

	else if ( metatype & META_TYPE_STRING || metatype & META_TYPE_WSTRING )
		new = gtk_entry_new ();

	else if ( metatype & META_TYPE_BOOLEAN )
		new = gtk_check_button_new ();

	if ( new == NULL ) {
		char stringa [ 100 ];
		( void ) snprintf ( stringa, 100, "UNDEFINED FOR TYPE %d", metatype );
		new = gtk_label_new ( stringa );
	}

	return new;
}

GtkWidget* kiazma_metadata_wrapper_new ( UINT64 meta ) {
	GtkWidget *new_item;
	GtkWidget *inside;

	new_item = GTK_WIDGET ( g_object_new ( kiazma_metadata_wrapper_get_type (), NULL ) );
	KIAZMA_METADATA_WRAPPER ( new_item )->metadata = meta;
	inside = choose_appropriate_meta ( meta );
	g_signal_connect ( G_OBJECT ( inside ), "changed", G_CALLBACK ( kiazma_metadata_wrapper_value_changed ), new_item );
	gtk_container_add ( GTK_CONTAINER ( new_item ), inside );
	return new_item;
}

UINT64 kiazma_metadata_wrapper_get_meta ( KiazmaMetadataWrapper *wrap ) {
	return wrap->metadata;
}

gchar* kiazma_metadata_wrapper_get_value ( KiazmaMetadataWrapper *wrap ) {
	int metatype;
	gchar *value			= NULL;
	GtkWidget *wid;

	wid = gtk_bin_get_child ( GTK_BIN ( wrap ) );
	metatype = AllMetadata [ wrap->metadata ].type;

	if ( metatype & META_TYPE_SHORT ) {
		if ( metatype & META_TYPE_UNSIGNED )
			value = g_strdup_printf ( "%hu", ( unsigned short ) gtk_spin_button_get_value ( GTK_SPIN_BUTTON ( wid ) ) );
		else
			value = g_strdup_printf ( "%hd", ( short ) gtk_spin_button_get_value ( GTK_SPIN_BUTTON ( wid ) ) );
	}

	else if ( metatype & META_TYPE_INT ) {
		if ( metatype & META_TYPE_UNSIGNED )
			value = g_strdup_printf ( "%u", ( unsigned int ) gtk_spin_button_get_value ( GTK_SPIN_BUTTON ( wid ) ) );
		else
			value = g_strdup_printf ( "%d", ( unsigned int ) gtk_spin_button_get_value ( GTK_SPIN_BUTTON ( wid ) ) );
	}

	else if ( metatype & META_TYPE_LONG ) {
		if ( metatype & META_TYPE_UNSIGNED )
			value = g_strdup_printf ( "%lu", ( unsigned long ) gtk_spin_button_get_value ( GTK_SPIN_BUTTON ( wid ) ) );
		else
			value = g_strdup_printf ( "%ld", ( long ) gtk_spin_button_get_value ( GTK_SPIN_BUTTON ( wid ) ) );
	}

	else if ( metatype & META_TYPE_LLONG ) {
		if ( metatype & META_TYPE_UNSIGNED )
			value = g_strdup_printf ( "%llu", ( unsigned long long ) gtk_spin_button_get_value ( GTK_SPIN_BUTTON ( wid ) ) );
		else
			value = g_strdup_printf ( "%lld", ( long long ) gtk_spin_button_get_value ( GTK_SPIN_BUTTON ( wid ) ) );
	}

	else if ( metatype & META_TYPE_DOUBLE )
		value = g_strdup_printf ( "%f", ( double ) gtk_spin_button_get_value ( GTK_SPIN_BUTTON ( wid ) ) );

	/**
		@todo	Non e' ancora contemplato il tipo META_TYPE_FLOAT
	*/

	else if ( metatype & META_TYPE_ENUM )
		value = g_strdup_printf ( gtk_combo_box_get_active_text ( GTK_COMBO_BOX ( wid ) ) );

	else if ( metatype & META_TYPE_TIME )
		value = g_strdup_printf ( "%ld", kiazma_time_get_value ( KIAZMA_TIME ( wid ) ) );

	else if ( metatype & META_TYPE_UID )
		value = g_strdup_printf ( "%u", kiazma_uidlist_get_value ( KIAZMA_UIDLIST ( wid ) ) );

	else if ( metatype & META_TYPE_GID )
		value = g_strdup_printf ( "%u", kiazma_gidlist_get_value ( KIAZMA_GIDLIST ( wid ) ) );

	else if ( metatype & META_TYPE_MODE )
		value = g_strdup_printf ( "%hu", kiazma_permissions_get_value ( KIAZMA_PERMISSIONS ( wid ) ) );

	else if ( metatype & META_TYPE_STRING || metatype & META_TYPE_WSTRING )
		value = g_strdup ( gtk_entry_get_text ( GTK_ENTRY ( wid ) ) );

	else if ( metatype & META_TYPE_BOOLEAN )
		value = g_strdup_printf ( "%s", gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON ( wid ) ) ? "TRUE" : "FALSE" );

	return value;
}

void kiazma_metadata_wrapper_set_value ( KiazmaMetadataWrapper *wrap, gchar *value ) {
	UINT64 metatype;
	GtkWidget *wid;

	wid = gtk_bin_get_child ( GTK_BIN ( wrap ) );
	metatype = AllMetadata [ wrap->metadata ].type;

	if ( metatype & META_TYPE_SHORT || metatype & META_TYPE_INT )
		gtk_spin_button_set_value ( GTK_SPIN_BUTTON ( wid ), ( gdouble ) atoi ( value ) );

	else if ( metatype & META_TYPE_DOUBLE )
		gtk_spin_button_set_value ( GTK_SPIN_BUTTON ( wid ), strtod ( value, NULL ) );

	else if ( metatype & META_TYPE_FLOAT )
		gtk_spin_button_set_value ( GTK_SPIN_BUTTON ( wid ), ( gdouble ) strtof ( value, NULL ) );

	else if ( metatype & META_TYPE_LONG ) {
		if ( metatype & META_TYPE_UNSIGNED )
			gtk_spin_button_set_value ( GTK_SPIN_BUTTON ( wid ), ( gdouble ) strtoul ( value, NULL, 10 ) );
		else
			gtk_spin_button_set_value ( GTK_SPIN_BUTTON ( wid ), ( gdouble ) atol ( value ) );
	}

	else if ( metatype & META_TYPE_LLONG ) {
		if ( metatype & META_TYPE_UNSIGNED )
			gtk_spin_button_set_value ( GTK_SPIN_BUTTON ( wid ), ( gdouble ) strtoull ( value, NULL, 10 ) );
		else
			gtk_spin_button_set_value ( GTK_SPIN_BUTTON ( wid ), ( gdouble ) atoll ( value ) );
	}

	else if ( metatype & META_TYPE_ENUM )
		gtk_combo_box_set_active ( GTK_COMBO_BOX ( wid ), atoi ( value ) );

	else if ( metatype & META_TYPE_TIME )
		kiazma_time_set_value ( KIAZMA_TIME ( wid ), atol ( value ) );

	else if ( metatype & META_TYPE_UID )
		kiazma_uidlist_set_value ( KIAZMA_UIDLIST ( wid ), atoi ( value ) );

	else if ( metatype & META_TYPE_GID )
		kiazma_gidlist_new ( KIAZMA_GIDLIST ( wid ), atoi ( value ) );

	else if ( metatype & META_TYPE_MODE )
		kiazma_permissions_set_value ( KIAZMA_PERMISSIONS ( wid ), atoi ( value ) );

	else if ( metatype & META_TYPE_STRING || metatype & META_TYPE_WSTRING )
		gtk_entry_set_text ( GTK_ENTRY ( wid ), value );

	else if ( metatype & META_TYPE_BOOLEAN )
		gtk_toggle_button_set_active ( GTK_TOGGLE_BUTTON ( wid ), strcmp ( value, "TRUE" ) == 0 ? TRUE : FALSE );
}
