/*  LibKiazma 0.2
 *  Copyright (C) 2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
	@file	kiazma_icon.c
*/

#include "libkiazma.h"
#include "kiazma_icon.h"

typedef enum {
	KIAZMA_ICON_MAIN_IMAGE,
	KIAZMA_ICON_TEXTUAL,
	KIAZMA_ICON_NE,
	KIAZMA_ICON_SE,
	KIAZMA_ICON_NO,
	KIAZMA_ICON_SO,
} KIAZMA_ICON_INTERNAL_AREA;

/**
*/
GdkPixbuf* kiazma_icon_get_pixbuf_by_meta ( HyppoVFSMeta *attr ) {
	char *string;
	GdkPixbuf *ret				= NULL;
	GError *error				= NULL;
	extern GtkIconTheme *IconTheme;

	switch ( attr->meta ) {
		case META_MIME:
			string = strdupa ( attr->value );
			hyppo_substitute_chars ( string, '/', '-' );
			ret = gtk_icon_theme_load_icon ( IconTheme, string, 32, GTK_ICON_LOOKUP_USE_BUILTIN, &error );
			break;

		/**
			@todo	Possible metadata types which can be directly handled here:
				META_UID
				META_GID
				META_MOD
				META_TYPE
		*/

		case META_ICON:
			ret = gdk_pixbuf_new_from_file ( attr->value, &error );
			break;

		/**
				META_IS_PRIVATE
				META_ARCHITECTURE
				META_AUDIO_ENCODING
				META_ORIGINAL_SENDER
				META_TEXT_ENCODING
				META_LANGUAGE
				META_VIDEO_ENCONDIG
		*/

		case META_PHOTO:
			string = hyppo_vfs_file_path ( strtoull ( attr->value, NULL, 10 ) );
			ret = gdk_pixbuf_new_from_file ( string, &error );
			g_free ( string );
			break;

		/**
				META_REQUIRE_SIGN_MESSAGES
				META_REQUIRE_ENCRYPT_MESSAGES
		*/

		default:
			break;
	}

	if ( ret == NULL )
		ret = gdk_pixbuf_new_from_file ( KIAZMA_INTERNALS_PATH "/icons/unknown.png", &error );

	return ret;
}

/**
@verbatim
	         w
	 +--------------+
	 |              |
	 |1/3w          |
	 +----+    +----+
	 |    |    |    |

	+----------------+
	|+----+    +----+| ----+-----------+
	|| NO |----| NE ||     | 1/3 i     |
	|+----+    +----+| ----+           |
	|  |          |  |                 |
	|  |          |  |                 | 2/3 h = i
	|+----+    +----+| ----+           |
	|| SO |----| SE ||     | 1/3 i     |
	|+----+    +----+| ----+-----------+
	|                |
	|  TEXTUAL LABEL |
	+----------------+
@endverbatim

	@param	pos
	@param	drawable
	@param	width
	@param	height
	@param	attr

	@return
*/
static gboolean kiazma_icon_add_attribute ( KiazmaIcon *main_wid, KIAZMA_ICON_INTERNAL_AREA pos, GdkDrawable *drawable,
						int width, int height, HyppoVFSMeta *attr ) {

	int x			= 0;
	int y			= 0;
	int w			= 0;
	int h			= 0;
	GdkPixbuf *icon		= NULL;
	GdkPixbuf *final_icon	= NULL;
	GdkGC *gc		= NULL;
	PangoLayout *pangolay	= NULL;

	switch ( pos ) {
		case KIAZMA_ICON_MAIN_IMAGE:
			icon = kiazma_icon_get_pixbuf_by_meta ( attr );
			x = width / 6;
			y = height - ( height / 9 );
			w = width;
			h = ( height / 3 ) * 2;
			break;

		case KIAZMA_ICON_TEXTUAL:
			pangolay = gtk_widget_create_pango_layout ( GTK_WIDGET ( main_wid ), attr->value );
			pango_layout_set_width ( pangolay, width );
			x = 0;
			y = 0;
			break;

		case KIAZMA_ICON_NE:
			icon = kiazma_icon_get_pixbuf_by_meta ( attr );
			w = width / 3;
			h = ( ( height / 3 ) * 2 ) / 3;
			x = ( width / 3 ) * 2;
			y = height - h;
			break;

		case KIAZMA_ICON_SE:
			icon = kiazma_icon_get_pixbuf_by_meta ( attr );
			w = width / 3;
			h = ( ( height / 3 ) * 2 ) / 3;
			x = ( width / 3 ) * 2;
			y = height - ( ( height / 3 ) * 4 );
			break;

		case KIAZMA_ICON_NO:
			icon = kiazma_icon_get_pixbuf_by_meta ( attr );
			w = width / 3;
			h = ( ( height / 3 ) * 2 ) / 3;
			x = 0;
			y = height - h;
			break;

		case KIAZMA_ICON_SO:
			icon = kiazma_icon_get_pixbuf_by_meta ( attr );
			w = width / 3;
			h = ( ( height / 3 ) * 2 ) / 3;
			x = 0;
			y = height - ( ( height / 3 ) * 4 );
			break;

		default:
			return FALSE;
			break;
	}

	if ( icon ) {
		final_icon = gdk_pixbuf_scale_simple ( icon, w, h, GDK_INTERP_BILINEAR );
		gdk_draw_pixbuf ( drawable, NULL, final_icon, 0, 0, x, y, w, h, GDK_RGB_DITHER_NONE, 0, 0 );
		g_object_unref ( icon );
		g_object_unref ( final_icon );
	}

	if ( pangolay ) {
		gc = gdk_gc_new ( drawable );
		gdk_draw_layout ( drawable, gc, x, y, pangolay );
		g_object_unref ( pangolay );
		g_object_unref ( gc );
	}

	return TRUE;
}

/**
	@internal
*/
static GdkWindow* init_drawable ( GtkWidget *widget ) {
	gint attributes_mask;
	GdkWindowAttr attributes;

	attributes.window_type = GDK_WINDOW_CHILD;
	attributes.x = widget->allocation.x;
	attributes.y = widget->allocation.y;
	attributes.width = widget->allocation.width;
	attributes.height = widget->allocation.height;
	attributes.wclass = GDK_INPUT_OUTPUT;
	attributes.visual = gtk_widget_get_visual ( widget );
	attributes.colormap = gtk_widget_get_colormap ( widget );
	attributes.event_mask = GDK_VISIBILITY_NOTIFY_MASK;
	attributes.event_mask = ( GDK_EXPOSURE_MASK | GDK_SCROLL_MASK | GDK_POINTER_MOTION_MASK |
				GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_KEY_PRESS_MASK |
				GDK_KEY_RELEASE_MASK ) | gtk_widget_get_events ( widget );

	attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
	return gdk_window_new ( gtk_widget_get_parent_window ( widget ), &attributes, attributes_mask );
}

/**
	@internal
*/
static void fake_window ( GtkWidget *widget ) {
	GdkWindowAttr attributes;
	gint attributes_mask;

	attributes.window_type = GDK_WINDOW_CHILD;
	attributes.x = 0;
	attributes.y = 0;
	attributes.width = widget->allocation.width;
	attributes.height = widget->allocation.height;
	attributes.wclass = GDK_INPUT_OUTPUT;
	attributes.visual = gtk_widget_get_visual ( widget );
	attributes.colormap = gtk_widget_get_colormap ( widget );
	attributes.event_mask = GDK_VISIBILITY_NOTIFY_MASK;
	attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

	widget->window = gdk_window_new ( NULL, &attributes, attributes_mask );
	gdk_window_set_back_pixmap ( widget->window, NULL, FALSE );
	gdk_window_set_user_data ( widget->window, widget );
	gdk_window_show_unraised ( widget->window );
}

/**
	@internal
*/
static void build_icon ( KiazmaIcon *icon ) {
	register int i;
	GList *iter;
	GdkDrawable *drawable;
	GtkWidget *widget;

	widget = GTK_WIDGET ( icon );

	if ( widget->allocation.width < KIAZMA_ICON_MIN_SIZE )
		widget->allocation.width = KIAZMA_ICON_MIN_SIZE;
	if ( widget->allocation.height < KIAZMA_ICON_MIN_SIZE )
		widget->allocation.height = KIAZMA_ICON_MIN_SIZE;

	if ( widget->parent == NULL )
		fake_window ( widget );
	else if ( widget->window == 0 )
		gtk_widget_realize ( GTK_WIDGET ( icon ) );

	drawable = GDK_DRAWABLE ( widget->window );

	for ( i = 0, iter = g_list_first ( icon->props ); iter; i++, iter = g_list_next ( iter ) )
		if ( kiazma_icon_add_attribute ( icon, i, drawable, widget->allocation.width,
						widget->allocation.height, ( HyppoVFSMeta* ) iter->data ) == FALSE )
			break;
}

/**
	@internal
*/
static void kiazma_icon_realize ( GtkWidget *widget ) {
	GTK_WIDGET_SET_FLAGS ( widget, GTK_REALIZED );

	widget->window = init_drawable ( widget );
	gdk_window_set_back_pixmap ( widget->window, NULL, FALSE );
	gdk_window_set_user_data ( widget->window, widget );
	widget->style = gtk_style_attach ( widget->style, widget->window );
	gdk_window_set_background ( widget->window, &widget->style->base [ widget->state ] );
	gdk_window_show ( widget->window );
}

/**
	@internal
*/
static gboolean kiazma_icon_expose ( GtkWidget *widget, GdkEventExpose *expose ) {
	GdkPixbuf *pixbuf;

	if ( expose->window != widget->window )
		return FALSE;

	pixbuf = kiazma_icon_prerender ( KIAZMA_ICON ( widget ) );
	gdk_draw_pixbuf ( widget->window, NULL, pixbuf, 0, 0, 0, 0,
			widget->allocation.width, widget->allocation.height, GDK_RGB_DITHER_NONE, 0, 0 );

	return TRUE;
}

/**
	@internal
*/
static void kiazma_icon_destroy ( GtkObject *object ) {
	KiazmaIcon *icon;
	KiazmaIconClass *icon_class;

	icon = KIAZMA_ICON ( object );

	if ( icon->props ) {
		hyppo_vfs_free_complete_xattrs ( icon->props );
		icon->props = NULL;
	}

	icon_class = KIAZMA_ICON_CLASS ( g_type_class_peek ( KIAZMA_ICON_TYPE ) );
	( *GTK_OBJECT_CLASS ( &( icon_class->parent_class ) )->destroy ) ( object );
}

/**
	@internal

	Init the class for KiazmaIcon

	@param	klass	Instance of the class to init
*/
static void kiazma_icon_class_init ( KiazmaIconClass *klass ) {
	GtkWidgetClass *widget_class;
	GtkObjectClass *object_class;

	widget_class = ( GtkWidgetClass* ) klass;
	widget_class->realize = kiazma_icon_realize;
	widget_class->expose_event = kiazma_icon_expose;

	object_class = ( GtkObjectClass* ) klass;
	object_class->destroy = kiazma_icon_destroy;
}

/**
	@internal

	Init the internals of KiazmaIcon

	@param	ttt	The KiazmaIcon to init
*/
static void kiazma_icon_init ( KiazmaIcon *ttt ) {
	GTK_WIDGET ( ttt )->window = 0;
	ttt->props = NULL;
	return;
}

/**
	Register the GType for KiazmaIcon

	@return		The newly registered type for the widget
*/
GType kiazma_icon_get_type () {
	static GType ttt_type = 0;

	if ( !ttt_type ) {
		static const GTypeInfo ttt_info = {
			sizeof ( KiazmaIconClass ),
			NULL,
			NULL,
			( GClassInitFunc ) kiazma_icon_class_init,
			NULL,
			NULL,
			sizeof ( KiazmaIcon ),
			0,
			( GInstanceInitFunc ) kiazma_icon_init,
		};

		ttt_type = g_type_register_static ( KIAZMA_WIDGET_TYPE, "KiazmaIcon", &ttt_info, 0 );
	}

	return ttt_type;
}

/**
	@internal
*/
static GtkWidget* alloc_icon ( UINT64 file ) {
	GtkWidget *new_item;

	new_item = GTK_WIDGET ( g_object_new ( kiazma_icon_get_type (), NULL ) );
	KIAZMA_WIDGET ( new_item )->file = file;
	return new_item;
}

/**
	Builds a new KiazmaIcon

	@param	file		Unique identifier of the rappresented file
	@param	...		List of metadata assigned to the file,
				expressed as couple of metadata identifier
				(UINT64) and value (gchar*). This list have
				to be terminated with 0

	@return			The new KiazmaIcon
*/
GtkWidget* kiazma_icon_new ( UINT64 file, ... ) {
	va_list li;
	UINT64 meta;
	char *value;
	HyppoVFSMeta *prop;
	GtkWidget *new_item;
	KiazmaIcon *item;

	new_item = alloc_icon ( file );
	item = KIAZMA_ICON ( new_item );

	va_start ( li, file );
	meta = va_arg ( li, UINT64 );
	while ( meta ) {
		value = va_arg ( li, char* );
		if ( value == NULL )
			break;

		prop = g_new0 ( HyppoVFSMeta, 1 );
		prop->meta = meta;
		prop->value = g_strdup ( value );
		item->props = g_list_prepend ( item->props, prop );
		meta = va_arg ( li, UINT64 );
	}
	va_end ( li );
	item->props = g_list_reverse ( item->props );

	return new_item;
}

/**
	As kiazma_icon_new(), get metadata assigned to the file into a GList*
	filled with HyppoVFSMeta structs

	@param	file		Unique identifier of the rappresented file
	@param	list		List of HyppoVFSMeta structs, each of them
				rappresenting a metadata assigned to the
				specified file which have to be materialized
				into the icon

	@return			The new KiazmaIcon
*/
GtkWidget* kiazma_icon_new_list ( UINT64 file, GList *list ) {
	GtkWidget *new_item;
	KiazmaIcon *item;

	new_item = alloc_icon ( file );
	item = KIAZMA_ICON ( new_item );
	item->props = list;
	return new_item;
}

/**
*/
GdkPixbuf* kiazma_icon_prerender ( KiazmaIcon *icon ) {
	gboolean destroy_window		= FALSE;
	GdkPixbuf *pixbuf;
	GtkWidget *widget;

	widget = GTK_WIDGET ( icon );

	if ( widget->parent == NULL )
		destroy_window = TRUE;

	build_icon ( icon );
	pixbuf = gdk_pixbuf_get_from_drawable ( NULL, widget->window, NULL, 0, 0, 0, 0,
						widget->allocation.width, widget->allocation.height );

	/**
		@todo	Ugly hack here applied: if the icon is not handled as a common widget,
			and only the inside pixbuf is required, a new GdkWindow is created,
			used as drawable surface, showed and destroyed. It is necessary to
			find a way to obtain a valid drawable without this workaround...
	*/
	if ( destroy_window ) {
		gdk_window_destroy ( widget->window );
		widget->window = NULL;
	}

	return pixbuf;
}
