/*  LibKiazma 0.2
 *  Copyright (C) 2006/2007 Roberto -MadBob- Guido <m4db0b@users.sourceforge.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
	@file	kiazma_gidlist.c

	Convenience list of groups installed in the system; usefull to select
	the group owner of a file or other related stuffs
*/

#include "libkiazma.h"
#include "kiazma_gidlist.h"

#include <pwd.h>
#include <grp.h>

/**
	@internal

	Init the class for KiazmaGidList

	@param	klass	Instance of the class to init
*/
static void kiazma_gidlist_class_init ( KiazmaGidList *klass ) {
	return;
}

/**
	@internal

	Init the widgets which compose the KiazmaGidList

	@param	ttt	The KiazmaGidList to init
*/
static void kiazma_gidlist_init ( KiazmaGidList *ttt ) {
	GtkTreeIter child;
	GtkTreeStore *model;
	GtkCellRenderer *cell;
	struct group *item;

	model = gtk_tree_store_new ( 1, G_TYPE_STRING );

	cell = gtk_cell_renderer_text_new ();
	gtk_cell_layout_pack_start ( GTK_CELL_LAYOUT ( ttt ), cell, TRUE );
	gtk_cell_layout_set_attributes ( GTK_CELL_LAYOUT ( ttt ), cell, "text", 0, NULL );

	item = getgrent ();
	while ( item ) {
		gtk_tree_store_append ( model, &child, NULL );
		gtk_tree_store_set ( model, &child, 0, item->gr_name, -1 );
		item = getgrent ();
	}

	endgrent ();
	gtk_combo_box_set_model ( GTK_COMBO_BOX ( ttt ), ( GtkTreeModel* ) model );
	kiazma_gidlist_clear ( ttt );
}

/**
	Register the GType for KiazmaGidList

	@return		The newly registered type for the widget
*/
GType kiazma_gidlist_get_type () {
	static GType ttt_type = 0;

	if ( !ttt_type ) {
		static const GTypeInfo ttt_info = {
			sizeof ( KiazmaGidListClass ),
			NULL,
			NULL,
			( GClassInitFunc ) kiazma_gidlist_class_init,
			NULL,
			NULL,
			sizeof ( KiazmaGidList ),
			0,
			( GInstanceInitFunc ) kiazma_gidlist_init,
		};

		ttt_type = g_type_register_static ( GTK_TYPE_COMBO_BOX, "KiazmaGidList", &ttt_info, 0 );
	}

	return ttt_type;
}

/**
	Retrive the GID of the group selected in the KiazmaGidList

	@param	lista	The KiazmaGidList from which get the information

	@return		GID of the selected group
*/
gid_t kiazma_gidlist_get_value ( KiazmaGidList *lista ) {
	gchar *username;
	struct group *user;
	GtkTreeIter iter;
	GtkTreeStore *model;

	gtk_combo_box_get_active_iter ( GTK_COMBO_BOX ( lista ), &iter );
	model = ( GtkTreeStore* ) gtk_combo_box_get_model ( GTK_COMBO_BOX ( lista ) );

	gtk_tree_model_get ( GTK_TREE_MODEL ( model ), &iter, 0, &username, -1 );
	user = getgrnam ( ( const char* ) username );
	g_free ( username );

	if ( user != NULL )
		return user->gr_gid;
	else
		return 0;
}

/**
	Permits to set a value in the KiazmaGidList

	@param	list	The KiazmaGidList to set
	@param	user	GID of the group to select into the widget
*/
void kiazma_gidlist_set_value ( KiazmaGidList *list, gid_t user ) {
	int i;
	struct group *item;

	i = 0;
	item = getgrent ();

	while ( item ) {
		if ( item->gr_gid == user )
			break;

		item = getgrent ();
		i++;
	}

	endgrent ();
	gtk_combo_box_set_active ( GTK_COMBO_BOX ( list ), i );
}

/**
	Reset the KiazmaGidList, selecting the current group of the user
	which is running the invoking application (the user himself)

	@param	wid	The KiazmaGidList to reset
*/
void kiazma_gidlist_clear ( KiazmaGidList *wid ) {
	gid_t now;

	now = getgid ();
	kiazma_gidlist_set_value ( wid, now );
}

/**
	Builds a new KiazmaGidList

	@return		The new KiazmaGidList
*/
GtkWidget* kiazma_gidlist_new () {
	return g_object_new ( KIAZMA_GIDLIST_TYPE, NULL );
}
