// Production in Action application. Code (C) by Benedikt Stefansson 1996

// Simple object that can output a t x agent dataset

#import "DataFile.h"

char emptyString[]="    .     ";
char formatString[]="%7.2f ";
char statisticsEmptyFormat[]="||    .       .       .       .   ";
char statisticsFormat[]="|| %7.2f %7.2f %7.2f %7.2f";

@implementation DataFile 

// Methods to make datasets
// which regist agent levevel data

-addFullDataSet: (char *) t withFeedFrom: d andSelector: (SEL) s {
	
	id fullDataSet;

	if(dataSets==NULL) dataSets=[List create:[self getZone]];
	
	fullDataSet=[FullDataSet createBegin: [self getZone]];
	[fullDataSet setDataSet: t withFeedFrom: d andSelector: s];
	[fullDataSet setManager: self];
	fullDataSet=[fullDataSet createEnd];
	
	// Add this dataset to the list we use to schedule calls
	[dataSets addLast: fullDataSet];	
	
	return self;
}

-addSimpleDataSet: (char *) t withFeedFrom: d andSelector: (SEL) s {
	id simpleDataSet;

	if(dataSets==NULL) dataSets=[List create:[self getZone]];
	
	simpleDataSet=[SimpleDataSet createBegin: [self getZone]];
	[simpleDataSet setManager: self];
	[simpleDataSet setDataSet: t withFeedFrom: d andSelector: s];
	simpleDataSet=[simpleDataSet createEnd];
	
	// Add this dataset to the list we use to schedule calls
	[dataSets addLast: simpleDataSet];	
	
	return self;
}

-addSimpleDataSet: (char *) t withFeedFrom: d andSelectorA: (SEL) sA andSelectorB: (SEL) sB {
	id simpleDataSet;
	
	if(dataSets==NULL) dataSets=[List create:[self getZone]];
	
	simpleDataSet=[SimpleDataSet createBegin: [self getZone]];
	[simpleDataSet setDataSet: t withFeedFrom: d andSelectorA: sA andSelectorB: sB];
	[simpleDataSet setManager: self];
	simpleDataSet=[simpleDataSet createEnd];
	
	// Add this dataset to the list we use to schedule calls
	[dataSets addLast: simpleDataSet];	
		
	return self;
}  

// Methods to create simple sequences which count, total or average


-addAverageSequence: (char *) t withFeedFrom: d andSelector: (SEL) s {
	id sequence;
	
	if(dataSets==NULL) dataSets=[List create: [self getZone]];
	sequence=[DataSequence createBegin: [self getZone]];
	[sequence setAverageSequence: t withFeedFrom: d andSelector: s];
	[sequence setManager: self];
	sequence=[sequence createEnd];

	// Add the sequence to the scheduling list
	[dataSets addLast: sequence];

	return self;
}


-addTotalSequence: (char *) t withFeedFrom: d andSelector: (SEL) s  {
	id sequence;
	
	if(dataSets==NULL) dataSets=[List create: [self getZone]];
	sequence=[DataSequence createBegin: [self getZone]];
	[sequence setTotalSequence: t withFeedFrom: d andSelector: s];
	[sequence setManager: self];
	sequence=[sequence createEnd];

	// Add the sequence to the scheduling list
	[dataSets addLast: sequence];

	return self;
}

-setAlwaysActive {

	if((deathProbe!=NULL)) 
		raiseEvent(WarningMessage,"DataFile: Attempt to disactivate death probe. Caution!\n");
	else 
	   deathProbe=self;

	return self;
}

-substituteCheckDeathWith: (SEL) s {
	
	// Reset the checkDeath method
	// Since it is unsafe to reset a
	// method on a probe we just kill an
	// existing probe (if it is there) and 
	// make a new one

	if((deathProbe!=NULL)) 
		[deathProbe drop];
	
	deathProbe=[MessageProbe createBegin: [self getZone]];
	[deathProbe setProbedSelector: s];
	deathProbe=[deathProbe createEnd];
	
	return self;
}


// Step function sends step to all created datasequences and datasets

-step {
	// Check if we already have a 'death probe'
	// and if not then create it	
	if((deathProbe==NULL)) {
		deathProbe=[MessageProbe createBegin: [self getZone]];
		[deathProbe setProbedSelector: M(checkDeath)];
		deathProbe=[deathProbe createEnd];
	}

	[dataSets forEach:M(step)];

	return self;
}

// Used by the underlying datasets and sequences
// to check if this agent is currently active


-(int)checkDeath: (id) anAgent {
	// If the user didn't set a death probe
	// we just assume that agents are always active
	if((deathProbe==self)) 
		return 0;
	if([deathProbe doubleDynamicCallOn: anAgent]==1) 
		return 1;
	else 
		return 0;
}

	
-(void) drop {
	[dataSets forEach:M(drop)];
	[dataSets drop];
}

	
@end

@implementation FullDataSet

-setDataSet: (char *) t withFeedFrom: d andSelector: (SEL) s {
	title=t;
	theCollection=d;
	theSelector=s;

	return self;
}


-setManager: (id) tO {
	theManager=tO;
	return self;
}


-createEnd {

	round=0; // Keeps track of the time variable
	theFile=fopen(title,"w"); // The file to print to
	
	if(!theFile){
    		fprintf(stderr,"Unable to open %s as an outfile!\n",title);
    		return nil ;
	}

	// Simple collection of one datapoint
	probe=[MessageProbe createBegin: [self getZone]];
	[probe setProbedSelector: theSelector];
	probe=[probe createEnd];

	// Add support for statistics
	[self initializeStatistics];	

	return self;
}


-step {
	id index,theMember;
	double data=0.0;

	// Start by printing the # of the current round
	round++;
	fprintf(theFile,"%4d| ",round);
	activeAgentCount=0;

	// Now iterate through the collection and ask
	// for the data, then print it to the outfile
	index=[theCollection begin: [self getZone]];
	while((theMember=[index next])) {	
	
		    data=(double)[probe doubleDynamicCallOn: theMember];

		    if([theManager checkDeath: theMember]==1){
    			fprintf(theFile,emptyString); // Member inactive
    		    } else {  
    		    	fprintf(theFile,formatString,data);
			
			activeAgentCount++;
			[self updateSummary: data]; // Make statistics
		    }
    
	}
	[index drop];

	// All agents have been called, need 
	// to update all the statistics
	[self newPeriod];
	[self printPeriodStatistics: theFile];
	// Carriage return
	fprintf(theFile,"\n");

	// Reset sums and other stuff for next period
	periodMax=-999999999.0; // A very small value - but arbitrary! 
	periodMin=999999999.0; //  A very large value - but aribtrary!
	periodSum=0.0;
	periodSumSquared=0.0;
	activeAgentCount=0.0;

	return self;
}	


-(void) drop {
	
	// Carriage return
	fprintf(theFile,"\n");
	
	fclose(theFile);
	[probe drop];
	
}

// Support for statistics


-initializeStatistics {

	// Initialize aggregate statistics
	activeAgentCount=0;
	periodSum=0.0;
	periodSumSquared=0.0;
	periodAverage=0.0;
	periodVariance=0.0;
	periodMin=999999999.0; // Hopefully above maximum possible
	periodMax=-99999999.0; // Hopefully below minum possible

	return self;
}

-updateSummary: (double) data {
	
	if(data>periodMax) 
		periodMax=data;
	if(data<periodMin) 
		periodMin=data;
	periodSum+=data;
	periodSumSquared+=pow(data,2.0);

	return self;
}

-newPeriod {
	
	// Then the period average and variance
	periodAverage=(activeAgentCount>0) ? periodSum/(double)activeAgentCount : 0.0;
	periodVariance=(activeAgentCount>0) ? periodSumSquared-((double)activeAgentCount*
				pow(periodAverage,2.0)) : 0.0;
	if(activeAgentCount>1) periodVariance /= ((double)activeAgentCount-1.0);
	
	return self;
}


-printPeriodStatistics: (FILE *) aFile {
	
	// Prints mean,variance,max and min
	// for the current period to a file

	if(activeAgentCount==0) {
		fprintf(aFile,statisticsEmptyFormat);
	} else {
		fprintf(aFile,statisticsFormat,periodAverage,periodVariance,periodMax,periodMin);
	}

	return self;
}




@end

@implementation SimpleDataSet 

-setDataSet: (char *) t withFeedFrom: d andSelector: (SEL) s {
	title=t;
	theCollection=d;
	theSelector=s;

	return self;
}

-setDataSet: (char *) t withFeedFrom: d andSelectorA: (SEL) sA andSelectorB: (SEL) sB {
	title=t;
	theCollection=d;
	theSelector=sA;
	theSelectorB=sB;
	complex=1;
	return self;
}  

-setManager: (id) tO {
	theManager=tO;
	return self;
}

-createEnd {

	round=0; // Keeps track of the time variable
	theFile=fopen(title,"w"); // The file to print to
	
	if(!theFile){
    		fprintf(stderr,"Unable to open %s as an outfile!\n",title);
    		return nil ;
  	}	
 	
	if(complex==1) {
		// For collection x,y coordinates from the same
		// agent. Have to create two different probes
		probeA=[MessageProbe createBegin: [self getZone]];
		[probeA setProbedSelector: theSelector];
		probeA=[probeA createEnd];
		
		probeB=[MessageProbe createBegin: [self getZone]];
		[probeB setProbedSelector: theSelectorB];
		probeB=[probeB createEnd];
	
	} else {
		// Simple collection of one datapoint
		probe=[MessageProbe createBegin: [self getZone]];
		[probe setProbedSelector: theSelector];
		probe=[probe createEnd];
	}

	return self;
}


-step {
	id index,theMember;
	int data=0;
	int dataA=0,dataB=0;
	
	// Start by printing the # of the current round
	round++;
	fprintf(theFile,"%4d| ",round);

	// Now iterate through the collection and ask
	// for the data, then print it to the outfile
	index=[theCollection begin: [self getZone]];
	while((theMember=[index next])) {	
	    if(complex==1) {
		    // Use this to get (x,y) type information
		    dataA=[probeA doubleDynamicCallOn: theMember];
		    dataB=[probeB doubleDynamicCallOn: theMember];
		    if([theManager checkDeath: theMember]==1) 
    			fprintf(theFile,"  .   "); // Member inactive
		    else
    			fprintf(theFile,"%2d,%2d ",dataA,dataB); // Note no statistics
	    } else {
		    // Use this to get one data point per member
		    data=(double)[probe doubleDynamicCallOn: theMember];
	
		    if([theManager checkDeath: theMember]==1) {
    			fprintf(theFile,emptyString); // Member inactive
    		    } else { 
    		    	fprintf(theFile,formatString,data);
		    }
	     }
	}
	[index drop];

	// Carriage return
	fprintf(theFile,"\n");

	return self;
}	


-(void) drop {
	
	fclose(theFile);
	if((complex)){
		[probeA drop];
		[probeB drop];
	} else {
		[probe drop];
	}
}


@end

@implementation DataSequence 

-setAverageSequence: (char *) t withFeedFrom: (id) d andSelector: (SEL) s {
	// What are we?
	average=1;
	
	title=t;
	theCollection=d;
	theSelector=s;
	return self;
}

-setTotalSequence: (char *) t withFeedFrom: (id) d andSelector: (SEL) s {
	// What are we?
	average=0;

	title=t;
	theCollection=d;
	theSelector=s;
	return self;
}


-setManager: (id) tM {
	theManager=tM;
	return self;
}


-createEnd {

	round=0; // Keeps track of the time variable
	theFile=fopen(title,"w"); // The file to print to
	
	if(!theFile){
    		fprintf(stderr,"Unable to open %s as an outfile!\n",title);
    		return nil ;
  	}	
 	
	probe=[MessageProbe createBegin: [self getZone]];
	[probe setProbedSelector: theSelector];
	probe=[probe createEnd];
	
	return self;
}


-step {
	id index,theMember;
	double data=0.0,sum=0.0;
	int active=0;
	double output=0.0;

	// Iterate through the collection and ask
	// for the data, calculate and print result to the outfile
	index=[theCollection begin: [self getZone]];
	while((theMember=[index next])) {
		if([theManager checkDeath: theMember]==0) {
    			data=[probe doubleDynamicCallOn: theMember];
			sum+=data;
			active++;
		} 
	}
    	[index drop];

	if(average) 
		output=(active>0) ? sum/(double)active : 0.0;
	else 
		output=sum;

    	fprintf(theFile,formatString,output);
	fprintf(theFile,"\n");

	return self;
}	

-(void) drop {
	[probe drop];
}	

@end









