/*$$
 * packages uchicago.src.*
 * Copyright (c) 1999, Trustees of the University of Chicago
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with 
 * or without modification, are permitted provided that the following 
 * conditions are met:
 *
 *	 Redistributions of source code must retain the above copyright notice,
 *	 this list of conditions and the following disclaimer.
 *
 *	 Redistributions in binary form must reproduce the above copyright notice,
 *	 this list of conditions and the following disclaimer in the documentation
 *	 and/or other materials provided with the distribution.
 *
 *	 Neither the name of the University of Chicago nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE TRUSTEES OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Nick Collier
 * nick@src.uchicago.edu
 *
 * packages cern.jet.random.*
 * Copyright (c) 1999 CERN - European Laboratory for Particle
 * Physics. Permission to use, copy, modify, distribute and sell this
 * software and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice appear in
 * supporting documentation. CERN makes no representations about the 
 * suitability of this software for any purpose. It is provided "as is" 
 * without expressed or implied warranty. 
 *
 * Wolfgang Hoschek
 * wolfgang.hoschek@cern.ch
 *$$*/


package space;

import java.util.Vector;

/**
 * A discrete 2 dimensional torus of objects, accessed by x and y
 * coordinates.
 *
 * @author Nick Collier
 * @version $Revision: 1.1.1.1 $ $Date: 2000/03/10 03:18:22 $
 */


public class Object2DTorus extends Object2DGrid implements Torus {


  /**
   * Creates a new torus of the specified size.
   *
   * @param xSize the size of the x dimension
   * @param ySize the size of the y dimension
   */
  public Object2DTorus(int xSize, int ySize) {
    super(xSize, ySize);
  }

  /**
   * Creates a new torus from the specified file.
   *
   * @param fileName the name of the file to create the torus from
   * @param type the type of the file
   * @see uchicago.src.sim.space.Object2DGrid
   */
  public Object2DTorus(String fileName, int type) {
    super(fileName, type);
  }

  /**
   * Puts the specified object at (x,y), wrapping the coordinates
   * around the torus if necessary.
   * @param x the x coordinate
   * @param y the y coordinate
   * @param object the object to put at (x,y)
   */
  public void putObjectAt(int x, int y, Object object) {
    matrix.put(xnorm(x), ynorm(y), object);
  }

  /**
   * Puts the specified double at (x,y), wrapping the coordinates
   * around the torus if necessary.
   * @param x the x coordinate
   * @param y the y coordinate
   * @param value the value to put at (x,y)
   */
  public void putValueAt(int x, int y, double value) {
    matrix.put(xnorm(x), ynorm(y), new Double(value));
  }

  /**
   * Gets the object at the specified coordinate
   * @param x the x coordinate
   * @param y the y coordinate
   * @return the object at x, y
   */
   public Object getObjectAt(int x, int y) {
    return matrix.get(xnorm(x), xnorm(y));
  }

  /**
   * Gets the value at (x,y)
   */
  public double getValueAt(int x, int y) {
    return super.getValueAt(xnorm(x), xnorm(y));
  }


  /**
   * Normalize the x value to the toroidal coordinates
   *
   * @param x the value to normalize
   * @return the normalized value
   */
  public int xnorm(int x) {
    while (x < 0) x += xSize;
    return x % xSize;
  }

  /**
   * Normalize the y value to the toroidal coordinates
   *
   * @param y the value to normalize
   * @return the normalized value
   */
  public int ynorm(int y) {
    while (y < 0) y += ySize;
    return y % ySize;
  }
}
