/*$$
 * packages uchicago.src.*
 * Copyright (c) 1999, Trustees of the University of Chicago
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with 
 * or without modification, are permitted provided that the following 
 * conditions are met:
 *
 *	 Redistributions of source code must retain the above copyright notice,
 *	 this list of conditions and the following disclaimer.
 *
 *	 Redistributions in binary form must reproduce the above copyright notice,
 *	 this list of conditions and the following disclaimer in the documentation
 *	 and/or other materials provided with the distribution.
 *
 *	 Neither the name of the University of Chicago nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE TRUSTEES OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Nick Collier
 * nick@src.uchicago.edu
 *
 * packages cern.jet.random.*
 * Copyright (c) 1999 CERN - European Laboratory for Particle
 * Physics. Permission to use, copy, modify, distribute and sell this
 * software and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice appear in
 * supporting documentation. CERN makes no representations about the 
 * suitability of this software for any purpose. It is provided "as is" 
 * without expressed or implied warranty. 
 *
 * Wolfgang Hoschek
 * wolfgang.hoschek@cern.ch
 *$$*/


package space;

import java.awt.Dimension;
import java.io.*;
import java.util.StringTokenizer;
import java.util.Vector;

import collection.Matrix;
import collection.BaseMatrix;
import space.Discrete2DSpace;

// import uchicago.src.sim.util.SimUtilities;

/**
 * A discrete 2 dimensional grid of objects, accessed by x and y
 * coordinates.
 *
 * @author Nick Collier
 * @version $Revision: 1.1.1.1 $ $Date: 2000/03/10 03:18:23 $
 */


public class Object2DGrid implements Discrete2DSpace {

  public static final int PGM_ASCII = 0;

  protected Matrix matrix;
  protected int xSize;
  protected int ySize;

  /**
   * Constructs a grid with the specified size.
   * @param xSize the size of the lattice in the x dimension.
   * @param ySize the size of the lattice in the y dimension.
   */
  public Object2DGrid(int xSize, int ySize) {
    this.xSize = xSize;
    this.ySize = ySize;
    matrix = new Matrix(xSize, ySize);
  }

  /**
   * Constructs a grid from a file. Only ASCII PGM files are supported
   * at this time. Code adapted from Nelson Minar's implementation of
   * SugarScape with Swarm.
   */
  public Object2DGrid(String fileName, int type) {
    if (type != PGM_ASCII) {
      throw new IllegalArgumentException("File type not supported.");
    }

    StringTokenizer tok;

    try {
      BufferedReader in = new BufferedReader(new FileReader(fileName));
      String str = in.readLine();
      if (!str.equals("P2")) {
        throw new UnsupportedEncodingException("File is not in PGM ascii format");
      }

      str = in.readLine();
      tok = new StringTokenizer(str);
      xSize = Integer.valueOf(tok.nextToken()).intValue();
      ySize = Integer.valueOf(tok.nextToken()).intValue();

      tok = null;
      in.readLine();

      str = "";
      String line = in.readLine();

      while (line != null) {
        str += line + " ";
        line = in.readLine();
      }
      in.close();

      tok = new StringTokenizer(str);
      matrix  = new Matrix(xSize, ySize);
      //System.out.println(xSize + " " + ySize);

      for (int i = 0; i < xSize; i++) {
        for (int j = 0; j < ySize; j++) {
          matrix.put(i, j, Integer.valueOf(tok.nextToken()));
        }
      }
    } catch (IOException ex) {
        // SimUtilities.showError("Error Reading image file", ex);
      ex.printStackTrace();
      System.exit(0);
    }
  }

  /**
   * Gets the von Neumann neighbors of the object at x, y. Objects are returned
   * in west, east, north, south order. The object at x, y is not returned.
   *
   * @param x the x coordinate of the object
   * @param y the y coordinate of the object
   * @param returnNulls whether nulls (nothing at x,y) should be returned
   * @return a vector of objects (and possibly nulls) in west, east, north,
   * south order
   */
  public Vector getVonNeumannNeighbors(int x, int y, boolean returnNulls) {
    return getVonNeumannNeighbors(x, y, 1, 1, returnNulls);
  }

  /**
   * Gets the extended von Neumann neighbors of the object at x, y. Then
   * extension in the x and y direction are specified by xExtent and yExtent.
   * Objects are return in west, east, north, south order with the
   * most distant object first. The Object at x,y is not returned.
   *
   * @param x the x coordinate of the object
   * @param y the y coordinate of the object
   * @param xExtent the extension of the neighborhood in the x direction
   * @param yExtent the extension of the neighborhood in the y direction
   * @param returnNulls whether nulls should be returned
   * @return a vector of objects (and possibly nulls) in west, east, north,
   * south order with the most distant object first.
   */
  public Vector getVonNeumannNeighbors(int x, int y, int xExtent, int yExtent,
                  boolean returnNulls)
  {
    Vector v = new Vector((xExtent * 2) + (yExtent * 2));

    for (int i = x - xExtent; i < x ; i++) {
      Object o = getObjectAt(i, y);
      if (returnNulls)
        v.add(o);
      else if (o != null)
        v.add(o);
    }

    for (int i = x + xExtent; i > x; i--) {
      Object o = getObjectAt(i, y);
      if (returnNulls)
        v.add(o);
      else if (o != null)
        v.add(o);
    }

    for (int i = y - yExtent; i < y; i++) {
      Object o = getObjectAt(x, i);
      if (returnNulls)
        v.add(o);
      else if (o != null)
        v.add(o);
    }

    for (int i = y + yExtent; i > y; i--) {
     Object o = getObjectAt(x, i);
      if (returnNulls)
        v.add(o);
      else if (o != null)
        v.add(o);
    }

    return v;
  }

  /**
   * Gets the Moore neighbors of the object at x, y. Objects are returned by
   * row starting with the "NW corner" and ending with the "SE corner."
   * The Object at x, y is not returned.
   *
   * @param x the x coordinate of the object
   * @param y the y coordinate of the object
   * @param returnNulls should the returned Vector contain null objects
   * @return a vector of objects (and possibly nulls) ordered by row starting
   * with the "NW corner" and ending with the "SE corner."
   */

  public Vector getMooreNeighbors(int x, int y, boolean returnNulls) {
    return getMooreNeighbors( x, y, 1, 1, returnNulls);
  }

  /**
   * Gets the extended Moore neighbors of the object at x, y. The
   * extension in the x and y direction are specified by xExtent and yExtent.
   * Objects are returned by row starting with the "NW corner" and ending with
   * the "SE corner." The Object at x,y is not returned.
   *
   * @param x the x coordinate of the object
   * @param y the y coordinate of the object
   * @param xExtent the extension of the neighborhood in the x direction
   * @param yExtent the extension of the neighborhood in the y direction
   * @param returnNulls should the returned Vector contain null objects
   * @return a vector of objects (and possibly nulls) ordered by row starting
   * with the "NW corner" and ending with the "SE corner."
   */

  public Vector getMooreNeighbors(int x, int y, int xExtent, int yExtent,
      boolean returnNulls)
  {
    Vector v = new Vector(xExtent * yExtent * 4 + (xExtent * 2) + (yExtent * 2));

    for (int j = y - yExtent; j <= y + yExtent; j++) {
      for (int i = x - xExtent; i <= x + xExtent; i++) {
        if (!(j == y && i == x)) {
          Object o = getObjectAt(i, j);
          if (returnNulls)
            v.add(o);
          else if (o != null) {
            v.add(o);
          }
        }
      }
    }

    return v;
  }

  // Discrete2dSpace interface
  /**
   * Gets the size of the x dimension.
   */
  public int getSizeX() {
    return xSize;
  }

  /**
   * Gets the size of the y dimension.
   */
  public int getSizeY() {
    return ySize;
  }

  /**
   * Gets the size as a Dimension.
   */
  public Dimension getSize() {
    return new Dimension(xSize, ySize);
  }

  /**
   * Gets the object at (x,y)
   */
  public Object getObjectAt(int x, int y) {
    return matrix.get(x, y);
  }

  /**
   * Gets the double value at (x,y) if possible
   * @throws java.lang.IllegalArgumentException if object at x,y cannot
   * be converted to a number.
   */
  public double getValueAt(int x, int y) {
    Object o = matrix.get(x, y);
    if (o instanceof Number) {
      Number n = (Number)o;
      return n.doubleValue();
    } else {
      throw new IllegalArgumentException("Object cannot be converted to a long");
    }
  }

  /**
   * Puts the specified object at (x,y)
   * @param x the x coordinate
   * @param y the y coordinate
   * @param object the object to put at (x,y)
   */
  public void putObjectAt(int x, int y, Object object) {
    matrix.put(x, y, object);
  }

  /**
   * Puts the specified double at (x,y)
   * @param x the x coordinate
   * @param y the y coordinate
   * @param value the double to put at (x,y)
   */
  public void putValueAt(int x, int y, double value) {
    matrix.put(x, y, new Double(value));
  }

  /**
   * Returns the matrix collection object associated with this 2d grid
   */
  public BaseMatrix getMatrix() {
    return matrix;
  }
}
