// JSpatialEpidemic - A simulation framework to study the spatial aspect of an epidemic.
// JSpatialEpidemic implements a model from "Bailey N.T.J., The Mathematical Theory of 
//     Infectious Diseases and its application - II ed., Charles Griffin & Co. LTD, 1975,
//     pp. 182-188"
// Copyright  2003 Pasquale Cariello, Raul Bagni & Roberto Berchi. 
// See Readme for further details.

import swarm.Globals;
import swarm.Selector;
import swarm.defobj.Zone;
import swarm.defobj.ZoneImpl;
import swarm.defobj.SymbolImpl;
import swarm.random.RandomBitDist;
import swarm.random.RandomBitDistImpl;

import swarm.defobj.FArguments;
import swarm.defobj.FArgumentsImpl;
import swarm.defobj.FCall;
import swarm.defobj.FCallImpl;

import swarm.activity.Activity;
import swarm.activity.ActionGroup;
import swarm.activity.ActionGroupImpl;
import swarm.activity.Schedule;
import swarm.activity.ScheduleImpl;
import swarm.activity.FActionForEach;

import swarm.objectbase.Swarm;
import swarm.objectbase.SwarmImpl;
import swarm.objectbase.VarProbe;
import swarm.objectbase.MessageProbe;
import swarm.objectbase.EmptyProbeMapImpl;

import swarm.collections.Index;
import swarm.collections.List;
import swarm.collections.ListImpl;

import swarm.space.Grid2d;
import swarm.space.Grid2dImpl;

/** 
 * The ModelSwarm encapsulates all the objects used in the
 * simulated world itself (but not the user interface objects)
 * */
 
public class ModelSwarm extends SwarmImpl
{
  // simulation parameters
  public int pop; //number of agents (2k+1)^2
  public double p; //infection rate per contact 
  public int lt; //latency time
  public int k; // the square lattice is bounded by the lines x = +-k, y = +-k.
  public int XGrid, YGrid; //coordinates
  
  public boolean randomizeAgentUpdateOrder;
 
  /** ActionGroup for holding an ordered sequence of action */
  public ActionGroup modelActions;
  
  /** Schedules */
  public Schedule modelSchedule;  // quello generale
    
  /** list of all the agents */
  public List agentList;   // the main list of all the living agents

  /**  the 2d world  */
  public Grid2d world, blackGrid;		
  
  /** total number of infected */
  //public int infected;  
  
  FActionForEach actionForEach;
     
  // These methods provide access to the objects inside the
  // ModelSwarm.  These objects are the ones visible to other
  // classes via message call.  In theory we could just let other
  // objects use Probes to read our state, but message access is
  // frequently more convenient.
  
  public List getAgentList () {
    return agentList;
  }
  
  public Grid2d getWorld () {
    return world;
  } 
  
  public int getPop () {
    return pop;
  } 
 
  public double getP () {
    return p;
  }
  
  public int getLt () {
    return lt;
  }
  
  public int getK () {
    return k;
  }
  
  /*
  public int getInfected() {
    return infected;
  }
  
  public void addInfected() {
    infected++;
  }

  public void remInfected() {
    infected--;
  }
  */

  
  public boolean toggleRandomizedOrder () {
    randomizeAgentUpdateOrder = !randomizeAgentUpdateOrder;
    syncUpdateOrder ();
    return randomizeAgentUpdateOrder;
  }

  public void syncUpdateOrder () {
    if (actionForEach != null)
      actionForEach.setDefaultOrder
        (randomizeAgentUpdateOrder
         ? Globals.env.Randomized
         : Globals.env.Sequential);
  }
  
  /** 
     * This method isn't normally used, but is convenient when running
     * probes: it lets you easily clone a agent and drag it into the
     * model. */
  public Object addAgent (Agent anAgent) {
    agentList.addLast (anAgent);
    return this;
  }


  public ModelSwarm (Zone aZone) {
    super (aZone);
        
    // Now fill in various simulation parameters with default values.
    k=5;
    p = 0.5;
    lt=0;
    XGrid = k*2+1;  
    YGrid = k*2+1; 
    pop = XGrid*YGrid;
    //infected=1;  //the epidemic is started by the single agent located at the origin
    
    randomizeAgentUpdateOrder = false;  //sequential
    
    // Now, build a customized probe map using a `local' subclass (a
    // special kind of Java `inner class') of the EmptyProbeMapImpl
    // class. Without a probe map, the default is to show all
    // variables and messages. Here we choose to customize the
    // appearance of the probe, to display a nicer interface.

    class SpatialModelProbeMap extends EmptyProbeMapImpl {
      private VarProbe probeVariable (String name) {
        return
          Globals.env.probeLibrary.getProbeForVariable$inClass
          (name, ModelSwarm.this.getClass ());
      }
      private MessageProbe probeMessage (String name) {
        return
          Globals.env.probeLibrary.getProbeForMessage$inClass
          (name, ModelSwarm.this.getClass ());
      }
      private void addVar (String name) {
        addProbe (probeVariable (name));
      }
      private void addMessage (String name) {
        addProbe (probeMessage (name));
      }
      public SpatialModelProbeMap (Zone _aZone, Class aClass) {
        super (_aZone, aClass);
        //addVar ("XSize");
        //addVar ("YSize");
        addVar ("k");
        addVar ("pop");
        addVar ("p");
        addVar ("lt");
        //addVar ("infected");
        //addMessage ("toggleRandomizedOrder");
        }
    }
    
    // Now, install our custom probeMap class directly into the
    // probeLibrary
    Globals.env.probeLibrary.setProbeMap$For
      (new SpatialModelProbeMap (aZone, getClass ()), getClass ());
  }
    
    
  /** 
     * Now it's time to build the model objects. We use various
     * parameters inside ourselves to choose how to create things.
     */

     
  public Object buildObjects ()
  {
    
    int i,j;
    
    // allow our parent class to build anything.
    super.buildObjects();
    
    // Recalculate the parameters (eventually changed by the model probe)
    XGrid = k*2+1;  
    YGrid = k*2+1; 
    pop = XGrid*YGrid;
    
    // First, set up objects used to represent the
    // environment.  The space agent represents the
    // spatial property of settore.  It is initialized via
    // various model parameters.
                             
    /* empty grid --> to clear the screen */
    blackGrid = new Grid2dImpl (this.getZone (), XGrid, YGrid);
       
    // Now set up the grid used to represent agent position
    world = new Grid2dImpl (this.getZone (), XGrid, YGrid);

    // Create a list to keep track of the agents in the model.
    agentList = new ListImpl (this.getZone ()); 
            
    // Create agents themselves. This is a fairly complex
    // step, as is appropriate: the agents are essential
    // aspects of the simulation.

    // First, a quick hack. During creation we might put
    // several agents in the same square. This is a design
    // flaw, but it's one that's not fatal, so we ask the
    // world object not to warn us about it. This is not an
    // example to be emulated :-)
    world.setOverwriteWarnings (false);
            
    // Now a loop to create a bunch of agents.
    pop=(k*2+1)*(k*2+1);
    for (i = 0; i <= k*2; i++)
      for (j = 0; j <= k*2; j++)  {
      Agent anAgent;
      //ZoneImpl newZone;
      /* Fill in the variables needed. Only at the end they will be set in the object */
      int state, tx, ty, square;
      
      /* health state */
      state = 0;
      if ((i == k) && (j == k)) state = 1;  // //the initial infected agent is located at the origin 
        
      /* calculate the subpopolation (square) */
      tx=i;
      ty=j;
      if (i>j) tx=2*k-i; 
        else ty=2*k-j;
      if (tx < ty) square=tx;
        else square=ty;
      
      // Now create and initialize the agent.
      //newZone = new ZoneImpl(Globals.env.globalZone);
      anAgent =  new Agent (world, this);
      anAgent.setStato(0);
      anAgent.setSquare(k-square);
      if (state == 1) { //case zero
          anAgent.setInfTime(0+lt); //it will be infective only after agentUpdate()
          }  
      else anAgent.setInfTime(-1);
      anAgent.setAgentColor((byte) (square)); 
      //System.out.println("Creating the agent located at x:"+ i + " and y:"+ j + " in the square n. " + square);
      anAgent.setX$Y(i,j);
      
      // And, finally, add the agent to the end of the list.
      agentList.addLast (anAgent);
      
    }  //end for
      
    world.setOverwriteWarnings (true);	   // ok, done cheating.
    
    return this;
    
  }

  /**
     * Here is where the model schedule is built, the data structures
     * that define the simulation of time in the mode. The core is an
     * actionGroup that has a list of actions. That's then put in a
     * Schedule.  */
  public Object buildActions () {
    super.buildActions();
    
    // Create the list of simulation actions. We put these in
    // an action group, because we want these actions to be
    // executed in a specific order, but these steps should
    // take no (simulated) time. The M(foo) means "The message
    // called <foo>". You can send a message To a particular
    // object, or ForEach object in a collection.
        
    // Note we update the space in two phases: first run
    // diffusion, then run "updateWorld" to actually enact the
    // changes the agents have made. The ordering here is
    // significant!
        
    // Note also, that with the additional
    // `randomizeAgentUpdateOrder' Boolean flag we can
    // randomize the order in which the agents actually run
    // their step rule.  This has the effect of removing any
    // systematic bias in the iteration throught the agents
    // list from timestep to timestep
        
    // By default, all `createActionForEach' modelActions have
    // a default order of `Sequential', which means that the
    // order of iteration through the `agentbugList' will be
    // identical (assuming the list order is not changed
    // indirectly by some other process).
    
    modelActions = new ActionGroupImpl (getZone ());

    syncUpdateOrder ();
        
    /** The tick is splitted in two step:
     *  1. calculate the new state of the agent;
     *  2. update the agents' state for next tick
     */ 
    try {
      Agent proto = (Agent) agentList.getFirst();
      Selector sel = 
        new Selector (proto.getClass (), "agentStep", false);
      actionForEach =
        modelActions.createFActionForEachHomogeneous$call
        (agentList,
         new FCallImpl (this, proto, sel,
                        new FArgumentsImpl (this, sel, true)));
    } catch (Exception e) {
      e.printStackTrace (System.err);
    }
    
    try {
      Agent proto = (Agent) agentList.getFirst();
      Selector sel = 
        new Selector (proto.getClass (), "agentUpdate", false);
      actionForEach =
        modelActions.createFActionForEachHomogeneous$call
        (agentList,
         new FCallImpl (this, proto, sel,
                        new FArgumentsImpl (this, sel, true)));
    } catch (Exception e) {
      e.printStackTrace (System.err);
    }
    
  
              
    // Then we create a schedule that executes the
    // modelActions. modelActions is an ActionGroup, by itself it
    // has no notion of time. In order to have it executed in
    // time, we create a Schedule that says to use the
    // modelActions ActionGroup at particular times.  This
    // schedule has a repeat interval of 1, it will loop every
    // time step.  The action is executed at time 0 relative to
    // the beginning of the loop.
  
    modelSchedule = new ScheduleImpl (getZone (), 1);
    modelSchedule.at$createAction (0, modelActions);
        
    return this;
  }
  
      
  /**
     * Now set up the model's activation. swarmContext indicates where
     * we're being started in - typically, this model is run as a
     * subswarm of an observer swarm. */
  public Activity activateIn (Swarm swarmContext) {
    // First, activate ourselves via the superclass
    // activateIn: method.  Just pass along the context: the
    // activity library does the right thing.
    super.activateIn (swarmContext);
    
    // Now activate our own schedules.
    modelSchedule.activateIn (this);

    // Finally, return our activity.
    return getActivity ();
  }
}
