/*
 * $Id: pst.c,v 1.114 2004/03/29 01:33:47 jylefort Exp $
 *
 * Copyright (c) 2003, 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include <Python.h>
#include <streamtuner.h>
#include "gettext.h"
#include "pst-category.h"
#include "pst-category-node.h"
#include "pst-handler-field.h"
#include "pst-handler.h"
#include "pst-helpers.h"
#include "pst-image.h"
#include "pst-stream.h"
#include "pst-transfer-session.h"

#define PST_API_MAJOR_VERSION		1
#define PST_API_MINOR_VERSION		0

/*** st-action-api.h *********************************************************/

static PyObject *pst_action_register (PyObject *dummy, PyObject *args);
static PyObject *pst_action_run (PyObject *dummy, PyObject *args);

/*** st-dialog-api.h *********************************************************/

static PyObject *pst_notice (PyObject *dummy, PyObject *args);
static PyObject *pst_info_dialog (PyObject *dummy, PyObject *args);
static PyObject *pst_error_dialog (PyObject *dummy, PyObject *args);
static PyObject *pst_search_dialog (PyObject *dummy, PyObject *args);

/*** st-handlers-api.h *******************************************************/

static PyObject *pst_handlers_add (PyObject *dummy, PyObject *args);

/*** st-m3u-api.h ************************************************************/

static PyObject *pst_m3u_mktemp (PyObject *dummy, PyObject *args);

/*** st-pls-api.h ************************************************************/

static PyObject *pst_pls_parse (PyObject *dummy, PyObject *args);

/*** st-state-api.h **********************************************************/

static PyObject *pst_is_aborted (PyObject *dummy, PyObject *args);

/*** st-settings-api.h *******************************************************/

static PyObject *pst_settings_get_private_dir (PyObject *dummy, PyObject *args);

/*** st-sgml-ref-api.h *******************************************************/

static PyObject *pst_sgml_ref_expand (PyObject *dummy, PyObject *args);

/*** st-transfer-api.h *******************************************************/

static PyObject *pst_transfer_escape (PyObject *dummy, PyObject *args);

/*** st-version-api.h ********************************************************/

static PyObject *pst_check_api_version (PyObject *dummy, PyObject *args);

/*** Python-specific API *****************************************************/

static PyObject *pst_find_icon (PyObject *dummy, PyObject *args);

static PyMethodDef methods[] = {
  { "action_register", pst_action_register, METH_VARARGS },
  { "action_run", pst_action_run, METH_VARARGS },

  { "notice", pst_notice, METH_VARARGS },
  { "info_dialog", pst_info_dialog, METH_VARARGS },
  { "error_dialog", pst_error_dialog, METH_VARARGS },
  { "search_dialog", pst_search_dialog, METH_NOARGS },

  { "handlers_add", pst_handlers_add, METH_VARARGS },

  { "m3u_mktemp", pst_m3u_mktemp, METH_VARARGS },

  { "pls_parse", pst_pls_parse, METH_VARARGS },

  { "is_aborted", pst_is_aborted, METH_NOARGS },

  { "settings_get_private_dir", pst_settings_get_private_dir, METH_NOARGS },

  { "sgml_ref_expand", pst_sgml_ref_expand, METH_VARARGS },

  { "transfer_escape", pst_transfer_escape, METH_VARARGS },

  { "check_api_version", pst_check_api_version, METH_VARARGS },

  { "find_icon", pst_find_icon, METH_VARARGS },
  
  { NULL, NULL, 0, NULL },
};

static char *private_icons_dir = NULL;

gboolean
pst_init (void)
{
  PyObject *module;

  private_icons_dir = g_build_filename(st_settings_get_private_dir(),
				       "streamtuner-python",
				       "icons",
				       NULL);

  module = Py_InitModule("ST", methods);
  
  PyModule_AddIntConstant(module, "MAJOR_VERSION", st_major_version);
  PyModule_AddIntConstant(module, "MINOR_VERSION", st_minor_version);
  PyModule_AddIntConstant(module, "MICRO_VERSION", st_micro_version);

  PyModule_AddIntConstant(module, "API_MAJOR_VERSION", PST_API_MAJOR_VERSION);
  PyModule_AddIntConstant(module, "API_MINOR_VERSION", PST_API_MINOR_VERSION);

  return pst_category_register(module)
    && pst_category_node_register(module)
    && pst_handler_field_register(module)
    && pst_handler_register(module)
    && pst_image_register(module)
    && pst_stream_register(module)
    && pst_transfer_session_register(module);
}

/*** st-action-api.h *********************************************************/

static PyObject *
pst_action_register (PyObject *dummy, PyObject *args)
{
  const char *id;
  const char *label;
  const char *command;

  if (! PyArg_ParseTuple(args, "sss", &id, &label, &command))
    return NULL;

  st_action_register(id, label, command);

  return pst_none();
}

static PyObject *
pst_action_run (PyObject *dummy, PyObject *args)
{
  const char *id;
  const char *uri;
  
  GError *err = NULL;

  if (! PyArg_ParseTuple(args, "ss", &id, &uri))
    return NULL;

  if (! st_action_run(id, uri, &err))
    {
      PyErr_SetString(PyExc_RuntimeError, err->message);
      g_error_free(err);

      return NULL;
    }
  
  return pst_none();
}

/*** st-dialog-api.h *********************************************************/

static PyObject *
pst_notice (PyObject *dummy, PyObject *args)
{
  const char *message;
  
  if (! PyArg_ParseTuple(args, "s", &message))
    return NULL;
  
  st_notice("%s", message);
  
  return pst_none();
}

static PyObject *
pst_info_dialog (PyObject *dummy, PyObject *args)
{
  const char *primary;
  const char *secondary;
  
  if (! PyArg_ParseTuple(args, "ss", &primary, &secondary))
    return NULL;
  
  st_info_dialog(primary, "%s", secondary);
  
  return pst_none();
}

static PyObject *
pst_error_dialog (PyObject *dummy, PyObject *args)
{
  const char *primary;
  const char *secondary;
  
  if (! PyArg_ParseTuple(args, "ss", &primary, &secondary))
    return NULL;
  
  st_error_dialog(primary, "%s", secondary);
  
  return pst_none();
}

static PyObject *
pst_search_dialog (PyObject *dummy, PyObject *args)
{
  char *str;

  str = st_search_dialog();
  if (str)
    {
      PyObject *pstr;
      
      pstr = PyString_FromString(str);
      g_free(str);

      return pstr;
    }
  else
    return pst_none();
}

/*** st-handlers-api.h *******************************************************/

static PyObject *
pst_handlers_add (PyObject *dummy, PyObject *args)
{
  PSTHandler *phandler;

  if (! PyArg_ParseTuple(args, "O!", &PSTHandler_Type, &phandler))
    return NULL;
  
  st_handlers_add(phandler->handler);

  return pst_none();
}

/*** st-m3u-api.h ************************************************************/

static PyObject *
pst_m3u_mktemp (PyObject *dummy, PyObject *args)
{
  const char *prefix;
  GSList *uri_list;

  GSList *l;
  GError *err = NULL;
  char *filename;
  PyObject *pfilename;

  if (! PyArg_ParseTuple(args, "sO&", &prefix, pst_strings_as_gslist, &uri_list))
    return NULL;
  
  filename = st_m3u_mktemp(prefix, uri_list, &err);

  /* free the list */
  for (l = uri_list; l; l = l->next)
    g_free(l->data);
  g_slist_free(uri_list);

  if (! filename)
    {
      PyErr_SetString(PyExc_RuntimeError, err->message);
      g_error_free(err);

      return NULL;
    }
      
  pfilename = PyString_FromString(filename);
  g_free(filename);

  return pfilename;
}

/*** st-pls-api.h ************************************************************/

static PyObject *
pst_pls_parse (PyObject *dummy, PyObject *args)
{
  const char *playlist;
  
  GSList *url_list;
  GSList *l;
  PyObject *tuple;

  if (! PyArg_ParseTuple(args, "s", &playlist))
    return NULL;

  url_list = st_pls_parse(playlist);
  tuple = pst_strings_from_gslist(url_list);
  
  /* free the list */
  for (l = url_list; l; l = l->next)
    g_free(l->data);
  g_slist_free(url_list);

  return tuple;
}

/*** st-state-api.h **********************************************************/

static PyObject *
pst_is_aborted (PyObject *dummy, PyObject *args)
{
  return PyInt_FromLong(st_is_aborted());
}

/*** st-settings-api.h *******************************************************/

static PyObject *
pst_settings_get_private_dir (PyObject *dummy, PyObject *args)
{
  const char *private_dir;
  
  private_dir = st_settings_get_private_dir();
  return PyString_FromString(private_dir);
}

/*** st-sgml-ref-api.h *******************************************************/

static PyObject *
pst_sgml_ref_expand (PyObject *dummy, PyObject *args)
{
  const char *str;

  char *expanded;
  PyObject *pexpanded;

  if (! PyArg_ParseTuple(args, "s", &str))
    return NULL;

  expanded = st_sgml_ref_expand(str);
  pexpanded = PyString_FromString(expanded);
  g_free(expanded);

  return pexpanded;
}

/*** st-transfer-api.h *******************************************************/

static PyObject *
pst_transfer_escape (PyObject *dummy, PyObject *args)
{
  const char *url;

  char *escaped;
  PyObject *string;

  if (! PyArg_ParseTuple(args, "s", &url))
    return NULL;

  escaped = st_transfer_escape(url);
  string = PyString_FromString(escaped);
  g_free(escaped);

  return string;
}

/*** st-version-api.h ********************************************************/

static PyObject *
pst_check_api_version (PyObject *dummy, PyObject *args)
{
  unsigned int required_major_version;
  unsigned int minimum_minor_version;

  if (! PyArg_ParseTuple(args, "ii", &required_major_version, &minimum_minor_version))
    return NULL;

  /* we check against the streamtuner/python API version */
  return PyInt_FromLong(PST_API_MAJOR_VERSION == required_major_version
			&& PST_API_MINOR_VERSION >= minimum_minor_version);
}

/*** Python-specific API *****************************************************/

static PyObject *
pst_find_icon (PyObject *dummy, PyObject *args)
{
  const char *filename;

  char *pathname;
  PyObject *result = NULL;

  if (! PyArg_ParseTuple(args, "s", &filename))
    return NULL;

  g_assert(private_icons_dir != NULL);

  pathname = g_build_filename(ICONS_DIR, filename, NULL);

  if (g_file_test(pathname, G_FILE_TEST_EXISTS))
    result = PyString_FromString(pathname);
  else
    {
      g_free(pathname);
      pathname = g_build_filename(private_icons_dir, filename, NULL);
      
      if (g_file_test(pathname, G_FILE_TEST_EXISTS))
	result = PyString_FromString(pathname);
    }

  g_free(pathname);

  if (! result)
    PyErr_Format(PyExc_RuntimeError, _("unable to find %s"), filename);
  
  return result;
}
