#!/usr/bin/python

# Copyright (C) 2006-2007 Arnau Sanchez
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License or any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation, 
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

# Standard Python modules
import os, sys, ConfigParser

##################################################
##################################################
class CfgParser(ConfigParser.ConfigParser):
	_default_confsection = "default"
	#######################################
	def __init__(self, debug=False, default_confsection=_default_confsection):
		ConfigParser.ConfigParser.__init__(self)
		self.debug = debug
		self.default_confsection = default_confsection
		self.name_debug = None
	
	#######################################
	def pdebug(self, line):
		if not self.debug: return
		if self.name_debug:
			line = "%s: %s"%(self.name_debug, line)
		sys.stderr.write(line+"\n")
		sys.stderr.flush()

	##################################################
	def read_section(self, section, options):
		config = {}	
		items = [x[0] for x in self.items(section)]
		for option in options:
			name = option["name"]
			if "type" in option:
				ptype = option["type"]
			else: ptype = "string"
			if name not in items:
				continue
			if ptype == "boolean":
				value = self.getboolean(section, name)
			elif ptype == "integer":
				value = self.getint(section, name)
			elif ptype == "list":
				value = [x.strip('"').strip() for x in self.get(section, name).split(",")]
			elif ptype == "string":
				value = self.get(section, name).strip('"').strip()
			else: 
				raise TypeError, "unknown type: %s" %ptype
			config[name] = value
		return config

	#######################################################
	def read_configuration(self, options_definition, conffile, sections=None):
		self.name_debug = conffile
		if not os.path.exists(conffile):
			self.pdebug("Configuration file not found: %s" %conffile)
			return
		self.pdebug("reading configuration: %s" %conffile)
		self.read(conffile)
		options_name = [o["name"] for o in options_definition]
		options_required = [o["name"] for o in options_definition if "needed" in o]
		default_options = dict.fromkeys(options_name)
		default_options.update(self.read_section(self.default_confsection, options_definition))
		
		config = {}
		config_sections = self.sections()
		if not sections:
			sections = config_sections
		for section in sections:
			if section not in config_sections: 
				self.pdebug("section not found: %s" %section)
				continue
			if section == self.default_confsection: 
				continue
			options = default_options.copy()
			options.update(self.read_section(section, options_definition))	
			for option in options_definition:
				name = option["name"]
				if "default" in option and options[name] is None:
					default = option["default"]
					self.pdebug("option '%s' not defined in section '%s', setting default value: %s" %(name, section, default))
					options[name] = default
			config[section] = options
		return config
