/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
package fr.gouv.culture.sdx.search.lucene.query;

import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.search.lucene.Field;
import fr.gouv.culture.sdx.search.lucene.queryparser.DefaultQueryParser;
import fr.gouv.culture.sdx.search.lucene.queryparser.QueryParser;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.search.MultiSearcher;
import org.apache.lucene.search.Searchable;
import org.apache.lucene.search.Searcher;

import java.io.IOException;
import java.util.Locale;
import java.util.Vector;

/**
 * A list of LuceneIndex where searches can be made.
 *
 */
public class SearchLocations implements LogEnabled {

    /** Avalon logger to write information. */
    private org.apache.avalon.framework.logger.Logger logger;

    /** A shortcut for locations with only one index. */
    private Index singleIndex = null;

    /** The list of indices. */
    private Vector indices = new Vector();

    /**The query parser*/
    private QueryParser queryParser = null;

    /**
     * Builds an empty search location.
     *
     * Use enableLoggin and addIndex() afterwards to add at least one index.
     *
     * @see #enableLogging
     * @see #addIndex
     */
    public SearchLocations() {
    }

    /**
     * Builds a search location with a single index.
     *
     * @param   index       The Lucene index to use.
     */
    public void setUp(Index index) {
        this.addIndex(index);
    }

    /**
     * Adds an index to locations.
     *
     * @param   index       The index to add.
     */
    /*TODO: fred and i discussed a toSAX() method could  be useful for this object, to nest in a Results.toSax(). Then,
    *this method mayb  need to take a documentbase instead of an index and we can get the index from the document base,
    *then have the id of the documentbase and also the associated index.-rbp
    */
    public void addIndex(Index index) {
        if (index != null) {
            //if this searchlocations doesn't already contain the index we add it
            if (!contains(index)) {
                this.indices.add(index);
                if (this.indices.size() > 1)
                    this.singleIndex = null;
                else
                    this.singleIndex = index;
            }
        }
    }

    /**
     * Returns the default field for the first search location (first index).
     */
    public Field getDefaultField() throws SDXException {
        if (singleIndex != null) {
            try {
                return singleIndex.getDefaultField();
            } catch (IOException e) {
                throw new SDXException(logger, SDXExceptionCode.ERROR_GET_DEFAULT_FIELD, null, e);
            }
        }
        if (indices.size() > 0) return ((LuceneIndex) indices.get(0)).getDefaultField();
        return null;
    }

    /** Returns a field given a name.
     *
     * Will return the first field having this name, <code>null</code> if none found.
     *
     * @param name The name of the field for which the Field is desired.
     * @return
     */
    public Field getField(String name) throws SDXException {
        return getFirstField(name);
    }

    /**
     * Returns the type of a field given its name.
     *
     * The first field with this name will be returned.
     *
     * @param   name    The field name.
     */
    public int getFieldType(String name) throws SDXException {
        Field f = getFirstField(name);
        if (f == null) return Field.WORD; // Default
        return f.getFieldType();
    }

    /**
     * Returns the typeName of a field given its name.
     *
     * The first field with this name will be returned.
     *
     * @param   name    The field name.
     */
    public String getTypeName(String name) throws SDXException {
        Field f = getFirstField(name);
        if (f != null)
            return f.getTypeName();
        else
            return null;
    }

    /**
     * Returns the first field object given its name.
     *
     * @param   name    The field name.
     */
    private Field getFirstField(String name) throws SDXException {
        if (name == null) return null;
        if (singleIndex != null) {
            try {
                return singleIndex.getField(name);
            } catch (IOException e) {
                throw new SDXException(logger, SDXExceptionCode.ERROR_GET_FIRST_FIELD, null, e);
            }
        }
        for (int i = 0; i < indices.size(); i++) {
            LuceneIndex index = (LuceneIndex) indices.elementAt(i);
            if (index.getField(name) != null) return index.getField(name);
        }
        // If none found, return null
        return null;
    }

    /**
     * Returns index readers for each searched index.
     *
     * The size of the returned array will always be the number of
     * search locations, but some readers may be null if we fail to
     * open one.
     */
/* Not used anymore? Opening too many readers at the same time is not a good idea I think... MS
    public IndexReader[] getIndexReaders() throws SDXException {
        IndexReader[] readers = new IndexReader[indices.size()];
        for (int i = 0; i < indices.size(); i++) {
            LuceneIndex index = (LuceneIndex) indices.elementAt(i);
            readers[i] = index.getReader();
        }
        return readers;
    }
*/

    /**
     * Returns an index reader at the specified index.
     *
     * @param idx   The index of the desired reader.
     */
    public IndexReader getIndexReader(int idx) throws SDXException {
        if (idx >= 0 && idx <= size()) {
            LuceneIndex lIndex = null;
            Index index = (Index) indices.elementAt(idx);
            if (index instanceof LuceneIndex)
                lIndex = (LuceneIndex) index;
            if (lIndex != null)
                return lIndex.getReader();
            else//we have a remote searcher and can't get a reader
                return null;
        } else
            return null;
    }


    /** Returns the locale for a field given its name.
     *
     * The first field found with this name will be used.
     *
     * @param name  The name of the field for which the Locale is desired.
     */
    public Locale getLocale(String name) throws SDXException {
        Field f = getFirstField(name);
        if (f == null) return null;
        return f.getLocale();
    }

    /**
     * Returns a searcher for these locations.
     */
    public Searcher getSearcher() throws SDXException {
        try {

            if (singleIndex != null) {
                Searchable[] searchable = new Searchable[1];
                searchable[0] = singleIndex.getSearcher();
                return new MultiSearcher(searchable);
            }

            Searchable[] searchers = new Searcher[indices.size()];
            for (int i = 0; i < indices.size(); i++) {
                LuceneIndex index = (LuceneIndex) indices.elementAt(i);
                searchers[i] = index.getSearcher();
            }
            return new MultiSearcher(searchers);
        } catch (IOException e) {
            //can't build multisearcher
            String[] args = new String[1];
            args[0] = e.getMessage();
            throw new SDXException(logger, SDXExceptionCode.ERROR_BUILD_MULTISEARCHER, args, e);
        }
    }

    /**
     * Sets the logger.
     *
     * @param   logger      The logger to use.
     */
    public void enableLogging(Logger logger) {
        this.logger = logger;
    }

    /**
     * Returns the number of indices in this search location.
     */
    public int size() {
        if (singleIndex != null)
            return 1;
        else
            return indices.size();
    }

    public QueryParser getQueryParser() {
        if (this.queryParser != null) return queryParser;
        if (singleIndex != null) {
            try {
                QueryParser qp = singleIndex.getQueryParser();
                if (qp == null) qp = new DefaultQueryParser();
                return qp;
            } catch (SDXException e) {
                return new DefaultQueryParser();
            } catch (IOException e) {
                return new DefaultQueryParser();
            }
        }

        if (indices.size() > 0) {
            try {
                return ((LuceneIndex) indices.get(0)).getQueryParser();
            } catch (SDXException e) {
                return new DefaultQueryParser();
            } catch (IOException e) {
                return new DefaultQueryParser();
            }
        }

        return new DefaultQueryParser();
    }

    public void setQueryParser(QueryParser qParser) {
        this.queryParser = qParser;
    }

    public boolean contains(Index index) {
        if (indices == null)
            return false;
        return indices.contains(index);
    }
}
