/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
 */
package fr.gouv.culture.sdx.search.lucene;

import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.framework.FrameworkImpl;
import fr.gouv.culture.sdx.search.lucene.analysis.Analyzer;
import fr.gouv.culture.sdx.search.lucene.analysis.AnalyzerManager;
import fr.gouv.culture.sdx.utils.Utilities;
import fr.gouv.culture.sdx.utils.constants.Node;
import fr.gouv.culture.sdx.utils.database.Property;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.LogEnabled;

import java.io.File;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Locale;

/**
 * Definition of fields for a document base.
 */
public class FieldsDefinition implements LogEnabled, Configurable, Cloneable {

    /** Avalon logger to write information. */
    private org.apache.avalon.framework.logger.Logger logger;

    /** Id for referencing*/
    private String id = null;

    /** The field definition's locale. */
    private Locale locale;

    /**The analyzer for this field definition.*/
    private Analyzer analyzer;

    /** The list of fields, indexed by names. */
    private Hashtable fieldTable;

    /** Default search field. */
    private String defaultField = "";

    /** Ignore case in field names or not. */
    private boolean ignoreCase = false;

    /**String representation of the 'fieldList' attribute named 'analyzerConf'. */
    private final String ATTRIBUTE_ANALYZER_CONF = "analyzerConf";

    /**String representation of the 'fieldList' attribute named 'analyzerConf'. */
    private final String ATTRIBUTE_ID = "id";

    /**String representation of the 'fieldList' attribute named 'analyzerClass'. */
    private final String ATTRIBUTE_ANALYZER_CLASS = "analyzerClass";

    /**String representation of the element name 'field'. */
    private final String ELEMENT_NAME_FIELD = Node.Name.FIELD;

    /**The application's properties. */
    private Hashtable props;

    /**<p>This object is used for indexing; it contains a list of fields for a DocumentBase object
     * and contains defaults for fields that do not specify indexing information.<p>
     *
     */
    public FieldsDefinition() {

    }

    /**
     * Sets the logger.
     *
     * @param   logger     The logger to use.
     */
    public void enableLogging(org.apache.avalon.framework.logger.Logger logger) {
        this.logger = logger;
    }

    /** Sets the configuration options for the field definition.
     *
     * @param configuration      The configuration object from which to build the field definition.
     * @throws ConfigurationException
     */
    public void configure(Configuration configuration) throws ConfigurationException {
        Utilities.checkConfiguration(configuration);

        if (!Utilities.checkString(getId())) {//if no id try to set one
            String flId = configuration.getAttribute(ATTRIBUTE_ID, null);
            setId(flId);
        }
        //getting 'analyzerConf' attr, default is null if it doesn't exist
        String confFileRelPath = configuration.getAttribute(ATTRIBUTE_ANALYZER_CONF, null);
        //getting 'analyzerClass' attr, default is null if it doesn't exist
        String className = configuration.getAttribute(ATTRIBUTE_ANALYZER_CLASS, "");

        this.locale = Utilities.buildLocale(configuration, null);

        File analyzerConfFile = null;
        try {
            if (confFileRelPath != null) {
                //verifying the attribute, if it was specified
                Utilities.checkConfAttributeValue(ATTRIBUTE_ANALYZER_CONF, confFileRelPath, configuration.getLocation());
                //getting the file
                analyzerConfFile = Utilities.resolveFile(null, configuration.getLocation(), props, confFileRelPath, false);
            }
        } catch (ConfigurationException e) {
            //we want to continue
            Utilities.logWarn(logger, null, e);
        } catch (SDXException e) {
            //we want to continue
            Utilities.logWarn(logger, null, e);
        }

        AnalyzerManager analyzeMgr = (AnalyzerManager) props.get(FrameworkImpl.ANALYZER_MGR);
        try {
            if (Utilities.checkString(className))
                this.analyzer = analyzeMgr.getAnalyzer(className, analyzerConfFile);
            else
                this.analyzer = analyzeMgr.getAnalyzer(locale, analyzerConfFile);
        } catch (SDXException sdxE) {
            throw new ConfigurationException(sdxE.getMessage(), sdxE.fillInStackTrace());
        }

        //giving the analyzer a logger
        this.analyzer.enableLogging(logger);

        configureFields(configuration);


    }

    public void configureFields(Configuration configuration) throws ConfigurationException {

        //at this point, we should have a <sdx:fieldList> element containing a list of fields
        Configuration[] fields = new Configuration[configuration.getChildren(ELEMENT_NAME_FIELD).length];
        //getting an array of configuration objects from each of the <sdx:fields> subElements of <sdx:fieldList> element
        fields = configuration.getChildren(ELEMENT_NAME_FIELD);
        //testing to see if we have something
        if (fields == null | fields.length == 0) {
            String[] args = new String[1];
            //getting the location of the configuration file
            args[0] = configuration.getLocation();
            //null logger passed to prevent double logging
            SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_NO_FIELDS_IN_CONFIG, args, null);
            throw new ConfigurationException(sdxE.getMessage(), sdxE);
        }

        Field firstField = null;
        //building fields objects
        for (int i = 0; i < fields.length; i++) {
            //building the field
            Field field = new Field();
            //passing the logger from the fieldsDef
            field.enableLogging(logger);
            //setting properties
            field.setProperties(this.props);
            //setting up the field
            field.setUp(this.locale, this.analyzer, null, null);//using the default analyzer class variable and locale for the fieldlist
            //configuring with each configuration object from the array
            field.configure(fields[i]);
            try {
                //adding them to this the fieldsDef
                add(field);
                if (i == 0) firstField = field;
            } catch (SDXException e) {
                throw new ConfigurationException(e.getMessage(), e.fillInStackTrace());
            }
        }
        //if there is no default field we use the first field
        if (!Utilities.checkString(this.defaultField)) this.defaultField = firstField.getCode();

    }

    /** Adds the internal fields defined in sdx.xconf to the list to allow searching
     * using these fields
     *
     * @param conf      The configuration object containing the list of internal fields
     * @throws SDXException
     * @throws ConfigurationException
     */
    public void addInternalFields(Configuration conf) throws SDXException, ConfigurationException {
        Configuration[] fields = conf.getChildren(ELEMENT_NAME_FIELD);
        //testing to see if we have something
        if (fields == null | fields.length == 0) {
            String[] args = new String[1];
            //getting the location of the configuration file
            args[0] = conf.getLocation();
            //null logger passed to prevent double logging
            throw new SDXException(null, SDXExceptionCode.ERROR_NO_FIELDS_IN_CONFIG, args, null);
        }
        for (int i = 0; i < fields.length; i++) {
            Field field = new Field();
            field.enableLogging(this.logger);
            //setting properties
            field.setProperties(this.props);
            //setting up the field
            field.setUp(this.locale, analyzer, null, null);
            field.configure(fields[i]);
            add(field);
        }
    }

    /**Provides access to the analyzer for the field definition.
     *
     * @return The analyzer for the field definition.
     */
    public Analyzer getAnalyzer() {
        return analyzer;
    }

    /**
     *	Returns the default search field.
     */
    public Field getDefaultField() {
        return (Field) this.fieldTable.get(defaultField);
    }

    /**
     *	Returns the field type.
     *
     *	@param	name	The name of the field.
     */
    public int getFieldType(String name) {
        if (name == null) return Field.WORD;		// Default value
        Field f = (Field) fieldTable.get(normalizeName(name));
        if (f == null) return Field.WORD;
        return f.getFieldType();
    }

    /**
     *	Adds a field to the list
     *
     *	@param	field		The field to add.
     */
    public void add(Field field) throws SDXException {
        if (fieldTable == null) fieldTable = new Hashtable();

        if ((Utilities.checkString(defaultField)) & field.isDefault()) {
            //a default field already exists please correct your application configuration file
            String[] args = new String[1];
            args[0] = field.getCode();
            throw new SDXException(logger, SDXExceptionCode.ERROR_DEFAULT_FIELD_EXISTS, args, null);
        }

        //otherwise we assign a value to the defaultField
        else if (field.isDefault())
            defaultField = field.getCode();

        //adding the field to the data structure
        fieldTable.put(field.getCode(), field);
    }

    /**
     *	Normalise le nom d'un champ en fonction des r�gles de capitalisation.
     */
    private String normalizeName(String name) {
        if (name == null || !ignoreCase)
            return name;
        else
            return name.toLowerCase();
    }

    /**
     * Gets a Lucene field from a property.
     *
     *@param prop   The property from which the field will be dervived
     */
    public org.apache.lucene.document.Field getLuceneField(Property prop) throws SDXException {
        Field f = null;

        if (prop != null) f = (Field) fieldTable.get(prop.getName());
        if (f != null)
            return f.getLuceneField(prop.getValue());
        else
            return null;
    }

    /**
     * Returns a field given a field name.
     *
     *@param  name  The field name
     */
    public Field getField(String name) {
        return (Field) this.fieldTable.get(name);
    }

    /**
     * Returns a locale for a given field.
     */
    public Locale getLocale(String name) {
        return ((Field) this.fieldTable.get(name)).getLocale();
    }

    /**Provides access to the locale for the field definition. */
    public Locale getLocale() {
        return locale;
    }

    /**Sets the properties object.
     *
     * @param props The properties object
     */
    public void setProperties(Hashtable props) {
        this.props = props;
    }

    /**Returns all fields in the FieldsDefinition*/
    public Enumeration getFields() {
        if (fieldTable != null)
            return fieldTable.elements();
        else
            return null;
    }

    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    /** Returns a clone of this FieldsDef. */
    public Object clone() {
        try {
            return (FieldsDefinition) super.clone();
        } catch (CloneNotSupportedException e) {
            //TODOException:
            throw new RuntimeException("Clone not supported: " + e.getMessage());
        }
    }

}
