; Copyright (c) 2004, Jonah Nathaniel Beckford
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without
; modification, are permitted provided that the following conditions
; are met:
;
;   Redistributions of source code must retain the above copyright
;   notice, this list of conditions and the following disclaimer.
;
;   Redistributions in binary form must reproduce the above copyright
;   notice, this list of conditions and the following disclaimer in
;   the documentation and/or other materials provided with the
;   distribution.
;
;   Neither the name of the author nor the names of its contributors
;   may be used to endorse or promote products derived from this
;   software without specific prior written permission.
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
; "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
; LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
; FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
; COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
; INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
; (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
; HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
; STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
; ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
; OF THE POSSIBILITY OF SUCH DAMAGE.
;
; jonah@usermail.com

(cond-expand
  [chicken
    (require-extension extras) ; hash-table
    (define (hash-table-map proc ht)
      (##sys#check-structure ht 'hash-table 'hash-table->list)
      (let* ([vec (##sys#slot ht 1)]
             [len (##sys#size vec)] )
        (let loop ([i 0] [lst '()])
          (if (fx>= i len)
              lst
              (let loop2 ([bucket (##sys#slot vec i)] [lst lst])
                (if (null? bucket)
                    (loop (fx+ i 1) lst)
                    (loop2 (##sys#slot bucket 1)
                      (let ([x (##sys#slot bucket 0)])
                        (cons
                          (proc (##sys#slot x 0) (##sys#slot x 1))
                          lst)))))))))

    ]
  (else))

(cond-expand
  [(and chicken srfi-40)
    (define (hash-table->stream ht)
      (##sys#check-structure ht 'hash-table 'hash-table->list)
      (let* ([vec (##sys#slot ht 1)]
             [len (##sys#size vec)] )
	(let iter ([i 0] [lst stream-null])
	  (stream-delay
            (if (fx>= i len)
                lst
                (let loop ([bucket (##sys#slot vec i)] [lst lst])
                  (if (null? bucket)
                      (iter (fx+ i 1) lst)
                      (loop (##sys#slot bucket 1)
                        (let ([x (##sys#slot bucket 0)])
                          (stream-cons
                            (cons (##sys#slot x 0) (##sys#slot x 1))
                            lst))))))))))]
  (else))

(cond-expand
  [srfi-40
    (define (vector->stream vct)
      (let ([l (vector-length vct)])       
        (let iter ([i 0])
          (stream-delay
            (cond
              [(= i l) stream-null]
              [else (stream-cons (vector-ref vct i) (iter (add1 i)))])))))
    (define (list->stream lst)
      (let iter ([lst lst])
        (stream-delay
          (cond
            [(null? lst) stream-null]
            [else (stream-cons (car lst) (iter (cdr lst)))]))))

    (define (stream->list strm)
      (cond
        [(null? strm) '()]
        [else (let iter ([strm strm])
                (if (stream-null? strm)
                    '()
                    (cons (stream-car strm)
                      (iter (stream-cdr strm)))))]
        ))
    
    (define (stream-append . strms)
      (let iter ([strm stream-null] [lstrm strms])
	(stream-delay
	 (cond 
	  ((stream-null? strm)
	   (if (null? lstrm) 
	       stream-null 
	       (iter (car lstrm) (cdr lstrm))))
	  (else (stream-cons (stream-car strm) (iter (stream-cdr strm) lstrm)))))))
    ]
  (else))

;; Get rid of dependency on SRFI-1
(define (rgraph-count pred lst)
  (let loop ([l lst] [i 0])
    (if (null? l)
	i
	(loop (cdr l) (if (pred (car l)) (add1 i) i)))))

(include "rgraph-prop.scm")
(include "rgraph-vis.scm")

;; Fill graph from a list of edges, where each edge is a pair of the
;; form '(vertex1 . vertex2).  vertex1, vertex2, etc. must be
;; comparable using eq? [or VERTEX-EQ? if defined].  Gets mutated
;; graph.  Will fill internal property 'vertex-name if defined.
(define (fill-graph! graph edges . vertex-eq?)
  (define vertex-equal?
    (cond [(pair? vertex-eq?) (car vertex-eq?)] [else eq?]))
  (define h (make-hash-table vertex-equal?))
  (define vertex-name (graph:get graph 'vertex-name))
  (for-each
    (lambda (edge)
      (let* ([v1 (car edge)]
             [v2 (cdr edge)]
             [vertex1 (hash-table-ref h v1)]
             [vertex2 (hash-table-ref h v2)])
        (cond [(not vertex1)
               (set! vertex1 (graph:add-vertex! graph))
               (when vertex-name (prop:put! vertex-name vertex1 v1))
               (hash-table-set! h v1 vertex1)])
        (cond [(not vertex2)
               (set! vertex2 (graph:add-vertex! graph))
               (when vertex-name (prop:put! vertex-name vertex2 v2))
               (hash-table-set! h v2 vertex2)])
        (graph:add-edge! graph vertex1 vertex2)))
    edges)
  graph)

