; Copyright (c) 2004, Jonah Nathaniel Beckford
; All rights reserved.
;
; Redistribution and use in source and binary forms, with or without
; modification, are permitted provided that the following conditions
; are met:
;
;   Redistributions of source code must retain the above copyright
;   notice, this list of conditions and the following disclaimer.
;
;   Redistributions in binary form must reproduce the above copyright
;   notice, this list of conditions and the following disclaimer in
;   the documentation and/or other materials provided with the
;   distribution.
;
;   Neither the name of the author nor the names of its contributors
;   may be used to endorse or promote products derived from this
;   software without specific prior written permission.
;
; THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
; "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
; LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
; FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
; COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
; INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
; (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
; SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
; HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
; STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
; ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
; OF THE POSSIBILITY OF SUCH DAMAGE.
;
; jonah@usermail.com

;;;; A <strong>property getter</strong> is a procedure <code>(lambda
;;;; (graph vertex-or-edge))</code> that gets an arbitrary scheme
;;;; object that is associated with the vertex-or-edge in graph.<br>
;;;;
;;;; A <strong>property setter</strong> is a procedure <code>(lambda
;;;; (graph vertex-or-edge value))</code> that associates an arbitrary
;;;; scheme object to the vertex-or-edge in graph.<br>
;;;;
;;;; A <strong>property map</strong> is a pair <code>(cons
;;;; property-getter property-setter!)</code>, whose car is a property
;;;; getter and whose cdr is a property setter.<br>
;;;;
;;;; A <strong>property key</strong> is a vertex or an edge, dependent
;;;; upon whether the context is a vertex property or an edge
;;;; property, respectively.<br>

;;;			      Properties.
;;;
;;; <p>The internal properties are defined when you define the graph.
;;; See <a href="rgraph-al.scm">Adjacency List</a> for an example.</p>

;; Create an external property map on top of a hashtable.
;;
;; .form (prop-external-hash eq? [num])
;; .parameter eq? A procedure that will equality compare two property keys.
;; .parameter num Optional.  The initial size of the hash-table.
;; .returns A property map.
(define (prop-external-hash eq? . num)
  (let ([store (if (null? num) 
		   (make-hash-table eq?) 
		   (make-hash-table eq? (car num)))])
    (cons
     (lambda (g k) (hash-table-ref store k))
     (lambda (g k v) (hash-table-set! store k v)))))

;; Create an external property map on top of a vector.  The property
;; key must be an integer.
;;
;; .form (prop-external-vector [num])
;; .parameter num Optional.  The initial size of the vector
;; .returns A property map.
(define (prop-external-vector . num)
  (let ([store (if (null? num) 
		   (make-vector)
		   (make-vector (car num)))])
    (cons
     (lambda (g k) (vector-ref store k))
     (lambda (g k v) 
       (when (> k (vector-length store))
	 (set! store (vector-resize! store (max 2 (* 1.7 k)))))
       (vector-set! store k v)))))
