------------------------------------------------------------------------------
-- RAPID is free software;  you can  redistribute it  and/or modify
-- it under terms of the  GNU General Public License as published
-- by the Free Software  Foundation;  either version 2,  or (at your
-- option) any later version.  RAPID is distributed in the hope that
-- it will be useful, but WITHOUT ANY WARRANTY;  without even the
-- implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
-- PURPOSE.  See the GNU General Public License for more details.
-- You should have  received  a copy of the GNU General Public License
-- distributed with RAPID; see file COPYING.  If not, write to the
-- Free Software Foundation,  59 Temple Place - Suite 330,  Boston,
-- MA 02111-1307, USA.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.  This exception does not apply to executables which
-- are GUI design tools, or that could act as a replacement
-- for RAPID.
--
-- Copyright (C) 2002, Martin C. Carlisle <carlislem@acm.org>
------------------------------------------------------------------------------
with Ada.Text_IO;             use Ada.Text_IO;
with Ada.Strings.Unbounded;   use Ada.Strings.Unbounded;
with Ada.Characters.Handling; use Ada.Characters.Handling;
with Ada.Strings.Fixed;       use Ada.Strings.Fixed;
with gui.Window;
with gui.Widget;
with gui.Widget.Button;
with gui.Menu;
with File_Helpers;

package body Novice_Mode is
   type Ada_File is array (Natural range <>) of Unbounded_String;

   function Get_Line (File : in File_Type) return Unbounded_String is
      Line : String (1 .. 80);
      Last : Natural;
   begin
      Get_Line (File => File, Item => Line, Last => Last);
      if Last < Line'Last then
         return To_Unbounded_String (Line (1 .. Last));
      else
         return Line & Get_Line (File);
      end if;
   end Get_Line;

   function Count_Lines (Filename : in String) return Natural is
      Count : Natural := 0;
      File  : File_Type;
   begin
      Open (File => File, Name => Filename, Mode => In_File);
      while not End_Of_File (File) loop
         Skip_Line (File);
         Count := Count + 1;
      end loop;
      Close (File);
      return Count;
   exception
      when others =>
         return Count;
   end Count_Lines;

   procedure Read_File (Filename : in String; Lines : out Ada_File) is
      File : File_Type;
   begin
      if Lines'Length > 0 then
         Open (File => File, Name => Filename, Mode => In_File);
         for I in Lines'Range loop
            Lines (I) := Get_Line (File);
         end loop;
         Close (File);
      end if;
   end Read_File;

   --------------------
   -- Generate_Files --
   --------------------

   procedure Generate_Files
     (Window_Name : in String;
      Window      : in gui.Window.GUI_Window)
   is
      Filename : constant String :=
         File_Helpers.Convert_Window_Name (Window_Name) & "_actions";
      Spec_Count       : Natural := Count_Lines (Filename & ".ads");
      Body_Count       : Natural := Count_Lines (Filename & ".adb");
      Spec_Lines       : Ada_File (1 .. Spec_Count);
      Body_Lines       : Ada_File (1 .. Body_Count);
      Spec_File        : File_Type;
      Body_File        : File_Type;
      Widgets          : gui.Widget.Widget_List := Window.Widget_List;
      Output_Something : Boolean := False;

      procedure Output_Headers_Or_Previous is
      begin
         if Spec_Count = 0 then
            Put_Line (Spec_File, "with Mcc.Gui.Widget.Button;");
            Put_Line (Spec_File, "with Mcc.Gui.Menu;");
            Put_Line (Spec_File, "package " & Window_Name & "_Actions is");
         else
            for I in Spec_Lines'Range loop
               exit when Index
                            (To_Lower (To_String (Spec_Lines (I))),
                             "end " & To_Lower (Window_Name) & "_actions") >
                         0;
               Put_Line
                 (File => Spec_File,
                  Item => To_String (Spec_Lines (I)));
            end loop;
            for I in Body_Lines'Range loop
               exit when Index
                            (To_Lower (To_String (Body_Lines (I))),
                             "end " & To_Lower (Window_Name) & "_actions") >
                         0;
               Put_Line
                 (File => Body_File,
                  Item => To_String (Body_Lines (I)));
            end loop;
            Output_Something := True;
         end if;
         if Body_Count = 0 then
            Put_Line (Body_File, "with " & Window_Name & "_Window;");
            Put_Line
              (Body_File,
               "package body " & Window_Name & "_Actions is");
         end if;
      end Output_Headers_Or_Previous;

      procedure Update_Button (Widget : gui.Widget.Button.Button'Class) is
         Found : Boolean;
      begin
         -- if this wasn't specified to be in the actions package,
         -- then ignore it
         if Index
               (To_Lower (Widget.Action.all),
                To_Lower (Window_Name) & "_actions") <=
            0
         then
            return;
         end if;
         Output_Something := True;
         Found            := False;
         for I in Spec_Lines'Range loop
            if Index
                  (To_Lower (To_String (Spec_Lines (I))),
                   "procedure " & To_Lower (Widget.Name.all) & "_pushed") >
               0
            then
               Found := True;
            end if;
         end loop;
         if not Found then
            Put_Line
              (File => Spec_File,
               Item => "   procedure " &
                       Widget.Name.all &
                       "_Pushed" &
                       "(Obj : in out Mcc.Gui.Widget.Button.Button'Class);");
         end if;
         for I in Body_Lines'Range loop
            if Index
                  (To_Lower (To_String (Body_Lines (I))),
                   "procedure " & To_Lower (Widget.Name.all) & "_pushed") >
               0
            then
               Found := True;
            end if;
         end loop;
         if not Found then
            New_Line (Body_File);
            Put_Line
              (File => Body_File,
               Item => "   procedure " &
                       Widget.Name.all &
                       "_Pushed" &
                       "(Obj : in out Mcc.Gui.Widget.Button.Button'Class) is");
            Put_Line (File => Body_File, Item => "   begin");
            Put_Line (File => Body_File, Item => "      null;");
            Put_Line
              (File => Body_File,
               Item => "   end " & Widget.Name.all & "_Pushed;");
         end if;
      end Update_Button;
      function Already_Present_Callback
        (Spec        : in Boolean;
         Action_Name : in String;
         Name        : in String)
         return        Boolean
      is
      begin
         -- if not a {Window_Name}_Actions. then skip
         if Index
               (To_Lower (Action_Name),
                To_Lower (Window_Name) & "_actions.") <=
            0
         then
            return True;
         end if;
         if Spec then
            for I in Spec_Lines'Range loop
               if Index
                     (To_Lower (To_String (Spec_Lines (I))),
                      "procedure " & To_Lower (Name) & "_selected") >
                  0
               then
                  return True;
               end if;
            end loop;
            return False;
         else
            for I in Body_Lines'Range loop
               if Index
                     (To_Lower (To_String (Body_Lines (I))),
                      "procedure " & To_Lower (Name) & "_selected") >
                  0
               then
                  return True;
               end if;
            end loop;
            return False;
         end if;
      end Already_Present_Callback;
   begin
      Read_File
        (Filename => Filename & ".ads",
         Lines    => Spec_Lines);
      Read_File
        (Filename => Filename & ".adb",
         Lines    => Body_Lines);
      Create
        (File => Body_File,
         Mode => Out_File,
         Name => Filename & ".adb");
      Create
        (File => Spec_File,
         Mode => Out_File,
         Name => Filename & ".ads");
      Output_Headers_Or_Previous;
      declare
         Current_Widget_Position : gui.Widget.Widget_List_Package.Position;
         Found                   : Boolean := False;
         Widget_Ptr              : gui.Widget.Widget_Access;
      begin
         Current_Widget_Position :=
            gui.Widget.Widget_List_Package.First (L => Widgets);
         while not gui.Widget.Widget_List_Package.IsPastEnd
                     (L => Widgets,
                      P => Current_Widget_Position)
         loop
            Widget_Ptr :=
               gui.Widget.Widget_List_Package.Retrieve
                 (L => Widgets,
                  P => Current_Widget_Position);
            if Widget_Ptr.all in gui.Widget.Button.Button'Class then
               Update_Button
                 (gui.Widget.Button.Button'Class (Widget_Ptr.all));
            end if;
            gui.Widget.Widget_List_Package.GoAhead
              (L => Widgets,
               P => Current_Widget_Position);
         end loop;
      end;
      gui.Menu.Generate_Menu_Novice
        (Menu      => Window.Menu,
         Spec_File => Spec_File,
         Body_File => Body_File,
         Callback  => Already_Present_Callback'Unrestricted_Access);

      Put_Line
        (File => Body_File,
         Item => "end " & Window_Name & "_Actions;");
      if not Output_Something then
         Put_Line (File => Spec_File, Item => "   pragma Elaborate_Body;");
      end if;
      Put_Line
        (File => Spec_File,
         Item => "end " & Window_Name & "_Actions;");

      Close (Body_File);
      Close (Spec_File);
   end Generate_Files;

end Novice_Mode;
