# -*- coding: utf-8 -*-
# Author:   $Author: merkosh $
# Revision: $Rev: 685 $
############################################################################
#    Copyright (C) 2006 by Uwe Mayer                                       #
#    merkosh@hadiko.de                                                     #
#                                                                          #
#    This is free software; you can redistribute it and/or modify it       #
#    under the terms of the GNU Lesser General Public License as           #
#    published by the Free Software Foundation; either version 2.1 of      #
#    the License, or (at your option) any later version.                   #
#                                                                          #
#    This software is distributed in the hope that it will be useful,      #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU      #
#    Lesser General Public License for more details.                       #
#                                                                          #
#    You should have received a copy of the GNU Lesser General Public      #
#    License along with this software; if not, write to the                #
#    Free Software Foundation, Inc.,                                       #
#    51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.         #
############################################################################
"""
Replacement for the python built-in wave module.

The python wave module comes with a number of restrictions, i.e.
- no (de-)compression
- mono and stereo channels only
- no 24, 32 bit audio files
- no mixed read /write
- ...

sndfile.wave implements the same interface, but uses libsndfile
as backend library, which does not suffer from all of these limitations.
"""
from sndfile import *


#-- some error the `wave` module raises ----------------------------------------
Error = ValueError


#-- used to open files for read and write access -------------------------------
def open(file, mode='r'):
    """open(file, [mode='r']) -> Wave_read /Wave_write object"""
    # read support
    if (mode in ('r', 'rb')):
        i = SF_INFO()
        f = sf_open(file, SFM_READ, i)
        if not f: raise IOError("Could not open file for read: %s"%file)
        return Wave_read(f, i)

    # write support
    elif (mode in ('w', 'wb')):
        i = SF_INFO()
        f = sf_open(file, SF_WRITE, i)
        if (not f): raise IOError("Could not open file for write: %s"%file)
        return Wave_write(f, i)

    else:
        raise Error("Valid values for mode are 'r', 'rb', 'w', 'wb'")
    
        
#-------------------------------------------------------------------------------
class Wave_read(object):
    _comp = {SF_FORMAT_ULAW: 'ulaw',
             SF_FORMAT_ALAW: 'alaw',
             SF_FORMAT_IMA_ADPCM: 'ima adpcm',
             SF_FORMAT_MS_ADPCM: 'ms adpcm',
             SF_FORMAT_GSM610: 'gsm610',
             SF_FORMAT_VOX_ADPCM: 'vox adpcm',
             SF_FORMAT_G721_32: 'g721 32',
             SF_FORMAT_G723_24: 'g723 24',
             SF_FORMAT_G723_40: 'g723 40',
             SF_FORMAT_DWVW_12: 'dwvw 12',
             SF_FORMAT_DWVW_16: 'swvw 16',
             SF_FORMAT_DWVW_24: 'swvw 24',
             SF_FORMAT_DWVW_N: 'swvw n',
             SF_FORMAT_DPCM_8: 'dpcm 8',
             SF_FORMAT_DPCM_16: 'dpcm 16',
             }

    def __init__(self, f, info):
        """support for reading audio files"""
        if not isinstance(info, SF_INFO): raise Error("SF_INFO object required")

        self._f = f
        self._info = info
        self._rpos = 0

    def close(self):
        """close the stream, make the instance unuseable

        This is called automatically on object collection.
        """
        if self._f: sf_close(self._f)

    def __del__(self):
        """collect object; automatically closes the file"""
        self.close()

    def getnchannels(self):
        """returns the number of audio channels"""
        return self._info.channels

    def getsampwidth(self):
        """returns the sample width in bytes or -1 if not PCM or floating point"""
        if   (self._info.format & SF_FORMAT_PCM_S8):   return 1
        elif (self._info.format & SF_FORMAT_PCM_U8): return 1
        elif (self._info.format & SF_FORMAT_PCM_16): return 2
        elif (self._info.format & SF_FORMAT_PCM_24): return 3
        elif (self._info.format & SF_FORMAT_PCM_32): return 4
        elif (self._info.format & SF_FORMAT_FLOAT):  return 4
        elif (self._info.format & SF_FORMAT_DOUBLE): return 8
        else: return -1

    def getframerate(self):
        """returns sampling frequency"""
        return self._info.samplerate

    def getnframes(self):
        """returns number of audio samples"""
        return self._info.frames

    def getcomptype(self):
        """returns compression type

        As this is `None` for the original wave module, this will return
        the rest of the libsndfile subtype constants.
        """
        if (Wave_read._comp.has_key( self._info.format & SF_FORMAT_SUBMASK )):
            return self._info.format & SF_FORMAT_SUBMASK

        # returns None

    def getcompname(self):
        """returns human-readable version of getcomptype()

        The wave module usually returns 'not compressed', this
        method adds string descriptions for the rest of the
        compression types supported by libsndfile.
        """
        if (Wave_read._comp.has_key (self._info.format & SF_FORMAT_SUBMASK )):
            return Wave_read._comp[ self._info.format & SF_FORMAT_SUBMASK ]

    def getparams(self):
        """returns a tuple equivalent to the output of the get*() methods

        result: (nchannels, sampwidth, framerate, nframes, comptype, compname)
        """
        return (self.getnchannels(),
                self.getsampwidth(),
                self.getframerate(),
                self.getnframes(),
                self.getcomptype(),
                self.getcompname())

    def readframes(self, n):
        """reads and returns at most n frames of audio as string of bytes"""
        width = self.getsampwidth()
        if not width: raise RuntimeError("Could not determine sample width: %s"%width)
        self._rpos += n
        return sf_read_raw(self._f, n*self.getnchannels()*width)
    
    def rewind(self):
        """rewind the file pointer to the beginning of the file"""
        self._rpos = 0
        sf_seek(self._f, 0, SEEK_SET)

    def getmarkers(self):
        """returns None"""
        return None

    def getmark(self, id):
        """raise an error"""
        raise Exception("invalid method error")

    def setpos(self, pos):
        """set the file pointer to the specified position

        Since the wave module does not define the `whence' position
        this is taken to be the beginning of the stream.
        """
        self._rpos = pos
        sf_seek(self._f, pos, SEEK_SET)

    def tell(self):
        """return file pointer position"""
        return self._rpos



#-------------------------------------------------------------------------------
class Wave_write(object):
    def __init__(self, name):
        """support for writing audio files"""
        if not isinstance(info, SF_INFO): raise Error("SF_INFO object required")

        self._name = name
        self._f = None
        self._info = SF_INFO(SF_FORMAT_WAV | SF_FORMAT_PCM_16)

        self._wpos = 0
        self._sampwidth = 2
    
    def close(self):
        """close the stream, make the instance unuseable

        This is called automatically on object collection.
        """
        if self._f: sf_close(self._f)

    def __del__(self):
        """collect object; automatically closes the file"""
        self.close()

    def setnchannels(self, n):
        """set the number of channels

        This only works if the file has not yet been written to.
        """
        self._info.channels = n

    def setsampwidth(self, n):
        """set the sample width to `n` bytes

        This only works if the file has not yet been written to.
        """
        MASK = SF_FORMAT_TYPEMASK | SF_FORMAT_ENDMASK
        if   (n == 1): self._info.format = (self._info.format & ~MASK) | SF_FORMAT_PCM_S8
        elif (n == 2): self._info.format = (self._info.format & ~MASK) | SF_FORMAT_PCM_16
        elif (n == 3): self._info.format = (self._info.format & ~MASK) | SF_FORMAT_PCM_24
        elif (n == 4): self._info.format = (self._info.format & ~MASK) | SF_FORMAT_FLOAT
        elif (n == 8): self._info.format = (self._info.format & ~MASK) | SF_FORMAT_DOUBLE
        else: raise Error("expected value must be 1,2,4 or 8")
        self._sampwidth = n

    def setframerate(self, n):
        """set framerate to `n`

        This only works if the file has not yet been written to.
        """
        self._info.samplerate = n

    def setnframes(self, n):
        """set number of frames to `n`

        For libsndfile this is a noop.
        """
        pass

    def setcomptype(self, type, name=""):
        """set compression type and description

        `type` must be one of:
           SF_FORMAT_ULAW
           SF_FORMAT_ALAW
           SF_FORMAT_IMA_ADPCM
           SF_FORMAT_MS_ADPCM
           SF_FORMAT_GSM610
           SF_FORMAT_VOX_ADPCM
           SF_FORMAT_G721_32
           SF_FORMAT_G723_24
           SF_FORMAT_G723_40
           SF_FORMAT_DWVW_12
           SF_FORMAT_DWVW_16
           SF_FORMAT_DWVW_24
           SF_FORMAT_DWVW_N
           SF_FORMAT_DPCM_8
           SF_FORMAT_DPCM_16

        `name` is ignored and can be left off.

        This only works if the file has not yet been written to.
        """
        if (not Wave_read._comp.has_key(type)):
            raise Error("invalid compression type")

        self._info.format |= type

    def setparams(self, tuple):
        """sets all parameters

        `tuple` is a 6-tuple: (nchannels, sampwidth, framerate, nframes, comptype, compname)
        and must contain valid values for the set*() methods.
        """
        self.setnchannels(tuple[0])
        self.setsampwidth(tuple[1])
        self.setframerate(tuple[2])
        self.setnframes(tuple[3])
        self.setcomptype(tuple[4], "")

    def tell(self):
        """return the current position in the file

        As seeking is not allowed for write-only files by wave, this
        is will return the value of an internal file pointer, not the
        value of the sndfile.sf_tell() function.
        """
        return self._wpos

    def writeframes(self, data):
        """write audio frames and make sure `nframes` is correct

        returns: number of bytes written; should be len(data)
        """
        if (len(data) %(self._sampwidth*self._info.channels) != 0):
            raise Error("data length not multiple of samplewidth*channels")

        if (self._f == None):
            self._f = sf_open(self._name, SFM_WRITE, self._info)
            if (not self._f): raise IOError("could not open file for writing")
                
        tmp = sf_write_raw(self._f, data)

        self._wpos += tmp /(self._sampwidth*self._info.channels)
        return tmp

    # write audio frames without correcting `nframes`
    # this is a noop; `nframes` is auto-corrected by libsndfile
    writeframesraw = writeframes
