"""PyCClass, a library for writting Python Classes that wrap C classes
"""

"""Copyright (C) 2006 ParIT Worker Co-operative <http://parit.ca>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
MA  02110-1301  USA
 
Authors: Mark Jenkins <mark@parit.ca>
"""

def ensure_is_tuple( value ):
    """Transforms a value into a tuple if it's not one allready

    If value is a tuple it is returned
    If value is a list it is converted to a tuple
    If value is anything else a tuple is returned consiting of just value
    """
    if type(value) != tuple:
        if type(value) != list:
            return (value,)
        else:
            return tuple(value)
    else:
        return value

class C_Class_Wrapper(object):
    """A python class that provides a wraper to a class in C.
    
    The way to implement classes in C is to have a struct containing the data,
    and a serries of method for acting on that data with a pointe to the
    struct as pointer struct as the first argument.

    Instances of this class all store that struct pointer in
    self._wrapped_c_value, and they are responsible for providing it
    via the c_value argument in the constructor.

    The class provides methods for common cases when dealing with classes in
    C, namely through the function, constructor, method, accessor, and mutator

    (TODO, show simple examples of using these keyword arguments here..)

    Class variables:
    arg_transforms is a tuple containing argument and function name
    transformation functions that can be applyed to the c function name and
    argument list everytime the function, constructor, method, accessor, and
    mutator methods are called.

    This is a tuple of tuples, the order of the outer tuple specifies the
    order these transformations are applyed.

    Each inner tuple specifies two values. The first is a string. The is the
    name of the keyword argument you should use when you call function,
    constructor, method, accessor, or mutator to specifiy that you would like
    a particular transformation to be applied.

    The second value in the inner tuple is the name of the actual
    transformation function. Note that name of the transformation needs
    to allready be declared, so they will allways be declared above
    the class variable arg_transforms.

    The interface for such functions is as follows:
    arg_transform_function(self, c_function, args, key, keyword_value,
    keywords)
    -> (c_function, args)

    self -- An instance of C_Class_Wrapper
    c_function -- The current name of the C function
    args -- The arguments being passed to the C function
    key -- The keyword from arg_transforms that was used to trigger this
           function
    keyword_value -- The value that the caller associated with the keyword
                     key in triggering this transformation function
    keywords      -- A dictionary of all the keywords

    This name/argument transformation functions are required to return a
    tuple consisting of the new name of the C function, and the new list/tuple
    of arguments
    

    Sub classes of C_Class_Wrapper that wish to extend arg_transforms should
    copy it and extend it. Example:
    class Blah(C_Class_Wrapper):
        def useless_transform(self, c_function, args, key, keyword_value,
                              keywords):
            return (c_function, args)
        
        arg_transforms = list(C_Class_Wrapper.arg_transforms)
        arg_transforms.extend( [
            ('do_nothing', useless_transform)
            ] )
        arg_transforms = tuple(arg_transforms)



    ret_transforms is a tuple containing return value transformation
    functions that can be applyed to the return value of c functions
    everytime the function, constructor, method, accessor, and mutator
    methods are called.

    It is a tuple of tuples. The outer tuple specifieds the order thise
    transformation functions are applied. The inner tuple consists of two
    values.

    The first value is name of a keyword argument that a caller to
    function, constructor, method, accessor, or mutator can specify to
    trigger the transformation.

    The second value is the name of the transformation function itself.

    The interface for such functions is as follows:
    return_transform_function(self, return_value, key, key_value, keywords)
    -> return_value

    self -- An instance of C_Class_Wrapper
    return_value -- The value being returned by the C function
    key -- The keyword argument that triggers this transformation function
    key_value -- The value that was given with the keyword argument key
    keywords -- A dictionary of all the keywords arguments

    The return value transformation functions are required to return the
    new return_value

    Sub classes of C_Class_Wrapper that wish to extend ret_transforms should
    copy it and extend it. Example:
    class Blah(C_Class_Wrapper):
        def useless_ret_transform(self, return_value, key, keyword_value,
                              keywords):
            return return_value
        
        ret_transforms = list(C_Class_Wrapper.ret_transforms)
        ret_transforms.extend( [
            ('do_nothing', useless_ret_transform)
            ] )
        ret_transforms = tuple(ret_transforms)
    
    """
    
    def __init__(self, c_value ):
        """Constructor. Where the wrapped value is provided.

        c_value -- The pointer to a C struct wrapped by this class
        
        """
        self._wrapped_c_value = c_value


    def method_prefix_transform( self, c_function, args, key, prefix_value,
                                 keywords ):
        """A transformation function that takes the name of the c_function and
        adds a prefix to it
        
        """
        return (prefix_value + c_function, args)

    def w_ob_args_transform( self, c_function, args, key, w_ob_args,
                             keywords ):
        """A transformation function that takes a list of arguments
        (w_ob_args), and replaces the co-responding elements in args with
        with the underlying _wrapped_c_value in each argument.

        args -- The arguments to the C function being transformed

        w_ob_args -- either a single integer specifying a specific argument,
        (0 indexed), or a tuple of integers specifying multiple arguments
        indicies to apply use this transformation to

        keywords -- The dict of keywords that was specifiyed by the
        whoever made the function call, that specified which transformation
        functions to apply, and with what arguments

        returns the transformed C function name and the transformed argument
        list
        """
        w_ob_args = ensure_is_tuple(w_ob_args)
        args = list(args) # make a list so we can mutate arguments

        # transform all the arguments specified as having an
        # underlying C value
        for w_ob_arg in w_ob_args:
            args[w_ob_arg] = args[w_ob_arg]._wrapped_c_value
        return (c_function, args) # return the transformed argument list

    def wrap_return_value_transform( self, return_value, key, return_class,
                                     keywords ):
        """A return value transformation function that wraps the
        return value with a new instance of some sub class of
        C_Class_Wrapper

        return_value -- The return value that will be sent to the new instance
        key -- The keyword that tiggered this function
        return_class -- The class that was passed through the keyword argument
        for use in this transformation.
        keywords -- The full dictionary of keywords for triggering
                    transformation functions

        Returns a new instance of return_class with return_value passed as
        c_value in the constructor
        """
        return return_class( c_value=return_value )
    

    arg_transforms = (
        ('accessor_prefix', method_prefix_transform ),
        ('mutator_prefix', method_prefix_transform ),
        ('method_prefix', method_prefix_transform),
        ('w_ob_args', w_ob_args_transform),
        )
    
    ret_transforms = (
        ('wrap_return_value', wrap_return_value_transform ),
    )

    def apply_argument_transforms( self, c_function, args, keywords ):
        """This takes a C function name and a set of arguments, and it applies
        all of the transformation functions in self.arg_transforms to that
        function name and argument list.

        c_function -- C function name
        args -- C function arguments
        keywords -- dict of keywords, where the key specifies which
        transformation from arg_transforms to apply, and the value specifies a
        value to pass to the matching transformation function.
        
        returns the new C function name and the new argument list as a tuple
        """
        for (key, transformer) in self.arg_transforms:
            if keywords.has_key( key ):
                (c_function, args) = transformer( self, 
                    c_function, args, key, keywords[key], keywords )
        return (c_function, args)
    
    def apply_return_value_transforms( self, return_value, keywords ):
        """This takes the value returned by a C function and
        applies all of the transformation functions in self.ret_transforms
        to it.

        return_value -- The value returned by the C function
        keywords -- The dict of keywords, where the key specifies a
        transformation from ret_transforms to apply, and the value is
        a value to be passed to the matching transformation function.

        returns the newly transformed C function return value
        """
        for (key, transformer) in self.ret_transforms:
            if keywords.has_key( key ):
                return_value = transformer( self, return_value, key,
                                            keywords[key], keywords )
        return return_value

    def function( self, c_function, *args, **keywords ):
        """Calls a C function from self.module with a set of arguments.

        c_function -- Name of the C function in module
        args -- Arguments to pass to the C function
        keywords -- Remaining keyword arguments are used to trigger
        transformations on the function name and arguments, and transformations
        on the return values. The keywords should match a value in
        arg_transforms and/or ret_transforms, and the values will be passed
        to the transformation function(s) associated with that keyword.

        Returns the value returned from the C function with transformations
        applied.
        """
        (c_function, args) = self.apply_argument_transforms(
            c_function, args, keywords )
        
        func = getattr(self.module, c_function)
        return_value = func(*args )
        return_value_old = return_value
        return_value = self.apply_return_value_transforms( return_value,
                                                   keywords )
        return return_value

    def add_keyword_if_not_there_yet( key, value, keywords):
        """Checks if key in in keywords, if not the key value pair is
        added to keywords
        """
        if not keywords.has_key(key):
            keywords[ key ] = value
    add_keyword_if_not_there_yet = staticmethod( add_keyword_if_not_there_yet)

    def add_method_prefix( self, keywords ):
        """Adds the key value pair ('method_prefix', self.method_prefix) to
        keywords if there is no 'method_prefix' key in keywords yet
        """
        self.add_keyword_if_not_there_yet( 'method_prefix',
                                           self.method_prefix,
                                           keywords )

    def extend_arg_list(keywords, key, new_args):
        """The value for key in keywords is intended to be a tuple of
        argument indicies. This function extends that tuple to include
        more indicies.

        keywords -- The dict of keywords for specifying transformations
        key -- The key in keywords to be extended
        new_args -- The additional argument indicies to be appended to
        keywords[key]

        If there is no key in keywords, new_args simply becomes the
        value in keywords[key]

        If the current value in keywords[key] is not a list or tuple,
        it is simply transformed into one.
        
        """
        
        new_args = list(ensure_is_tuple(new_args))
        if keywords.has_key(key):
            keywords[key] = ensure_is_tuple(keywords[key])
            new_args.extend( keywords[key] )
        keywords[key] = tuple(new_args)
    extend_arg_list = staticmethod( extend_arg_list )

    def method( self, c_function, *args, **keywords ):
        """Calls a c function that represents a C class method.

        Recall that c functions used as a C class method allways has a
        pointer to a struct as the first argument. This method inserts that
        first argument for you! The value it inserts is self._wrapped_c_value.
        If you're applying any argument transformtions you need to keep
        this in mind! The argument inserted by this method is argument 0,
        so don't apply any transformations to it! The nice thing about
        this function, the implicit insertion of that required argument is
        also disadvantageous when you get so used to the convienence that you
        forget what it's doing for you.

        Not only does method() automatically insert that 0th argument for
        you, but it also allows you to specify the only method specific part
        of the function name. method() will call the c function from
        self.module with the name
        self.method_prefix + c_function
        This is the optimal behavior for most well written C libraries.
        For example, GnuCash has an Account class and a series of methods
        for manipulating instances that all of the same prefix, xaccAccount,
        such as xaccAccountBeginEdit, xaccAccountCommitEdit, xaccAccountOrder()
        If you subclass C_Class_Wrapper and define a method_prefix class
        variable equal to 'xaccAccount', you can wrapped begin_edit with
        def begin_edit(self):
            self.method('BeginEdit')
        
        
        If you need to override this behavior, you can change the
        method_prefix to blank or something else by passing utilizing the
        keyword argument method_prefix.

        c_function -- The part of the c function name that comes after
                      self.method_prefix
        args -- arguments to pass to the C class method
        keywords -- a dict of keywords that specify transformations

        Returns the value returned from the c function, with transformations
        applied
        """
        self.add_method_prefix( keywords )
        self.extend_arg_list( keywords, 'w_ob_args', 0)
        return self.function( c_function, self,
                              *args, **keywords )
    
    def constructor( self, *args, **keywords ):
        """Calls a C function that known to be the constructor

        args -- Arguments to pass to the constructor
        keywords -- Keywords specifying function name, argument, and return
        value transformations.

        It is expected that sub classes have defined the class variables
        method_prefix and _malloc. This function will attemp to call
        a c function in self.module with the name
        self.method_prefix + self._malloc
        and it will pass *args to it.

        If this use of self.method_prefix is undesired, and you just want a
        function self._malloc to be called, set method_prefix='' when you
        call this. (this sets the method prefix to blank). You could also
        set method_prefix to some other value, and the function name
        will end up being
        method_prefix + self._malloc
        
        If that behavior is not desired, override this method and
        implement it yourself. Your overrided implementation should be super
        easy, just use self.function to get the job done
        """
        self.add_method_prefix( keywords )
        return self.function( self._malloc, *args, **keywords )

    def accessor( self, attribute, *args, **keywords ):
        """Calls C functions that are C Class accessors

        This is similar to method(), you should read and understand
        it before reading on here.

        So, by now you undertstand that method helps elimanate the redunancy in
        calling C class methods that allways have the struct pointer as the
        first pointer, and it can also help eliminate the redundancy of
        calling C class methods that allways have the same prefix.
        There is more redundancy to eliminate! Good C libraries with C classes
        will have consistant naming for accessors. Consider for example
        the Account class in GnuCash, there is xaccAccountSetName, and
        xaccAccountSetDescription, and xaccAccountSetCode and ...

        With accessor you just need to specify the name of the attribute
        (Name, Code, Description...).

        accessor will call the c function self.module with the name
        self.method_prefix + self.accessor_prefix + attribute
        so with the GnuCash example you can do stuff like
        class Account(C_Class_Wrapper):
            method_prefix = 'xaccAccount'
            accessor_prefix = 'Set'
            def get_name(self):
                return self.accessor('Name')


        Like method, you can muck with the transformations, including the
        transformations that prepend method_prefix and accessor_prefix
        by setting keyword arguments.

        attribute -- The name of the attribute access
        args -- Arguments to pass to the accessor function, remember, you
                don't need to put in argument 0, the pointer to the C struct
        
        """
        self.add_keyword_if_not_there_yet( 'accessor_prefix',
                                           self.accessor_prefix,
                                           keywords) 
        return self.method( attribute, *args, **keywords )

    def mutator( self, attribute, *args, **keywords ):
        """Calls C functions that are C Class mutators

        This really similar to the accessor method,
        please read the documentation for method() and accessor()

        The only difference is that this one prepends self.mutator_prefix, or
        keyword mutator_prefix to the atrribute
        """
        self.add_keyword_if_not_there_yet( 'mutator_prefix',
                                           self.mutator_prefix,
                                           keywords )
        self.method( attribute, *args, **keywords )


class SelfMallocs(C_Class_Wrapper):
    """Subclasses of this class are meant to be wrappers for C classes that
    have a memory allocation function which can be called to provide
    the _wrapped_c_value for a new instance.
    To facilitate this they define the method_prefix and _malloc class
    variables.

    If you instantiate one of these, self.constructor() is automatically
    called, and the resulting struct pointer passed up to the superclass
    (C_Class_Wrapper)
    
    """

    
    def __init__(self, *malloc_args):
        """Constructor. Optionally arguments can be passed to the
        c function that constructs this.
        
        """
        C_Class_Wrapper.__init__( self,
                                  self.constructor(*malloc_args) )


class SelfMallocsUnlessValueProvided(SelfMallocs):
    """Subclasses of this class are meant to be wrappers for C classes
    where the _wrapped_c_value is provided by either allocation function,
    or mannually provided. It allows a choice between the the model used
    by C_Class_Wrapper and SelfMallocs depending on the situation.
    """
    def __init__(self, c_value=None):
        if c_value == None:
            SelfMallocs.__init__(self)
        else:
            C_Class_Wrapper.__init__( self, c_value )
