#!/usr/bin/env python3
##
##  SPDX-FileCopyrightText: © 2007-2021 Benedict Verhegghe <bverheg@gmail.com>
##  SPDX-License-Identifier: GPL-3.0-or-later
##
##  This file is part of pyFormex 2.6  (Mon Aug 23 15:13:50 CEST 2021)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: https://pyformex.org
##  Project page: https://savannah.nongnu.org/projects/pyformex/
##  Development: https://gitlab.com/bverheg/pyformex
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##
#
"""manifest.py

This script creates the list of files to be included in
the pyFormex source distribution.
"""
import os,re

def prefixFiles(prefix,files):
    """Prepend a prefix to a list of filenames."""
    return [ os.path.join(prefix,f) for f in files ]

def matchMany(regexps,target):
    """Return multiple regular expression matches of the same target string."""
    return [re.match(r,target) for r in regexps]


def matchCount(regexps,target):
    """Return the number of matches of target to regexps."""
    return len([ m for m in matchMany(regexps,target) if m ])


def matchAny(regexps,target):
    """Check whether target matches any of the regular expressions."""
    return matchCount(regexps,target) > 0


def matchNone(regexps,target):
    """Check whether targes matches none of the regular expressions."""
    return matchCount(regexps,target) == 0


def listTree(path,listdirs=False,topdown=True,sorted=True,excludedir=[],excludefile=[],excludepath=[],includedir=[],includefile=[],includepath=[],filtr=None):
    """List all files in path.

    If ``dirs==False``, directories are not listed.
    By default the tree is listed top down and entries in the same directory
    are unsorted.

    `exludedir`, `excludefile`, `excludepath` are lists of regular expressions
    with dirnames, resp. filenames, resp. path names to exclude from the result.

    `includedir`, `includefile`, `includepath` can be given to include only the
    directories, resp. files, resp. paths  matching any of those patterns.

    The filters are applied in the following order: excludedir, includedir,
    excludefile, includefile, excludepath, includepath.

    Note that 'excludedirs' and 'includedirs' force top down handling.
    """
    filelist = []
    if excludedir or includedir:
        topdown = True
    for root, dirs, files in os.walk(path, topdown=topdown):
        if sorted:
            dirs.sort()
            files.sort()
        if excludedir:
            remove = [ d for d in dirs if matchAny(excludedir,d) ]
            for d in remove:
                dirs.remove(d)
        if includedir:
            remove = [ d for d in dirs if not matchAny(includedir,d) ]
            for d in remove:
                dirs.remove(d)
        if listdirs and topdown:
            filelist.append(root)
        if excludefile:
            files = [ f for f in files if not matchAny(excludefile,f) ]
        if includefile:
            files = [ f for f in files if matchAny(includefile,f) ]
        files = prefixFiles(root,files)
        if excludepath:
            files = [ f for f in files if not matchAny(excludepath,f) ]
        if includepath:
            files = [ f for f in files if matchAny(includepath,f) ]
        filelist.extend(files)
        if listdirs and not topdown:
            filelist.append(root)
    if filtr:
        filelist = [ f for f in filelist if filtr(f) ]
    return filelist


# pyFormex documentation (installed in the pyformex tree)
SPHINX_DIR = 'pyformex/sphinx'
DOC_FILES = []
# DOC_FILES += listTree(
#     SPHINX_DIR, excludedir=['_build', '__pycache__'],
#     excludepath=[f'{SPHINX_DIR}/ref/.*\.rst',
#                  f'{SPHINX_DIR}/pyformex\.help',
#                  f'{SPHINX_DIR}/static/.*\.png'
#     ])
DOC_FILES += listTree('pyformex/doc/html',)
DOC_FILES += listTree(
    'pyformex/doc',
    excludedir=['dutch','doctree'],
    includefile=[
        'README.rst',
        'Description',
        'LICENSE',
        'ReleaseNotes',
        ],
    )


# pyFormex data files (installed in the pyformex tree)
DATA_FILES = listTree(
    'pyformex/data',
    excludedir=['benchmark','ply'],
    excludefile=['.*\.pyc','.*~$','PTAPE.*'],
    includefile=[
        'README',
        'benedict_6\.jpg',
        'bifurcation\.off.gz',
        'blippo\.pgf',
        'blippok\.ttf',
        'butterfly\.png',
        'fewgl\.js',
        'hesperia-nieve\.prop',
        'horse\.off',
        'horse\.pgf',
        'image_not_loaded\.png',
        'man\.off',
        'man\.pgf',
        'mark_cross\.png',
        'materials\.json',
        'sections\.json',
        'splines\.pgf',
        'supershape\.txt',
        'teapot\.off',
        'template.py',
        'world\.jpg',
        'xtk_xdat\.gui\.js',
        ],
) + listTree(
    'pyformex/scripts',
) + listTree(
    'pyformex/glsl',
    includefile=[
        '.*\.c',
    ],
) + listTree(
    'pyformex/fonts',
    includefile=[
        '.*\.png',
    ],
)

# scripts to install extra programs
EXTRA_FILES = listTree(
    'pyformex/extra',listdirs=True,
    includedir=[
        'calpy',
        'dxfparser',
        'gts',
        'instant',
        'postabq',
        'pygl2ps',
        ],
    excludefile=['.*~$'],
    includefile=[
        'README',
        'Makefile',
        '.*\.sh$',
        '.*\.rst$',
        '.*\.patch$',
        '.*\.c$',
        '.*\.cc$',
        '.*\.h$',
        '.*\.i$',
        '.*\.py$',
        '.*\.1$',
        '.*\.pc$',
        ],
    )

# Data files to be installed outside the pyformex tree
# These are tuples (installdir,filelist)
OTHER_DATA = [
    ('share/pixmaps', [
        'pyformex/icons/pyformex-64x64.png',
        'pyformex/icons/pyformex.xpm',
        ]),
    ('share/applications', ['pyformex.desktop']),
    ('share/man/man1', [
        'pyformex/doc/pyformex.1',
        ]),
    # the full html documentation
    ## ('share/doc/pyformex/html',DOC_FILES),
    ]

DIST_FILES = [
    'README.rst',
    'LICENSE',
    'Description',
    'Makefile',
    'apt_install_deps',
    'ReleaseNotes',
    'manifest.py',
    'setup.py',
    'setup.cfg',
    ] + \
    listTree('pyformex',
             includedir=['gui','menus','plugins','opengl',
                          'fe','appsdir'],   # DO NOT ADD scripts HERE
             includefile=['.*\.py$','pyformex(rc)?$','pyformex.conf$'],
             excludefile=['core.py','curvetools.py','backports.py'],
             ) + \
    listTree('pyformex/freetype',
             includedir=['ft_enums'],
             includefile=['.*\.py$'],
             ) + \
    listTree('pyformex/icons',
             includefile=['README','.*\.xpm$','.*\.png$','.*\.gif$']
             ) + \
    listTree('pyformex/lib',
             includefile=['.*\.c$','.*\.py$','.*\.pyx$']
             ) + \
    listTree('pyformex/bin',
             excludefile=['.*~$'],
             ) + \
    listTree('pyformex/examples',
             excludedir=['Demos'],
             excludefile=['.*\.pyc','.*~$',
                           ],
             includefile=['[_A-Z].*\.py$','apps.cat','README']
             ) + \
    listTree('pyformex/vtk_light',
             includedir=['util'],
             includefile=['README','.*\.py$'],
             ) + \
    DATA_FILES + \
    DOC_FILES + \
    EXTRA_FILES


for i in OTHER_DATA:
    DIST_FILES += i[1]


if __name__ == '__main__':
    import sys

    todo = sys.argv[1:]
    if not todo:
        todo = ['usage']

    for a in todo:
        if a == 'doc':
            print("=========DOC_FILES=========")
            print('\n'.join(DOC_FILES))
        elif a == 'data':
            print("=========DATA_FILES========")
            print('\n'.join(DATA_FILES))
        elif a == 'other':
            print("=========OTHER_DATA========")
            for i in OTHER_DATA:
                print('\n'.join(i[1]))
        elif a == 'dist':
            print("=========DIST_FILES========")
            print('\n'.join(DIST_FILES))
        elif a == 'manifest':
            with open('MANIFEST', 'wt') as fil:
                fil.write('\n'.join(DIST_FILES))
                fil.write('\n')
            print("Created new MANIFEST")
        else:
            print("""\
Usage: manifest.py CMD [...]
where CMD is one of:
doc data other dist manifest
""")



# End
