#
##
##  This file is part of pyFormex 2.3  (Mon Feb 22 15:38:03 CET 2021)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: http://pyformex.org
##  Project page:  http://savannah.nongnu.org/projects/pyformex/
##  Copyright 2004-2020 (C) Benedict Verhegghe (benedict.verhegghe@ugent.be)
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##

"""pyFormex core module initialization.

This module initializes the pyFormex global variables and
defines a few essential functions.
It is the very first thing that is loaded when starting pyFormex.
"""

import os
import sys
import time
import datetime
import logging
import logging.config

__prog__ = "pyformex"
__version__ = "2.3"
__revision__ = __version__
__branch__ = "master"
Copyright = 'Copyright (C) 2004-2021 Benedict Verhegghe'
Url = 'http://pyformex.org'
Description = ("pyFormex is a tool for generating, manipulating and "
               "transforming large geometrical models of 3D structures "
               "by sequences of mathematical transformations.")


def Version():
    """Return a string with the pyFormex name and version"""
    return f"pyFormex {__version__}"


def fullVersion():
    """Return a string with the pyFormex name, version and revision"""
    if __revision__ == __version__:
        return Version()
    else:
        return f"{Version()} ({__revision__}) DEVELOPMENT VERSION"


# set start date/time
StartTime = datetime.datetime.now()

startup_messages = ''
startup_warnings = ''
started = False

#########  Check Python version #############

# A single variable to flag Python2
PY2 = sys.hexversion < 0x03000000
PY3 = not PY2

if PY2:
    raise RuntimeError("This version of pyFormex does not support Python2.x. "
                       "Please use Python3 instead, or use an older pyFormex "
                       "version (<= 1.0.7)")


# intended Python version
# We only support 3.6+
minimal_version = 0x03060000
target_version = 0x03070000
future_version = 0x03080000


def major(v):
    """Return the major component of version"""
    return v >> 24


def minor(v):
    """Return the minor component of version"""
    return (v & 0x00FF0000) >> 16


def human_version(v):
    """Return the human readable string for version"""
    return f"{major(v)}.{minor(v)}"


if sys.hexversion < minimal_version:
    # Older than minimal
    startup_warnings += """
#######################################################################
##  Your Python version is %s, but pyFormex requires Python >= %s. ##
##  We advice you to upgrade your Python version.                    ##
#######################################################################
""" % (human_version(sys.hexversion), human_version(minimal_version))
    print(startup_warnings)
    sys.exit()

if sys.hexversion & 0xFFFF0000 > target_version:
    # Major,minor newer than target:
    startup_warnings += """
#######################################################################
##  Your Python version is %s, but pyFormex has only been tested    ##
##  with Python <= %s. We expect pyFormex to run correctly with     ##
##  your Python version, but if you encounter problems, please       ##
##  contact the developers at http://pyformex.org.                   ##
#######################################################################
""" % (human_version(sys.hexversion), human_version(minimal_version))

#### Detect install type ########

# Install type.
# This can have the following values:
#     'S' : obsolete (used to refer to Subversion sources)
#     'G' : unreleased version running from GIT sources: no installation
#           required. This is set if directory containing the
#           pyformex start script is a git repository.
#     'R' : normal (source) Release, i.e. tarball (default). Installation
#           is done with 'python setup.py install' in the unpacked source.
#           The builtin pyFormex removal is activated.
#     'D' : Distribution package of a released version, official or not
#           (e.g. Debian package). The distribution package tools should
#           be used to install/uninstall. The builtin pyFormex removal is
#           deactivated.
#

installtype = 'R'

# DO NOT USE Path here!!! We need to defer import of Path
# until the options have been parsed
pyformexdir = os.path.dirname(__file__)
parentdir = os.path.dirname(pyformexdir)
if os.path.exists(os.path.join(parentdir, '.git')):
    installtype = 'G'

########## start the logger ####################
LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'formatters': {
        'default': {
            'format': "%(asctime)s - %(name)s - %(levelname)s - %(message)s"
        },
        'short': {
            'format': "%(asctime)s %(message)s",
        },
        'detail': {
            'format': "%(asctime)s::%(levelname)s::%(name)s::"
            "%(filename)s::%(lineno)d::%(message)s",
        },
    },
    'handlers': {
        'console': {
            'class': 'logging.StreamHandler',
            'formatter': 'detail',
        },
        'memory': {
            'class': 'logging.handlers.MemoryHandler',
            'capacity': 100,
            'formatter': 'default',
        },
    },
    'loggers': {
        'pyformex': {
            'level': 'INFO',
            'handlers': ['memory'],
        }
    },
}

logging.config.dictConfig(LOGGING)
logger = logging.getLogger('pyformex')
logger.info(f"{fullVersion()} start on Python{human_version(sys.hexversion)}")

########## acceleration libraries ##############

libraries = ['misc_', 'nurbs_']

########## Special adjustements for source versions ############

if installtype in 'SG':
    # We are running from a source tree (Git)

    def run_cmd(cmd):
        """Run command"""
        import subprocess
        P = subprocess.Popen(
            str(cmd), stdout=subprocess.PIPE, shell=True, universal_newlines=True
        )
        out, err = P.communicate()
        return P.returncode, out, err

    # Clean the source tree
    source_clean = os.path.join(pyformexdir, 'source_clean')
    if os.path.exists(source_clean):
        try:
            import importlib
            suffix = importlib.machinery.EXTENSION_SUFFIXES[0]
            cmd = "%s --keep_modules '*%s'" % (source_clean, suffix)
            run_cmd(cmd)
        except Exception:
            print(f"Error while executing {source_clean}\n"
                  "Ignore error and continue")

    # Running from source tree: make sure the compiled libraries are up-to-date
    libdir = os.path.join(pyformexdir, 'lib')

    def checkLibraries():
        # find extension used by compiled library (.so)
        import sysconfig
        ext = sysconfig.get_config_var('EXT_SUFFIX')
        if ext is None:
            ext = '.so'
        msg = ''
        for lib in libraries:
            src = os.path.join(libdir, lib) + '.c'
            obj = os.path.join(libdir, lib) + ext
            if not os.path.exists(
                obj
            ) or os.stat(obj).st_mtime < os.stat(src).st_mtime:
                msg += "\nThe compiled library '%s' is not up to date!" % lib
        return msg

    msg = checkLibraries()
    if msg:
        print(msg)
        print("Rebuilding pyFormex libraries, please wait")
        target = 'lib3'
        cmd = "cd %s/..; make %s" % (pyformexdir, target)
        print(f"Running command: {cmd}")
        os.system(cmd)
        msg = checkLibraries()

    if msg:
        # No success
        msg += """

I had a problem rebuilding the libraries in %s/lib.
You should probably exit pyFormex, fix the problem first and then restart pyFormex.
""" % pyformexdir
    startup_warnings += msg

    # Set the proper revision number when running from git sources

    def set_revision():
        global __revision__
        cmd = 'cd %s && git describe --always' % pyformexdir
        sta, out, err = run_cmd(cmd)
        if sta == 0:
            __revision__ = out.split('\n')[0].strip()
        else:
            print("Could not set revision")

    def set_branch():
        global __branch__
        cmd = f"cd {pyformexdir} && git rev-parse --abbrev-ref HEAD"
        sta, out, err = run_cmd(cmd)
        if sta == 0:
            __branch__ = out.split('\n')[0]
        else:
            print("Could not set branch name")

    set_revision()
    set_branch()

logger.info(f"This is {fullVersion()}")
logger.info(f"Installation type {installtype} at {pyformexdir}")

################ pyFormex global variables and functions ###############

# The GUI parts
app_started = False
interactive = False
app = None  # the Qapplication
GUI = None  # the GUI QMainWindow
canvas = None  # the OpenGL Drawing widget controlled by the running script
console = None  # alternate Python console

# initialize some global variables used for communication between modules

# the options found on the command line
# this is replaced with the options read from the command line
# but we define it here with an attribute debuglevel, so that we can
# use debug (unintentionally) before the debuglevel is set


class options:
    debuglevel = 0  # Make sure we can use
    verbose = 1   # Make sure we can use
    uselib = False  # initial default to keep sphinx happy
    pathlib = False  # initial default to keep sphinx happy


# Are we importing from Sphinx doc generator?
sphinx = installtype == 'G' and 'sphinx' in sys.modules

PF = {}  # explicitely exported globals

scriptName = None
scriptlock = set()
scriptMode = None

# define default of warning and error
warning = print
error = print


def _busy(state=True):
    """Flag a busy state to the user"""
    if state:
        print("This may take some time...")


def busy(state=True):
    """Flag a busy state to the user"""
    if GUI:
        GUI.setBusy(state)
    else:
        _busy(state)


######################################################################
###  DEBUG levels
################################


class DebugLevels(object):

    """A class with debug levels.

    This class holds the defined debug levels as attributes.
    Each debug level is a binary value with a single bit set (a power of 2).
    Debug levels can be combined with & (AND), | (OR) and ^ (XOR).
    Two extra values are defined: None swtiches off all debugging, All
    activates all debug levels.

    """
    ALL = -1
    NONE = 0
    (INFO, WARNING, OPTION, CONFIG, DETECT, MEM, SCRIPT, GUI, MENU, DRAW,
     CANVAS, OPENGL, LIB, MOUSE, APPS, IMAGE, MISC, ABQ, WIDGET, PROJECT,
     WEBGL, MULTI, LEGACY, PLUGIN, UNICODE, FONT, PGF, VTK, DOCTEST,
     ) = [2 ** i for i in range(29)]


try:
    # Python2 stores the local variable 'i'
    delattr(DebugLevels, 'i')
except Exception:
    # Python3 seems not to store it
    pass

DEBUG = DebugLevels


def debugItems():
    """Return all existing debug items"""
    return [k.lower() for k in dir(DebugLevels) if not k.startswith('_')]


def debugLevel(sl):
    lev = 0
    for l in sl:
        try:
            lev |= getattr(DEBUG, l.upper())
        except Exception:
            pass
    return lev


def debugOn(level):
    """Test if some debug level is on"""
    return options.debuglevel & level


def debug(s, level=DEBUG.ALL):
    """Print a debug message"""
    try:  # to make sure that debug() can be used before options are set
        if options.debuglevel & level:
            raise exception
    except Exception:
        print(f"DEBUG({level}): {s}")


def debugt(s, level):
    """Print a debug message with timer"""
    debug(f"{time.time()}: {s}")


def verbosity(verbose):
    """Check that verbosity is equal or higher than given value"""
    return options.verbose >= verbose


def verbose(verbose, *args, **kargs):
    """Print a message if the verbosity is high enough"""
    if options.verbose >= verbose:
        print(*args, **kargs)


def run(args=[]):
    from pyformex import main
    main.run(args)


###################################################################
## Load selected Path class
###########################

if len(sys.argv) > 1 and sys.argv[1].endswith('pyformex'):
    from pyformex.options import parseOptions
    if not parseOptions(sys.argv[2:]):
        sys.exit(1)

if options.pathlib:
    from pathlib import Path
else:
    from pyformex.path import Path

# Placeholder for excutable, if pyformex: filled in by startup.py
executable = Path(sys.executable)

############################################################################
###  Load default configuration
### (pyFormex does not work if not loaded)
##########################################

from pyformex.config import Config  # noqa: E402

# Create a config instance
cfg = Config()
# Make pyformexdir a Path and set in the cfg
pyformexdir = Path(pyformexdir)
cfg['pyformexdir'] = pyformexdir
# load the factory defaults
defaults = cfg['pyformexdir'] / 'pyformexrc'
cfg.load(defaults)
# The default user config path (do not change!)
cfg['userprefs'] = cfg['userconfdir'] / 'pyformex.conf'

# Set the current session configurations in pyformex module
prefcfg = None  # the preferenced configuration
refcfg = None  # the reference configuration
preffile = None  # the file where the preferenced configuration will be saved

# End
