<?php
/* vim: set expandtab tabstop=4 shiftwidth=4 foldmethod=marker: */
 /* @(#) $Header: /sources/phpprintipp/phpprintipp/tiny/PrintIPP.php,v 1.5 2008/05/12 22:15:24 harding Exp $
 *
 * Version: 0.4 
 * Class PrintIPP (Tiny) - Send Basic IPP requests, Get IPP Responses.
 *
 *   Copyright (C) 2005-2006  Thomas HARDING
 *   Parts Copyright (C) 2005-2006 Manuel Lemos
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License for more details.
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *   mailto:thomas.harding@laposte.net
 *   Thomas Harding, 56 rue de la bourie rouge, 45 000 ORLEANS -- FRANCE
 *   
 */
    
/*

    This class is intended to implement Internet Printing Protocol on client side.

    References needed to debug / add functionnalities:
        - RFC 2910
        - RFC 2911
*/
/*
    TODO: beta tests on other servers than Cups
*/

/***********************
*
* class PrintIPP (Tiny)
*
************************/


class PrintIPP {

    // {{{ variables declaration

    
    public $jobs = array();
    public $jobs_uri = array();
    public $status = array();
    public $last_job = "";
    public $debug = array();
    public $printer_attributes; // object you can read: printer's attributes after getPrinterAttributes()
    public $job_attributes; // object you can read: last job attributes
    public $jobs_attributes; // object you can read: jobs attributes after getJobs()
    public $debug_level = 2; // max 3: very verbose
    public $available_printers = array();
    public $printers_uri = array();
    public $with_exceptions = 0;
    public $response;
    public $http_timeout = 0; // timeout at http connection (seconds) 0 => default => 30.
    public $http_data_timeout = 0; // data reading timeout (milliseconds) 0 => default => 30.
    public $ssl = false;

    // {{{ protected variables;
    
    protected $log_level = 2; // max 3: very verbose
    protected $log_type = 3; // 3: file | 1: e-mail | 0: logger
    protected $log_destination; // e-mail or file
    protected $serveroutput;
    protected $setup;
    protected $stringjob;
    protected $data;    
    protected $debug_count = 0;
    protected $username;
    protected $charset;
    protected $password;
    protected $requesring_user;
    protected $client_hostname = "localhost";
    protected $stream;    
    protected $host = "localhost";
    protected $port = "631";
    protected $tls = 0;
    protected $printer_uri;
    protected $timeout = "20"; //20 secs
    protected $errNo;
    protected $errStr;
    protected $datatype;
    protected $datahead;
    protected $datatail;
    protected $meta;
    protected $operation_id;
    protected $delay;
    protected $error_generation; //devel feature
    protected $debug_http;
    protected $no_disconnect;
    // }}}
    // }}}
    
    // {{{ constructor
    public function __construct() {
        $tz = getenv("date.timezone");
        if (!$tz)
            $tz = date_default_timezone_get();
        date_default_timezone_set($tz);

        $this->meta = new stdClass();
        $this->setup = new stdClass();
        $this->values = new stdClass();
        $this->serveroutput = new stdClass();
        $this->error_generation = new stdClass();
        $this->_parsing = new stdClass();
        self::_initTags ();
    }
    // }}}

/*****************
*
* PUBLIC FUNCTIONS
*
*******************/


// SETUP

    // {{{ setPort($port='631')
    public function setPort($port='631'){
        $this->port = $port;
	self::_putDebug("Port is ".$this->port."\n");
    }
    // }}}

    // {{{ setHost($host='localhost')
    public function setHost($host='localhost'){
        $this->host = $host;
	self::_putDebug("Host is ".$this->host."\n");
    }
    // }}}

    // {{{ setPrinterURI ($uri)
    public function setPrinterURI ($uri) {

        // CURRENTLY MANDATORY
        // NOTE: IF NOT SET, AUTOMATICALLY DONE TO DEFAULT PRINTER IN CupsIPP
        
        $length = strlen ($uri);
        $length = chr($length);
        
        while (strlen($length) < 2)
            $length = chr(0x00) . $length;
        
        $this->meta->printer_uri = chr(0x45) // uri type | value-tag
                         .  chr(0x00) . chr(0x0B) // name-length
                         .  "printer-uri" // printer-uri | name
                         . $length
                         . $uri;

        $this->printer_uri = $uri;
        
        self::_putDebug(sprintf(_("Printer URI: %s\n"),$uri));
        

        $this->setup->uri = 1;

    }
    // }}}

    // {{{ setData($data)
    public function setData($data){ 
        // 
        $this->data = $data;
        self::_putDebug("Data set\n");
    }
    // }}}
    
    // {{{ setRawText ()
    public function setRawText () {
        $this->setup->datatype = 'TEXT';
        $this->meta->mime_media_type = "";
        $this->setup->mime_media_type = 1;
        $this->datahead = chr(0x16);
                       
        if (is_readable($this->data)){
            //It's a filename.  Open and stream.
            $data = fopen($this->data, "rb");
            while(!feof($data))
                $output = fread($data, 8192);
        } else {
            $output = $this->data;
            }

        if (substr($output,-1,1) != chr(0x0c))
            if (!isset($this->setup->noFormFeed))
                $this->datatail = chr(0x0c);
        
        self::_putDebug(_("Forcing data to be interpreted as RAW TEXT\n"));
    }
    // }}}

    // {{{ unsetRawText ()
    public function unsetRawText () {
        
        $this->setup->datatype = 'BINARY';
        $this->datahead = '';
        $this->datatail = '';
        self::_putDebug(_("Unset forcing data to be interpreted as RAW TEXT\n"));

    }
    // }}}

    // {{{ setBinary()
    public function setBinary () {
        self::unsetRawText();
    }
    // }}}

    // {{{ setFormFeed ()
    public function setFormFeed () {
        $this->datatail = "\r\n".chr(0x0c);
        unset($this->setup->noFormFeed);
    }
    // }}}

    // {{{ unsetFormFeed ()
    public function unsetFormFeed () {
        $this->datatail = '';
        $this->setup->noFormFeed = 1;
    }
    // }}}

    // {{{ setCharset ($charset)
    public function setCharset ($charset='us-ascii') {

        $charset = strtolower($charset);
        $this->charset = $charset;

        $this->meta->charset = chr(0x47) // charset type | value-tag
                         . chr(0x00) . chr(0x12) // name-length
                         . "attributes-charset" // attributes-charset | name
                         . self::_giveMeStringLength($charset) // value-length
                         . $charset; // value
        
        self::_putDebug(sprintf(_("Charset: %s\n"),$charset));

        $this->setup->charset = 1;
    }
    // }}}
 
    // {{{ setLanguage($language)
    public function setLanguage($language='en_us') {

        $language = strtolower($language);
        
        $this->meta->language = chr(0x48)    // natural-language type | value-tag
                         .  chr(0x00) . chr(0x1B) //  name-length
                         .  "attributes-natural-language" //attributes-natural-language
                         .  self::_giveMeStringLength($language) // value-length
                         .  $language; // value
 
        self::_putDebug(sprintf(_("Language: %s\n"),$language));
        
        $this->setup->language = 1;
    }
    // }}}

    // {{{ setMimeMediaType ($mime_media_type='application/octet-stream')
    public function setMimeMediaType ($mime_media_type='application/octet-stream') {
        
        self::setBinary();
                                     
        $length = strlen ($mime_media_type);
        $length = chr($length);
        
        while (strlen($length) < 2)
            $length = chr(0x00) . $length;
        
        
        self::_putDebug( sprintf(_("mime type: %s\n"),$mime_media_type));
 
        $this->meta->mime_media_type = chr(0x49) // mimeMediaType tag
                                . self::_giveMeStringLength('document-format')
                                . 'document-format' // mimeMediaType
                                . self::_giveMeStringLength($mime_media_type)
                                . $mime_media_type; // value
                                    
        $this->setup->mime_media_type = 1;
    }
    // }}}

    // {{{ setCopies ($nbrcopies=1)
    public function setCopies ($nbrcopies=1) {
        
        $this->meta->copies = "";
        if ($nbrcopies == 1)
            return;
        
        if ($nbrcopies > 65535 || $nbrcopies < 0) {
            trigger_error(_("Copies must be between 0 and 65535"),E_USER_WARNING);
            return FALSE;
            }
        
        $copies = self::_integerBuild($nbrcopies);

        $this->meta->copies = chr(0x21) // integer type | value-tag
                    . chr(0x00) .chr(0x06) //             name-length
                    . "copies" // copies    |             name
                    . self::_giveMeStringLength($copies) // value-length 
                    . $copies;
        
        self::_putDebug( sprintf(_("Copies: %s\n"),$nbrcopies));
        $this->setup->copies = 1;
    }
    // }}}
    
    // {{{ setDocumentName ($document_name)
    public function setDocumentName ($document_name="") {
        
        $this->meta->document_name = "";
        if (!$document_name)
            return;
        
        $document_name = substr($document_name,0,1023);
        
        $length = strlen ($document_name);
        $length = chr($length);
        
        while (strlen($length) < 2)
            $length = chr(0x00) . $length;
        
        
        self::_putDebug( sprintf(_("document name: %s\n"),$document_name));
 
        $this->meta->document_name = chr(0x41) // textWithoutLanguage tag
                                . chr(0x00) . chr(0x0d) // name-length 
                                . "document-name" // mimeMediaType
                                . self::_giveMeStringLength($document_name)
                                . $document_name; // value
                                    
    }
    // }}}

    // {{{ setJobName ($jobname='(PHP)',$absolute=false)
    public function setJobName ($jobname='(PHP)',$absolute=false) {

        $postpend = date('His') . $this->_setJobId();
        
        if ($absolute)
            $postpend = '';
        
        if (isset($this->values->jobname) && $jobname == '(PHP)')
            $jobname = $this->values->jobname;
        
        $this->values->jobname = $jobname;
        
        $jobname .= $postpend;
        
        $value_length = 0x00;
        for ($i = 0 ; $i < strlen($jobname)  ; $i ++) {
            $value_length += 0x01;
            }
        $value_length = chr($value_length);

        while (strlen($value_length) < 2)
            $value_length = chr(0x00) . $value_length;
        
        $this->meta->jobname = chr(0x42)  // nameWithoutLanguage type || value-tag
                    . chr(0x00) . chr(0x08) //                           name-length
                    . "job-name" //        job-name   ||                 name
                    . $value_length    //                                value-length
                    . $jobname ; //                                      value
        
        self::_putDebug( sprintf(_("Job name: %s\n"),$jobname)); 
        $this->setup->jobname = 1;
    }
    // }}}

    // {{{ setUserName ($username='PHP-SERVER')
    public function setUserName ($username='PHP-SERVER') {
        
        $this->requesting_user = $username;

        if ($username == 'PHP-SERVER' && isset($this->meta->username))
            return TRUE;
        
        $value_length = 0x00;
        for ($i = 0 ; $i < strlen($username)  ; $i ++) {
            $value_length += 0x01;
            }
        $value_length = chr($value_length);

        while (strlen($value_length) < 2)
            $value_length = chr(0x00) . $value_length;
                       
        $this->meta->username = chr(0x42) // keyword type || value-tag
                            . chr(0x00). chr(0x14) // name-length
                            . "requesting-user-name"
                            . $value_length
                            . $username;
                            
        self::_putDebug( sprintf(_("Username: %s\n"),$username));
        $this->setup->username = 1;
    }
    // }}}

    // {{{ setAuthentication ($username,$password)
    public function setAuthentication ($username,$password) {
        $this->password = $password;
        $this->username = $username;
        
        self::_putDebug( _("Setting password\n"));
        $this->setup->password = 1;
    }
    // }}}

    // {{{ setAuthentification ($username,$password)
    public function setAuthentification ($username,$password) {
        self::setAuthentication ($username,$password);
    }
    // }}}

    // {{{ setSides ($sides=2)
    public function setSides ($sides=2) {
        
        switch ($sides) {
            case 1:
                $sides = "one-sided";
                break;
            case 2:
                $sides = "two-sided-long-edge";
                break;
            case "2CE":
                $sides = "two-sided-short-edge";
                break;
            default:
                $sides = $sides; // yeah, what ?
                break;
            }
        
                   
        $this->meta->sides = chr(0x44) // keyword type | value-tag
                        . chr(0x00). chr(0x05) //        name-length
                        . "sides" // sides |             name
                        . self::_giveMeStringLength($sides) //               value-length
                        . $sides ; // one-sided |          value
                        
        self::_putDebug( sprintf(_("Sides value seted to %s\n"), $sides));
    }
    // }}}
   
    // {{{ setFidelity ()
    public function setFidelity () {
    
    // whether the server can't replace any attributes (eg, 2 sided print is not possible,
    // so print one sided) and DO NOT THE JOB.
        $this->meta->fidelity = chr(0x22) // boolean type  |  value-tag
                    . chr(0x00). chr(0x16) //                  name-length
                    . "ipp-attribute-fidelity" // ipp-attribute-fidelity | name
                    . chr(0x00) . chr(0x01) //  value-length
                    . chr(0x01); //  true | value
        
        self::_putDebug( _("Fidelity attribute is set (paranoid mode)\n"));
 
    }
    // }}}
    
    // {{{ unsetFidelity ()
    public function unsetFidelity () {
    
    // whether the server can replace any attributes (eg, 2 sided print is not possible,
    // so print one sided) and DO THE JOB.
    
        $this->meta->fidelity = chr(0x22) //  boolean type | value-tag
                            . chr(0x00). chr(0x16) //        name-length
                    . "ipp-attribute-fidelity" // ipp-attribute-fidelity | name
                    . chr(0x00) . chr(0x01) //               value-length
                    . chr(0x00); // false |                   value
        
        self::_putDebug( _("Fidelity attribute is unset\n"));
    }
    // }}}
    
    // {{{ setMessage()
    public function setMessage ($message='') {
        
        $this->meta->message = '';
        if (!$message)
            return;
    
        $this->meta->message = chr(0x40) // attribute type = text
                            . chr(0x00) . chr(0x07)
                            . "message"
                            . self::_giveMeStringLength(substr($message,0,127))
                            . substr($message,0,127);
        
        self::_putDebug( sprintf(_("Setting message to \"%s\"\n"),$message));
    }
    // }}}

    // {{{ setPageRanges($page_ranges)
    public function setPageRanges($page_ranges) {
    
        // $pages_ranges = string:  "1:5 10:25 40:52 ..."
        // to unset, specify an empty string.
        
        $this->meta->page_range = '';
        
        if (!$page_ranges)
            return;

        $page_ranges = trim(str_replace("-",":",$page_ranges));

        $first = true;
        $page_ranges = split(' ',$page_ranges);
        foreach ($page_ranges as $page_range) {
            $value = self::_rangeOfIntegerBuild($page_range);
            
            if ($first)
                $this->meta->page_ranges .= $this->tags_types['rangeOfInteger']['tag']
                                         .  self::_giveMeStringLength('page-ranges')
                                         .  'page-ranges'
                                         .  self::_giveMeStringLength($value)
                                         .  $value;
            else
                $this->meta->page_ranges .= $this->tags_types['rangeOfInteger']['tag']
                                         .  self::_giveMeStringLength('')
                                         .  self::_giveMeStringLength($value)
                                         .  $value;
            $first = false;                        
            }
            
    }
    // }}}
    
    // {{{ setAttribute($attribute,$value)
    public function setAttribute($attribute,$value) {

        $printer_attributes_tags = array_keys($this->printer_tags);
        $job_attributes_tags = array_keys($this->job_tags);
        
        if (in_array($attribute,$printer_attributes_tags)) {

            $tag_type = $this->printer_tags[$attribute]['tag'];
            switch ($tag_type) {
                case 'integer':
                    $this->printer_tags[$attribute]['value'] = self::_integerBuild($value);
                    break;
                case 'keyword':
                case 'naturalLanguage':
                    $this->printer_tags[$attribute]['value'] = $value;
                    break;
                default:
                    trigger_error(sprintf(_('SetAttribute: Tag "%s": cannot set attribute'),$attribute),E_USER_NOTICE);
                    self::_putDebug(sprintf(_('SetAttribute: Tag "%s": cannot set attribute'),$attribute),2);
                    self::_errorLog(sprintf(_('SetAttribute: Tag "%s": cannot set attribute'),$attribute),2);
                    return FALSE;
                    break;
                }
            $this->printer_tags[$attribute]['systag'] = $this->tags_types[$tag_type]['tag'];
                
        } elseif (in_array($attribute,$job_attributes_tags)) {
        
             $tag_type = $this->job_tags[$attribute]['tag'];
             switch ($tag_type) {
                case 'integer':
                    $this->job_tags[$attribute]['value'] = self::_integerBuild($value);
                    break;
                case 'keyword':
                case 'naturalLanguage':
                    $this->job_tags[$attribute]['value'] = $value;
                break;
                case 'enum':
                    $value = self::_enumBuild($attribute,$value); 
                    $this->job_tags[$attribute]['value'] = $value;
                    break;
                case 'rangeOfInteger':
                    // $value have to be: INT1:INT2 , eg 100:1000
                    $this->job_tags[$attribute]['value'] = self::_rangeOfIntegerBuild($value);
                break;
                default:
                    trigger_error(sprintf(_('SetAttribute: Tag "%s": cannot set attribute'),$attribute),E_USER_NOTICE);
                    self::_putDebug(sprintf(_('SetAttribute: Tag "%s": cannot set attribute'),$attribute),2);
                    self::_errorLog(sprintf(_('SetAttribute: Tag "%s": cannot set attribute'),$attribute),2);
                    return FALSE;
                    break;
                }
            $this->job_tags[$attribute]['systag'] = $this->tags_types[$tag_type]['tag'];
            
        } else {
            trigger_error(sprintf(_('SetAttribute: Tag "%s" is not a printer or a job attribute'),$attribute),E_USER_NOTICE);
            self::_putDebug(sprintf(_('SetAttribute: Tag "%s" is not a printer or a job attribute'),$attribute),2);
            self::_errorLog(sprintf(_('SetAttribute: Tag "%s" is not a printer or a job attribute'),$attribute),2);
            return FALSE;
            }

    }
    // }}}
    
    // {{{ unsetAttribute($attribute)
    public function unsetAttribute($attribute) {

        $printer_attributes_tags = array_keys($this->printer_tags);
        $job_attributes_tags = array_keys($this->job_tags);
        
        if (in_array($attribute,$printer_attributes_tags)) 
            unset  ($this->printer_tags[$attribute]['value'], $this->printer_tags[$attribute]['systag']);
        elseif (in_array($attribute,$job_attributes_tags))
            unset ($this->job_tags[$attribute]['value'], $this->job_tags[$attribute]['systag']);
        else {
            trigger_error(sprintf(_('unsetAttribute: Tag "%s" is not a printer or a job attribute'),$attribute),E_USER_NOTICE);
            self::_putDebug(sprintf(_('unsetAttribute: Tag "%s" is not a printer or a job attribute'),$attribute),2);
            self::_errorLog(sprintf(_('unsetAttribute: Tag "%s" is not a printer or a job attribute'),$attribute),2);
            return FALSE;
            }
    }
    // }}}

// LOGGING / DEBUGGING

    // {{{ setLog ($log_destination,$destination_type='file',$level=2)
    public function setLog ($log_destination,$destination_type='file',$level=2) {
        
        if (is_string($log_destination) && !empty($log_destination))
            $this->log_destination = $log_destination;
        
        switch ($destination_type) {
            case 'file':
            case 3:
                $this->log_destination = $log_destination;
                $this->log_type = 3;
                break;
            case 'logger':
            case 0:
                $this->log_destination = '';
                $this->log_type = 0;
                break;
            case 'e-mail':
            case 1:
                $this->log_destination = $log_destination;
                $this->log_type = 1;
                break;
            }
            $this->log_level = $level;
    
    }
    // }}}

    // {{{ printDebug()
    public function printDebug() {
        echo "<pre>";
        for ($i = 0 ; $i < $this->debug_count ; $i++)
            echo $this->debug[$i];
        echo "</pre>";
        $this->debug = array();
        $this->debug_count = 0;
    }
    // }}}
    
    // {{{ getDebug()
    public function getDebug() {
        $debug = '';
        for ($i = 0 ; $i < $this->debug_count ; $i++)
            $debug .= $this->debug[$i];
        $this->debug = array();
        $this->debug_count = 0;
        return $debug;
    }
    // }}}

// OPERATIONS

    // {{{ printJob()
    public function printJob(){
        // this version of printJob do not parse server output for job's attributes
        
        self::_putDebug( sprintf("*************************\nDate: %s\n*************************\n\n",date('Y-m-d H:i:s')));

        if (!$this->_stringJob())
            return FALSE;
                       
        if (is_readable($this->data)){
            self::_putDebug( _("Printing a FILE\n")); 
                
            $this->output = $this->stringjob;
           
            if ($this->setup->datatype == "TEXT")
                $this->output .= chr(0x16);
            
             
            $post_values = array( "Content-Type" => "application/ipp",
                                  "Data" => $this->output,
                                  "File" => $this->data);
            
            if ($this->setup->datatype == "TEXT" && !isset($this->setup->noFormFeed))
                $post_values = array_merge($post_values,array("Filetype"=>"TEXT"));
            
        } else {                      
            self::_putDebug( _("Printing DATA\n")); 
            
            $this->output = $this->stringjob;
            $this->output .= $this->datahead;    
            $this->output .= $this->data;
            $this->output .= $this->datatail;
            
            $post_values = array( "Content-Type" => "application/ipp",
                                  "Data" => $this->output);
             
            
            }
            
        if (self::_sendHttp ($post_values,'/printers/'))
            self::_parseServerOutput();
        
        if (isset($this->serveroutput) && isset($this->serveroutput->status)) {
            
            $this->status = array_merge($this->status,array($this->serveroutput->status));

            if ($this->serveroutput->status == "successfull-ok")
                self::_errorLog(sprintf("printing job %s: ",$this->last_job) .$this->serveroutput->status,3);
            else 
                self::_errorLog(sprintf("printing job: ",$this->last_job) .$this->serveroutput->status,1);

            return $this->serveroutput->status; 
            
            }

        $this->status = array_merge($this->status,array("OPERATION FAILED"));
        $this->jobs = array_merge($this->jobs,array(""));
        $this->jobs_uri = array_merge($this->jobs_uri,array(""));
        self::_errorLog("printing job : OPERATION FAILED",1);
    
    return false;
    }
    // }}}

/******************
*
* PROTECTED FUNCTIONS
*
*******************/

// HTTP OUTPUT

    // {{{ _sendHttp ($post_values,$uri)
    protected function _sendHttp ($post_values,$uri) {
    /*
        This function Copyright (C) 2005-2006 Thomas Harding, Manuel Lemos  
    */
            
            $this->response_completed[] = "no";       
            unset($this->serverouptut);

            self::_putDebug( _("Processing HTTP request\n\n"));
            
            $this->serveroutput->headers = array();
            $this->serveroutput->body = "";
            
            $http = new http_class;
            $http->with_exceptions = $this->with_exceptions;
            
            
            if ($this->debug_http) {
                $http->debug = 1;
                $http->html_debug=0;
            } else {
                $http->debug=0;
                $http->html_debug=0;
            }
            
            $url="http://".$this->host;
            if ($this->ssl)
                $url="https://".$this->host;
            
            
            $http->port = $this->port;
            $http->timeout = $this->http_timeout;
            $http->data_timeout= $this->http_data_timeout;
            
            $http->force_multipart_form_post = false;
            $http->user = $this->username;
            $http->password = $this->password;
            
            $error=$http->GetRequestArguments($url,$arguments);
            $arguments["RequestMethod"]="POST";
            $arguments["Headers"] = array("Content-Type" => "application/ipp");
            $arguments["BodyStream"] = array( array("Data" => $post_values["Data"]) );

            if (isset($post_values["File"]))
                $arguments["BodyStream"][]=array("File"=>$post_values["File"]);
            if (isset($post_values["FileType"])
             && !strcmp($post_values["FileType"],"TEXT"))
                 $arguments["BodyStream"][]=array("Data"=>Chr(12));
            
            $arguments["RequestURI"] = $uri;
            
            $error=$http->Open($arguments);
            
            //echo "<pre>";print_r(memory_get_usage());echo "<pre>";
            // ~1.5M

            if($error=="") {
                $error=$http->SendRequest($arguments);
                
                if($error=="") {
                
                    self::_putDebug( "\nH T T P    R E Q U E S T :\n\n");
                    self::_putDebug("Request headers:\n");
                    
                    for(Reset($http->request_headers),$header=0;$header<count($http->request_headers);Next($http->request_headers),$header++) {
                    
                        $header_name=Key($http->request_headers);
                        if(GetType($http->request_headers[$header_name])=="array") {
                             for($header_value=0;$header_value<count($http->request_headers[$header_name]);$header_value++)
                                self::_putDebug( $header_name.": ".$http->request_headers[$header_name][$header_value]."\n");
                        } else
                              self::_putDebug( $header_name.": ".$http->request_headers[$header_name]."\n");
                         
                        }
                    self::_putDebug( "\n\nRequest body:\n");
                    self::_putDebug( htmlspecialchars($http->request_body)
                                 ."\n\n********************\n* END REQUEST BODY *\n********************\n\n");

                         
                    $i = 0;
                    $headers=array();
                    unset($this->serveroutput->headers);
                    $error=$http->ReadReplyHeaders($headers);
                    
                    self::_putDebug( "\nH T T P    R E S P O N S E :\n\n");
                    if($error=="") {
                    
                        "Response headers:\n\n";
                        for(Reset($headers),$header=0;$header<count($headers);Next($headers),$header++) {
                            $header_name=Key($headers);
                            if(GetType($headers[$header_name])=="array") {
                                for($header_value=0;$header_value<count($headers[$header_name]);$header_value++) {
                                    self::_putDebug( $header_name.": ".$headers[$header_name][$header_value]."\n");
                                    $this->serveroutput->headers[$i] = $header_name.": ".$headers[$header_name][$header_value];
                                    $i++;
                                    }
                            } else {
                                self::_putDebug( $header_name.": ".$headers[$header_name]."\r\n");
                                $this->serveroutput->headers[$i] = $header_name.": ".$headers[$header_name];
                                $i++;
                                }
                            }
                             
                        self::_putDebug( "\n\nResponse body:\n");
                        $this->serveroutput->body = "";
                        for(;;) {
                            $error=$http->ReadReplyBody($body,1024);
                            if($error!="" || strlen($body)==0)
                            break;
                            self::_putDebug(htmlentities($body));
                            $this->serveroutput->body .= $body;
                            }
                        }
                        self::_putDebug( "\n\n*********************\n* END RESPONSE BODY *\n*********************\n\n");
                    }
                }
            $http->Close();
            if(strlen($error)) {
                self::_putDebug($error,1);
                trigger_error($error,E_USER_WARNING);
                $this->serveroutput->status = $error;
                return FALSE;
                }
    return true;
    }
    // }}}

// INIT
    
    // {{{ _initTags ()
    protected function _initTags () {
        
        $this->tags_types = array ( "unsupported" => array ("tag" => chr(0x10),
                                                        "build" => ""),
                              "reserved"    => array ("tag" => chr(0x11),
                                                        "build" => ""),
                              "unknown"     => array ("tag" => chr(0x12),
                                                        "build" => ""),
                              "no-value"    => array ("tag" => chr(0x13),
                                                        "build" => "no_value"),
                              "integer"     => array ("tag" => chr(0x21),
                                                        "build" => "integer"),
                              "boolean"     => array ("tag" => chr(0x22),
                                                        "build" => "boolean"),
                              "enum"        => array ("tag" => chr(0x23),
                                                        "build" => "enum"),
                              "octetString" => array ("tag" => chr(0x30),
                                                        "build" => "octet_string"),
                              "datetime"    => array ("tag" => chr(0x31),
                                                        "build" => "datetime"),
                              "resolution"  => array ("tag" => chr(0x32),
                                                        "build" => "resolution"),
                              "rangeOfInteger" => array ("tag" => chr(0x33),
                                                        "build" => "range_of_integers"),
                              "textWithLanguage" => array ("tag" => chr(0x35),
                                                        "build" => "string"),
                              "nameWithLanguage" => array ("tag" => chr(0x36),
                                                        "build" => "string"),
                              "text string" => array ("tag" => chr(0x40),
                                                        "build" => "string"),
                              "textWithoutLanguage" => array ("tag" => chr(0x41),
                                                        "build" => "string"),
                              "nameWithoutLanguage" => array ("tag" => chr(0x42),
                                                        "buid" => "string"),
                              "keyword"     => array ("tag" => chr(0x44),
                                                        "build" => "string"),
                              "uri"         => array ("tag" => chr(0x45),
                                                        "build" => "string"),
                              "uriScheme"   => array ("tag" => chr(0x46),
                                                        "build" => "string"),
                              "charset"     => array ("tag" => chr(0x47),
                                                        "build" => "string"),
                              "naturalLanguage" => array ("tag" => chr(0x48),
                                                        "build" => "string"),
                              "mimeMediaType" => array ("tag" => chr(0x49),
                                                        "build" => "string"),
                              "extendedAttributes" => array ("tag" => chr(0x7F),
                                                        "build" => "extended"),
                              );
                                                        
 
        $this->printer_tags = array ("compression" => array("tag" => "keyword"),
                                     "document-natural-language" => array("tag" => "naturalLanguage"),
                                     "job-k-octets" => array("tag" => "integer"),
                                     "job-impressions" => array("tag" => "integer"),
                                     "job-media-sheets" => array("tag" => "integer"),
                                     ); 

        $this->job_tags = array (   "job-priority" => array("tag" => "integer"),
                                    "job-hold-until" => array("tag" => "keyword"),
                                    "job-sheets" => array("tag" => "keyword"),
                                    "multiple-document-handling" => array("tag" => "keyword"),
                                    //"copies" => array("tag" => "integer"), // has its own function
                                    "finishings" => array("tag" => "enum"),
                                    //"page-ranges" => array("tag" => "rangeOfInteger"), // has its own function
                                    //"sides" => array("tag" => "keyword"), // has its own function
                                    "number-up" => array("tag" => "integer"),
                                    "orientation-requested" => array("tag" => "enum"),
                                    "media" => array("tag" => "keyword"),
                                    "printer-resolution" => array("tag" => "resolution"),
                                    "print-quality" => array("tag" => "enum"),
                                    );
    }
    // }}}

// SETUP

    // {{{ _setOperationId ()
    protected function _setOperationId () {
            $prepend = '';
            $this->operation_id += 1;
            $this->meta->operation_id = chr($this->operation_id);
            for ($i = strlen($this->meta->operation_id); $i < 4 ; $i++)
                $prepend .= chr(0x00);
            $this->meta->operation_id = $prepend . $this->meta->operation_id;

            self::_putDebug( "operation id is: ".$this->operation_id."\n");
    }
    // }}}
    
    // {{{ _setJobId()
    protected function _setJobId() {

        $this->meta->jobid +=1;
        $prepend = '';

        while ((strlen($this->meta->jobid) + strlen($prepend)) < 4 )
    	     $prepend .= "0";
       
        return $prepend . $this->meta->jobid;
    }
    // }}}
    
    // {{{ _setJobUri ($job_uri) 
    protected function _setJobUri ($job_uri) {
       
        $this->meta->job_uri = chr(0x45) // type uri
                             . chr(0x00).chr(0x07) // name-length
                             . "job-uri"
                             //. chr(0x00).chr(strlen($job_uri))
                             . self::_giveMeStringLength($job_uri)
                             . $job_uri;
        
        self::_putDebug( "job-uri is: ".$job_uri."\n");
    }
    // }}}
    
// RESPONSE PARSING

    // {{{ _parseServerOutput ()
    protected function _parseServerOutput () {

        $this->serveroutput->response = array();
        
        if (!self::_parseHttpHeaders ()) return FALSE;
        //$this->serveroutput->body = preg_split ("//",$this->serveroutput->body);
        
        $this->_parsing->offset = 0;
        
        self::_parseIppVersion ();
        self::_parseStatusCode ();
        self::_parseRequestID  ();
        self::_parseResponse ();
        
        //devel
        self::_putDebug( sprintf("***********\nIPP STATUS: %s\n***********\n\n",$this->serveroutput->status));
        self::_putDebug( "***************************** END OF OPERATION *****************************\n\n\n");
       
   return true;
   }
   // }}}
   
    // {{{ _parseHttpHeaders
    protected function _parseHttpHeaders () {
        
        $response = "";
        
        switch ($this->serveroutput->headers[0]) {
            
            case "http/1.1 200 ok: ":
                $this->serveroutput->httpstatus = "HTTP/1.1 200 OK";
                $response = "OK";
                break;
            case "":
                $this->serveroutput->httpstatus = "HTTP/1.1 000 No Response From Server";
                $this->serveroutput->status = "HTTP-ERROR-000_NO_RESPONSE_FROM_SERVER";
                trigger_error("No Response From Server",E_USER_WARNING);
                self::_errorLog("No Response From Server",1);
                $this->disconnected = 1;
                return FALSE;
                break;
            default:
                $server_response = preg_replace("/: $/",'',$this->serveroutput->headers[0]);
                $strings = split(' ',$server_response,3);
                $errno = $strings[1];
                $string = strtoupper(str_replace(' ','_',$strings[2]));
                trigger_error(sprintf(_("server responds %s"),$server_response),E_USER_WARNING);
                self::_errorLog("server responds ".$server_response,1);
                $this->serveroutput->httpstatus = strtoupper($strings[0])." ".$errno." ".ucfirst($strings[2]);
                $this->serveroutput->status = "HTTP-ERROR-".$errno."-".$string;
                $this->disconnected = 1;
                return FALSE;
                break;
            }
        
        unset ($this->serveroutput->headers);
        
    return TRUE;
    }
    // }}}
    
    // {{{ _parseIppVersion ()
    protected function _parseIppVersion () {

        $ippversion = (ord($this->serveroutput->body[ $this->_parsing->offset]) *  256) 
                    +  ord($this->serveroutput->body[$this->_parsing->offset + 1]);
        
        switch($ippversion) {
            case 0x0101:
                $this->serveroutput->ipp_version = "1.1";
                break;
            default:
                $this->serveroutput->ipp_version = sprintf("%u.%u (Unknown)",ord($this->serveroutput->body[ $this->_parsing->offset]) *  256,ord($this->serveroutput->body[$this->_parsing->offset + 1]));
                break;
        }
        
   self::_putDebug( "I P P    R E S P O N S E :\n\n"); 
   self::_putDebug( "IPP version ".$this->serveroutput->body[ $this->_parsing->offset]
                                 .$this->serveroutput->body[$this->_parsing->offset + 1]
                                 .": "
                                 .$this->serveroutput->ipp_version."\n");
    
        $this->_parsing->offset += 2;
    return;
    }
    // }}}

    // {{{ _parseStatusCode ()
    protected function _parseStatusCode () {
        $status_code = (ord($this->serveroutput->body[$this->_parsing->offset]) * 256)
                     +  ord($this->serveroutput->body[$this->_parsing->offset + 1]);
        

        $this->serveroutput->status ="NOT PARSED";
        
        $this->_parsing->offset += 2;
        
        if (strlen($this->serveroutput->body) < $this->_parsing->offset) 
        return false;

        
        if ($status_code < 0x00FF) {
            $this->serveroutput->status = "successfull";
        } elseif ($status_code < 0x01FF) {
            $this->serveroutput->status = "informational";
        } elseif ($status_code < 0x02FF) {
            $this->serveroutput->status = "redirection";
        } elseif ($status_code < 0x04FF) {
            $this->serveroutput->status = "client-error";
        } elseif ($status_code < 0x05FF) {
            $this->serveroutput->status = "server-error";
            }  

        switch ($status_code) {
            case 0x0000:
                $this->serveroutput->status = "successfull-ok";
                break;
            case 0x0001:
                $this->serveroutput->status = "successful-ok-ignored-or-substituted-attributes";
                break;
            case 0x002:
                $this->serveroutput->status = "successful-ok-conflicting-attributes";
                break;
            case 0x0400:
                $this->serveroutput->status = "client-error-bad-request";
                break;
            case 0x0401:
                $this->serveroutput->status = "client-error-forbidden";
                break;
            case 0x0402:
                $this->serveroutput->status = "client-error-not-authenticated";
                break;
            case 0x0403:
                $this->serveroutput->status = "client-error-not-authorized";
                break;
            case 0x0404:
                $this->serveroutput->status = "client-error-not-possible";
                break;
            case 0x0405:
                $this->serveroutput->status = "client-error-timeout";
                break;
            case 0x0406:
                $this->serveroutput->status = "client-error-not-found";
                break;       
            case 0x0407:
                $this->serveroutput->status = "client-error-gone";
                break;      
            case 0x0408:
                $this->serveroutput->status = "client-error-request-entity-too-large";
                break;      
            case 0x0409:
                $this->serveroutput->status = "client-error-request-value-too-long";
                break;      
            case 0x040A:
                $this->serveroutput->status = "client-error-document-format-not-supported";
                break;      
            case 0x040B:
                $this->serveroutput->status = "client-error-attributes-or-values-not-supported";
                break;      
            case 0x040C:
                $this->serveroutput->status = "client-error-uri-scheme-not-supported";
                break;      
            case 0x040D:
                $this->serveroutput->status = "client-error-charset-not-supported";
                break;      
            case 0x040E:
                $this->serveroutput->status = "client-error-conflicting-attributes";
                break;      
            case 0x040F:
                $this->serveroutput->status = "client-error-compression-not-supported";
                break;      
            case 0x0410:
                $this->serveroutput->status = "client-error-compression-error";
                break;      
            case 0x0411:
                $this->serveroutput->status = "client-error-document-format-error";
                break;      
            case 0x0412:
                $this->serveroutput->status = "client-error-document-access-error";
                break;      
            case 0x0500:
                $this->serveroutput->status = "server-error-internal-error";
                break;      
            case 0x0501:
                $this->serveroutput->status = "server-error-operation-not-supported";
                break;      
            case 0x0502:
                $this->serveroutput->status = "server-error-service-unavailable";
                break;      
            case 0x0503:
                $this->serveroutput->status = "server-error-version-not-supported";
                break;      
            case 0x0504:
                $this->serveroutput->status = "server-error-device-error";
                break;      
            case 0x0505:
                $this->serveroutput->status = "server-error-temporary-error";
                break;      
            case 0x0506:
                $this->serveroutput->status = "server-error-not-accepting-jobs";
                break;      
            case 0x0507:
                $this->serveroutput->status = "server-error-busy";
                break;      
            case 0x0508:
                $this->serveroutput->status = "server-error-job-canceled";
                break;      
            case 0x0509:
                $this->serveroutput->status = "server-error-multiple-document-jobs-not-supported";
                break;      
            default:
                break;
        }
        
    
    self::_putDebug( "status-code ".$this->serveroutput->body[$this->_parsing->offset]
                                  .$this->serveroutput->body[$this->_parsing->offset + 1]
                                  .": "
                                  .$this->serveroutput->status."\n");
        
    return;
    }
    // }}}
    
    // {{{ _parseRequestID ()
    protected function _parseRequestID () {
        
        $this->serveroutput->request_id = 
                      ord($this->serveroutput->body[$this->_parsing->offset]) * 4096
                    + ord($this->serveroutput->body[$this->_parsing->offset + 1]) * 256
                    + ord($this->serveroutput->body[$this->_parsing->offset + 2]) * 16
                    + ord($this->serveroutput->body[$this->_parsing->offset + 3]);

     self::_putDebug( "request-id ".$this->serveroutput->body[$this->_parsing->offset]
                             .$this->serveroutput->body[$this->_parsing->offset + 1]
                             .$this->serveroutput->body[$this->_parsing->offset + 2]
                             .$this->serveroutput->body[$this->_parsing->offset + 3]
                             .": "
                             .$this->serveroutput->request_id."\n");
                             
        $this->_parsing->offset += 4;
    return;    
    }
    // }}}
    
    // {{{ _parseResponse () {}
    protected function _parseResponse () {}
    // }}}
    
// REQUEST BUILDING
    
    // {{{ _stringJob ()
    protected function _stringJob () {
    
        if (!isset($this->setup->charset))
            self::setCharset('us-ascii');
        if (!isset($this->setup->datatype))
            self::setBinary();

        if (!isset($this->setup->uri)) {
            $this->getPrinters();
            unset($this->jobs[count($this->jobs) - 1]);
            unset($this->jobs_uri[count($this->jobs_uri) - 1]);
            unset($this->status[count($this->status) - 1]);
            
            if (array_key_exists(0,$this->available_printers))
               self::setPrinterURI($this->available_printers[0]);
            else {
                trigger_error(_("_stringJob: Printer URI is not set: die"),E_USER_WARNING);
                self::_putDebug( _("_stringJob: Printer URI is not set: die\n"));
                self::_errorLog(" Printer URI is not set, die",2);
                return FALSE;
                }
            }
            
        if (!isset($this->setup->copies))
            self::setCopies(1);
        
        if (!isset($this->setup->language))
            self::setLanguage('en_us');

        if (!isset($this->setup->mime_media_type))
            self::setMimeMediaType();
        if ($this->setup->datatype != "TEXT")
        unset ($this->setup->mime_media_type);
            
        if (!isset($this->setup->jobname))
            if (is_readable($this->data))
                self::setJobName(basename($this->data),true);
            else
                self::setJobName();
        unset($this->setup->jobname);

        if (!isset($this->meta->username))
            self::setUserName();

        if (!isset($this->meta->fidelity))
            $this->meta->fidelity = '';
        
        if (!isset($this->meta->document_name))
            $this->meta->document_name = '';

        if (!isset($this->meta->sides))
            $this->meta->sides = '';
        
        if (!isset($this->meta->page_ranges))
            $this->meta->page_ranges = '';
       
        $jobattributes = '';
        foreach ($this->job_tags as $key => $value) {
            if (array_key_exists('value',$value))
                $jobattributes .= $value['systag']
                                . self::_giveMeStringLength($key)
                                . $key
                                . self::_giveMeStringLength($value['value'])
                                . $value['value'];
        }
        reset ($this->job_tags);
        
        $printerattributes = '';
        foreach ($this->printer_tags as $key => $value) {
            if (array_key_exists('value',$value))
                $printerattributes .= $value['systag']
                                . self::_giveMeStringLength($key)
                                . $key
                                . self::_giveMeStringLength($value['value'])
                                . $value['value'];
        }
        reset ($this->printer_tags);
        
        self::_setOperationId();

        if (!isset($this->error_generation->request_body_malformed))
            $this->error_generation->request_body_malformed = "";
       
        $this->stringjob = chr(0x01) . chr(0x01) // 1.1  | version-number
                         . chr(0x00) . chr (0x02) // Print-Job | operation-id
                         . $this->meta->operation_id //           request-id
                         . $this->error_generation->request_body_malformed
                         . chr(0x01) // start operation-attributes | operation-attributes-tag
                         . $this->meta->charset
                         . $this->meta->language
                         . $this->meta->printer_uri
                         . $this->meta->username
                         . $this->meta->jobname
                         . $this->meta->fidelity
                         . $this->meta->document_name
                         . $this->meta->mime_media_type
                         . $printerattributes
                         . chr(0x02) // start job-attributes | job-attributes-tag
                         . $this->meta->copies
                         . $this->meta->sides
                         . $this->meta->page_ranges
                         . $jobattributes
                         . chr(0x03); // end-of-attributes | end-of-attributes-tag
        

        self::_putDebug( sprintf(_("String sent to the server is:\n%s\n"), $this->stringjob));
        return TRUE;
    }
    // }}}

    // {{{ _giveMeStringLength ($string)   
    protected function _giveMeStringLength ($string) {
        
        $prepend = '';
        $length = strlen($string);
        $lengthlength = strlen(chr($length));
        while ($lengthlength < 2) {
            $prepend .= chr(0x00);
            $lengthlength += 1;
            }
        $length = $prepend . chr($length);
 
    return $length;
    }
    // }}}

    // {{{ _enumBuild ($tag,$value)
    protected function _enumBuild ($tag,$value) {
        
        switch ($tag) {
            case "orientation-requested":
                switch ($value) {
                    case 'portrait':
                        $value = chr(3);
                        break;
                    case 'landscape':
                        $value = chr(4);
                        break;
                    case 'reverse-landscape':
                        $value = chr(5);
                        break;
                    case 'reverse-portrait':
                        $value = chr(6);
                        break;
                }
            break;
            case "print-quality":
                switch($value) {
                    case 'draft':
                        $value = chr(3);
                        break;
                    case 'normal':
                        $value = chr(4);
                        break;
                    case 'high':
                        $value = chr(5);
                        break;
                }
            break;
            case "finishing":
                switch ($value) {
                        case 'none':
                            $value = chr(3);
                            break;
                        case 'staple':
                            $value = chr(4);
                            break;
                        case 'punch':
                            $value = chr(5);
                            break;
                        case 'cover':
                            $value = chr(6);
                            break;
                        case 'bind':
                            $value = chr(7);
                            break;
                        case 'saddle-stitch':
                            $value = chr(8);
                            break;
                        case 'edge-stitch':
                            $value = chr(9);
                            break;
                        case 'staple-top-left':
                            $value = chr(20);
                            break;
                        case 'staple-bottom-left':
                            $value = chr(21);
                            break;
                        case 'staple-top-right':
                            $value = chr(22);
                            break;
                        case 'staple-bottom-right':
                            $value = chr(23);
                            break;
                        case 'edge-stitch-left':
                            $value = chr(24);
                            break;
                        case 'edge-stitch-top':
                            $value = chr(25);
                            break;
                        case 'edge-stitch-right':
                            $value = chr(26);
                            break;
                        case 'edge-stitch-bottom':
                            $value = chr(27);
                            break;
                        case 'staple-dual-left':
                            $value = chr(28);
                            break;
                        case 'staple-dual-top':
                            $value = chr(29);
                            break;
                        case 'staple-dual-right':
                            $value = chr(30);
                            break;
                        case 'staple-dual-bottom':
                            $value = chr(31);
                            break;
                    }
                break;
            }

        $prepend = '';
        while ((strlen($value) + strlen($prepend)) < 4)
            $prepend .= chr(0);
        
    return $prepend.$value;
    }
    // }}}

    // {{{ _integerBuild ($integer)
    protected function _integerBuild ($value) {
              
        if ($value > 65535 || $value < 0) {
            trigger_error(_("Values must be between 0 and 65535: assuming '0'"),E_USER_WARNING);
            return chr(0x00).chr(0x00).chr(0x00).chr(0x00);
            }
   
        $value = chr($value);
        $prepend = '';
        while ((strlen($value) + strlen($prepend)) < 4)
        $prepend .= chr(0);
        
    return $prepend.$value;
    }
    // }}}

    // {{{ _rangeOfIntegerBuild ($integers)
    protected function _rangeOfIntegerBuild ($integers) {
        
        $integers = split(":",$integers);
        
        
        $i = 0;
        foreach ($integers as $value) {
        $value = chr($value);
        $prepend = '';
        while ((strlen($value) + strlen($prepend)) < 4)
            $prepend .= chr(0x0);
        $outvalue[$i] = $prepend.$value;
        $i ++;
        }
        return $outvalue[0].$outvalue[1];
    }
    // }}}
    
// DEBUGGING

    // {{{ _putDebug($string,$level)
    protected function _putDebug($string,$level=3) {
        if ($level > $this->debug_level)
            return;
        $this->debug[$this->debug_count] = substr($string,0,1024);
        $this->debug_count ++;
        //$this->debug .= substr($string,0,1024);
    }
    // }}}

// LOGGING

    // {{{ _errorLog($log,$level)
    protected function _errorLog($string_to_log,$level) {
        
        if ($level > $this->log_level)
            return;
    
        $string = sprintf('%s : %s:%s user %s : %s',
                            basename($_SERVER['PHP_SELF']),
                            $this->host,
                            $this->port,
                            $this->requesting_user,
                            $string_to_log);
            
        if ($this->log_type == 0) {
            error_log($string);
            return;
            }
        
        $string = sprintf("%s %s Host %s:%s user %s : %s\n",
                            date('M d H:i:s'),
                            basename($_SERVER['PHP_SELF']),
                            $this->host,
                            $this->port,
                            $this->requesting_user,
                            $string_to_log);
        error_log($string,$this->log_type,$this->log_destination);

    return;
    }
    // }}}
   
};

/*************************
*
* Embedded HTTP client
*
**************************/

class http_class {

    // {{{ variables declaration
    public $debug;
    public $html_debug;
    public $timeout = 30; // time waiting for connection, seconds
    public $data_timeout = 30; // time waiting for data, milliseconds
    public $force_multipart_form_post;
    public $username;
    public $password;
    public $request_headers = array();
    public $request_body = "Not a useful information";
    public $status;
    public $window_size = 1400; // chunk size of data
    public $with_exceptions = 0; // compatibility mode for old scripts
    public $port;

    private $default_port = 631;
    private $headers;
    private $reply_headers = array();
    private $reply_body = array();
    private $connection;
    private $arguments;
    private $bodystream = array();
    private $last_limit;
    private $connected;
    private $nc = 1;
    private $user_agent = "PrintIPP http backend V0.3";
    // }}}
            
    // {{{ constructor
    public function __construct() {
        true;
    }
    // }}}

/*********************
*
* Public functions
*
**********************/

    // {{{ GetRequestArguments ($url,&$arguments)
    public function GetRequestArguments ($url,&$arguments) {
        
        $this->arguments = array();
        
        $arguments["URL"] = $this->arguments["URL"] = $url;
        $arguments["RequestMethod"] = $this->arguments["RequestMethod"] = "POST";
        
        $this->headers["User-agent"] = $this->user_agent;
        $this->headers["Content-Type"] = "application/octet-stream";
    
    }
    // }}}

    // {{{ Open ($arguments)
    public function Open ($arguments) {
        
        if (!$this->timeout)
            $this->timeout = 30;
        $url = $arguments["URL"];
        $port = $this->default_port;
        
        $url = split (':',$url,2);
        $transport_type = $url[0];
        switch($transport_type) {
                case 'http':
                    $transport_type = 'tcp://';
                    break;
                case 'https':
                    $transport_type = 'tls://';
                    break;
            }
        $url = $url[1];
        $url = split("/",preg_replace("#^/{1,}#",'',$url),2);
        $url = $url[0];
        $port = $this->port;
        $error = sprintf(_("Cannot resolve url: %s"),$url);
        $ip = gethostbyname($url);
        $ip = @gethostbyaddr($ip);
        if (!$ip)
            if ($this->with_exceptions)
                throw new httpException($error);
            else
                trigger_error($error,E_USER_WARNING);
        if (strstr($url,":")) // we got an ipv6 address
            if (!strstr($url,"[")) // it is not escaped
                $url = sprintf("[%s]",$url);
        $this->connection = @fsockopen($transport_type.$url, $port, $errno, $errstr, $this->timeout);
        $error = sprintf (_('Unable to connect to "%s%s port %s": %s'),
            $transport_type,
            $url,
            $port,
            $errstr);

        if (!$this->connection)
            if ($this->with_exceptions)
                throw new httpException($error,$errno);
            else
                trigger_error($error,E_USER_WARNING);
        $this->connected = 1;
    }
    // }}}

    // {{{ SendRequest($arguments)
    public function SendRequest($arguments) {
         
         if (!$this->data_timeout)
            $this->data_timeout = 30;
            
         if(!$result = self::_StreamRequest($arguments))
            return("SendRequest: unknown error");
         
         self::_ReadReply();

         if (!preg_match('#http/1.1 401 unauthorized#',$this->status))
            return;

         
         $headers = array_keys ($this->reply_headers);
         if (!in_array("www-authenticate",$headers))
            return("SendRequest: need authentication but no mechanism provided");

         $authtype = split(' ',$this->reply_headers["www-authenticate"]);
         $authtype = strtolower($authtype[0]);

         switch ($authtype) {
            case 'basic':
                $pass = base64_encode($this->user.":".$this->password);
                $arguments["Headers"]["Authorization"] = "Basic ".$pass;
                break;
            case 'digest':
                $arguments["Headers"]["Authorization"] = self::_BuildDigest();
                break;
            default:
                return sprintf(_("http_class: need '%s' authentication mechanism, but have not, sorry"),$authtype[0]);
         }
        
         self::Close();
         self::Open($arguments);
         
         if(!$result = self::_StreamRequest($arguments))
            return("SendRequest: unknown error");

          self::_ReadReply();
    
    }
    // }}}

    // {{{ ReadReplyHeaders (&$headers)
    public function ReadReplyHeaders (&$headers) {
        $headers = $this->reply_headers;
    }
    // }}}

    // {{{ ReadReplyBody (&$body,$chunk_size)
    public function ReadReplyBody (&$body,$chunk_size) {
        $body = substr($this->reply_body,$this->last_limit,$chunk_size);
        $this->last_limit += $chunk_size;
    }
    // }}}

    // {{{ Close ()
    public function Close () {
        fclose ($this->connection);
    }
    // }}}

/*********************
*
*  Private functions
*
**********************/

    // {{{ _StreamRequest ($arguments)
    private function _StreamRequest ($arguments) {
        
        $this->status = "";
        $this->reply_headers = array();
        $this->reply_body = "";
        
        $this->arguments = $arguments;
        $this->arguments["Headers"] = array_merge($this->headers,$this->arguments["Headers"]);
        
        //$read = array($this->connection);
        $status = stream_get_meta_data($this->connection);
        //if(stream_select($read, $write = NULL, $except = NULL, 0) === 1) {
        if (isset($status['unread-bytes']) && $status['unread-bytes']) {
            // server talks!
            trigger_error(_("http_class: server talk first, quit"),E_USER_WARNING);
            return false;
            } 

        if ($this->arguments["RequestMethod"] != "POST") {
            trigger_error (sprintf(_("%s: method not implemented"),$arguments["RequestMethod"]),E_USER_WARNING);
            return sprintf(_("%s: method not implemented"),$arguments["RequestMethod"]);
            }
    
        $string = sprintf("POST %s HTTP/1.1\r\n",$this->arguments["RequestURI"]);
        $error = fwrite($this->connection,$string);
        $this->request_headers[$string] = '';
        
        //if(stream_select($read, $write = NULL, $except = NULL, 0) === 1) // server talks!
        $status = stream_get_meta_data($this->connection);
        if (isset($status['unread-bytes']) && $status['unread-bytes'])
             return "server-talk";
        
        if (!$error) {
            trigger_error(_("Error while puts first header"),E_USER_WARNING);
            return _("Stream closed while puts first header");
            }
        
        $content_length = 0;
        foreach ($this->arguments["BodyStream"] as $argument) {
            
            list($type,$value) = each($argument);
            reset ($argument);
            
            if ($type == "Data")
                $length = strlen($value);
                
            elseif ($type == "File")
                if (is_readable($value))
                    $length = filesize($value);
                else {
                    $length = 0;
                    trigger_error(sprintf(_("%s: file is not readable"),$value),E_USER_WARNING);
                    }
            
            else {
                $length = 0;
                trigger_error(sprintf(_("%s: not a valid argument for content"),$type),E_USER_WARNING);
                }
                
            $content_length += $length;
            }
        
        $this->request_body = sprintf(_("%s Bytes"), $content_length);
        $this->arguments["Headers"]["Content-Length"] =  $content_length;
        foreach ($this->arguments["Headers"] as $header => $value) {
            
            $error = @fwrite($this->connection,sprintf("%s: %s\r\n", $header, $value));
            $this->request_headers[$header] = $value;
            //if(stream_select($read, $write = NULL, $except = NULL, 0) === 1)
            $status = stream_get_meta_data($this->connection);
            if (isset($status['unread-bytes']) && $status['unread-bytes'])
                return "server-talk";
                
            if (!$error) {
                trigger_error(_("Error while puts HTTP headers"),E_USER_WARNING);
                return _("Stream closed while puts HTTP headers");
                }
            }
        
            $error = fwrite($this->connection,"\r\n");
            fflush($this->connection);
            
            //if($strselect = stream_select($read, $write = NULL, $except = NULL, 0,$this->data_timeout*1000) === 1) 
            usleep($this->data_timeout*1000);
            $status = stream_get_meta_data($this->connection);
            if (isset($status['unread-bytes']) && $status['unread-bytes'])
                return "server-talk";

            if (!$error) {
                trigger_error(_("Error while ends HTTP headers"),E_USER_WARNING);
                return _("Stream closed while ends HTTP headers");
                }
         
         foreach ($this->arguments["BodyStream"] as $argument) {
            
            list($type,$value) = each($argument);
            reset ($argument);
            
            
            if ($type == "Data") {
                $streamed_length = 0;
                while ($streamed_length < strlen($value)) {
                    //if(stream_select($read, $write = NULL, $except = NULL, 0) === 1)
                    usleep($this->data_timeout*1000);
                    $status = stream_get_meta_data($this->connection);
                    if (isset($status['unread-bytes']) && $status['unread-bytes'])
                        return "server-talk";
                
                    // not very clean...
                    $error = @fwrite($this->connection,substr($value,$streamed_length,$this->window_size));
                    if (!$error)
                        return "error-while-push-data";
                    $streamed_length += $this->window_size;
                    }
                }

                if (!$error) 
                    return _("error-while-push-data");
             
            elseif ($type == "File") {
                if (is_readable($value)) {
                    $file = fopen($value,'rb');
                    while(!feof($file)) {
                    
                        if(gettype($block = @fread($file,$this->window_size)) != "string") {
                            trigger_error(_("cannot read file to upload"),E_USER_WARNING);
                            return _("cannot read file to upload");
                            }

                        //if(stream_select($read, $write = NULL, $except = NULL, 0) === 1)
                        $status = stream_get_meta_data($this->connection);
                        if (isset($status['unread-bytes']) && $status['unread-bytes'])
                            return "server-talks";
                
                        // not very clean...
                        $error = @fwrite($this->connection,$block);
                        if (!$error) 
                            return "error-while-push-data";
                            
                        }
                    }
                }
                
                //if(stream_select($read, $write = NULL, $except = NULL, 0) === 1)
                 $status = stream_get_meta_data($this->connection);
                 if (isset($status['unread-bytes']) && $status['unread-bytes'])
                    return "server-talks";
                
            }

    return true;
    }
    // }}}
    
    // {{{ _ReadReply ()
    private function _ReadReply () {
      
        $this->reply_headers = array();
        $this->reply_body = "";
        
        
        
        $line = "1\r\n";
        $headers = "";
        $body = "";
        while (!feof($this->connection)) {
            $line = fgets($this->connection,1024);
            if (strlen($line) <= 2)
                break;
            $headers .= $line;
            }
        $chunk = true;
        
        $headers = preg_split('#\r\n#',$headers);

        $this->status = strtolower($headers[0]);
      
        foreach($headers as $header) {
            if (!preg_match('#www-authenticate: #i',$header))
                $header = strtolower($header);
            
            $header = preg_split("#: #",$header);
            $header[0] =  strtolower($header[0]);

                $this->reply_headers["{$header[0]}"] = array_key_exists(1,$header) ? $header[1] : "";
            }
            unset ($this->reply_headers['']);
         
        //giving 3 chances to complete reading
        $read = array($this->connection);
        for ($i = 0 ; $i < 2 ; $i++) {
            if (self::_ReadStream() === "completed")
                break;
            //$strselect = stream_select($read, $write = NULL, $except = NULL, 0,$this->data_timeout*1000);
            //if (!$strselect)
            usleep($this->data_timeout*1000);
            $status = stream_get_meta_data ($this->connection);
            if ($status["unread_bytes"] == 0)
                break;
            }
            
    return true;      
    }
    // }}}
    
    // {{{ _ReadStream ()
    private function _ReadStream () {
        
        $content_length = 0;
        if (array_key_exists("content-length",$this->reply_headers));
            $content_length = $this->reply_headers["content-length"];

        stream_set_blocking ($this->connection, 0 );
        usleep($this->data_timeout * 1000);
        $total = 0;
        $chunk = true;
        while (true) {
            if ($content_length)
                if (strlen($this->reply_body) >= $content_length)
                    return "completed";
            else
                if (!$chunk)
                    break;
            usleep (1000);
            $chunk = @fread($this->connection,$this->window_size);
            $this->reply_body .= $chunk;
            
            $status = stream_get_meta_data ($this->connection);
            if ($status["unread_bytes"] == 0)
                break;
            }
        stream_set_blocking ($this->connection, 1 );
    return true;
    }
    // }}}

    // {{{ _BuildDigest ()
    private function _BuildDigest () {
        
        $auth = $this->reply_headers["www-authenticate"];
        
        list($head, $auth) = split(" ",$auth,2);

        $auth=split(", ",$auth);
        foreach ($auth as $sheme) {
            list($sheme,$value) = split('=',$sheme);
            $fields[$sheme] = trim(trim($value),'"');
        }

        $nc = sprintf('%x',$this->nc);
        $prepend = "";
        while ((strlen($nc) + strlen($prepend)) < 8)
            $prepend .= "0";
        $nc=$prepend.$nc;
        
        $cnonce = "printipp";
        
        $username = $this->user;
        $password = $this->password;
    
        $A1 = $username.":".$fields["realm"].":".$password;
        
        if (array_key_exists("algorithm",$fields)) {
            $algorithm = strtolower($fields["algorithm"]);
            switch ($algorithm) {
                case "md5":
                    break;
                case "md5-sess":
                    $A1 = $username.":".$fields["realm"].":".$password.":".$fields['nonce'].":".$cnonce;
                    break;
                case "token":
                    trigger_error("http_class: digest Authorization: algorithm 'token' not implemented", E_USER_WARNING);
                    return false;
                    break;
                }

            }
        
        $A2 = "POST:".$this->arguments["RequestURI"];

        
        if (array_key_exists("qop",$fields)) {
            $qop = strtolower($fields["qop"]);
            $qop = split(" ",$qop);
            if (in_array("auth",$qop))
                $qop = "auth";
            else {
                trigger_error("http_class: digest Authorization: qop others than 'auth' not implemented", E_USER_WARNING);
                return false;
                }
            }

            
        $response = md5(md5($A1).":". $fields["nonce"].":" .md5($A2));
        
        if (isset($qop) && ($qop == "auth"))
        $response = md5(md5($A1).":".$fields["nonce"].":".$nc.":".$cnonce.":".$qop.":".$A2);

        
        $auth_scheme = sprintf('Digest username="%s", realm="%s", nonce="%s", uri="%s", response="%s"',
                                $username,
                                $fields["realm"],
                                $fields['nonce'],
                                $this->arguments["RequestURI"],
                                $response
                                );
        
        if (isset($algorithm))
            $auth_scheme .= sprintf(', algorithm="%s"',$algorithm);

        if (isset($qop))
            $auth_scheme .= sprintf(', cnonce="%s"',$cnonce);
        
        if(array_key_exists("opaque",$fields))
            $auth_scheme .= sprintf(', opaque="%s"',$fields['opaque']);
      
        if (isset($qop))
            $auth_scheme .= sprintf(', qop="%s"',$qop);
      
        $auth_scheme .= sprintf(', nc=%s',$nc);
      
        $this->nc++;
        
    return $auth_scheme;
    }
    // }}}

};



/*
 * Local variables:
 * mode: php
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 */
?>
