/*
 * pdsoctave.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <stdlib.h>
#include <string.h>
#include <pds/pdsra.h>
#include <pds/pdsoctave.h>

#define PDS_OCTAVE_FONTSIZE 16
#define PDS_OCTAVE_IMAGESIZE "800,600"

/** \fn pds_octave_surf_matrix(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *imgfile)
 *  \brief Imprime una matriz M en un archivo de imagen usando como intermedio un 
 *  archivo *.m de octave. Usando el comando surf.
 *
 *  <center>
 *  \image html pds_octave_surf_matrix.png "Usando la función surf() de Octave."
 *  </center>
 *  \param[in] M Matriz a imprimir.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] labelz Etiqueta del eje Z: Altura.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_surf_matrix(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *imgfile)
{
	int i,j;
	FILE *fd=NULL;
	char *orden=NULL;

	if(M==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(labelz==NULL)	return FALSE;
	if(imgfile==NULL)	return FALSE;

    char *OCTAVEFILE=NULL;
    if(octfile==NULL)
    {
        OCTAVEFILE=(char*)calloc(64,sizeof(char));
        if(OCTAVEFILE==NULL)    return FALSE;
        sprintf(OCTAVEFILE,"temp%d.m",rand());
    }
    else
    {
        OCTAVEFILE=strdup(octfile);
        if(OCTAVEFILE==NULL)    return FALSE;
    }



	fd=fopen(OCTAVEFILE,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"M=[");
	for(i=0 ; i<M->Nlin ; i++)	
	{
		for(j=0 ; j<M->Ncol ; j++)
		{
			if(j!=M->Ncol-1)	fprintf(fd,"%e\t",M->M[i][j]);
			else			fprintf(fd,"%e\n",M->M[i][j]);
		}
	}
	fprintf(fd,"];\n");
	fprintf(fd,"h=figure(1);\n");// h=figure('Visible', 'off');
	fprintf(fd,"surf(M);\n");
	fprintf(fd,"shading interp;\n");
	fprintf(fd,"grid on;\n");

	fprintf(fd,"hx=xlabel(\'%s\');\n",labelx);
	fprintf(fd,"hy=ylabel(\'%s\');\n",labely);
	fprintf(fd,"hz=zlabel(\'%s\');\n",labelz);
    fprintf(fd,"ha = gca(); \n");
    fprintf(fd,"FONTSIZE=%d;\n",PDS_OCTAVE_FONTSIZE);
    fprintf(fd,"set(hx,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(hy,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(hz,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(ha,'fontsize',FONTSIZE);\n");

    fprintf(fd,"refresh();\n");
	fprintf(fd,"saveas (h,\'%s\');\n",imgfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,48+strlen(OCTAVEFILE));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave --no-gui -q %s",OCTAVEFILE);
	i=system(orden);

    if(octfile==NULL)
    {
        remove(OCTAVEFILE); 
    }
    free(OCTAVEFILE);
	return TRUE;
}


/** \fn int pds_octave_pcolor_matrix(const PdsMatrix *M,const char *labelx,const char *labely,const char *octfile,const char *imgfile)
 *  \brief Imprime una matriz M en un archivo de imagen usando como intermedio un 
 *  archivo *.m de octave. Usando el comando pcolor.
 *
 *  <center>
 *  \image html pds_octave_pcolor_matrix.png "Usando la función pcolor() de Octave."
 *  </center>
 *  \param[in] M Matriz a imprimir.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_pcolor_matrix(const PdsMatrix *M,const char *labelx,const char *labely,const char *octfile,const char *imgfile)
{
	int i,j;
	FILE *fd=NULL;
	char *orden=NULL;

	if(M==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(imgfile==NULL)	return FALSE;

    char *OCTAVEFILE=NULL;
    if(octfile==NULL)
    {
        OCTAVEFILE=(char*)calloc(64,sizeof(char));
        if(OCTAVEFILE==NULL)    return FALSE;
        sprintf(OCTAVEFILE,"temp%d.m",rand());
    }
    else
    {
        OCTAVEFILE=strdup(octfile);
        if(OCTAVEFILE==NULL)    return FALSE;
    }

	fd=fopen(OCTAVEFILE,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"M=[");
	for(i=0 ; i<M->Nlin ; i++)	
	{
		for(j=0 ; j<M->Ncol ; j++)
		{
			if(j!=M->Ncol-1)	fprintf(fd,"%e\t",M->M[i][j]);
			else			fprintf(fd,"%e\n",M->M[i][j]);
		}
	}
	fprintf(fd,"];\n");
	fprintf(fd,"h=figure(1);\n");// h=figure('Visible', 'off');
	fprintf(fd,"pcolor(M);\n");
	fprintf(fd,"colorbar;\n");
	fprintf(fd,"shading flat;\n");

	fprintf(fd,"hx=xlabel(\'%s\');\n",labelx);
	fprintf(fd,"hy=ylabel(\'%s\');\n",labely);
    fprintf(fd,"ha = gca(); \n");
    fprintf(fd,"FONTSIZE=%d;\n",PDS_OCTAVE_FONTSIZE);
    fprintf(fd,"set(hx,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(hy,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(ha,'fontsize',FONTSIZE);\n");

    fprintf(fd,"refresh();\n");
	fprintf(fd,"saveas (h,\'%s\');\n",imgfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,48+strlen(OCTAVEFILE));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave --no-gui -q %s",OCTAVEFILE);
	i=system(orden);

    if(octfile==NULL)
    {
        remove(OCTAVEFILE); 
    }
    free(OCTAVEFILE);
	return TRUE;
}


/** \fn int pds_octave_imwrite_matrix(const PdsMatrix *M,const char *octfile,const char *imgfile)
 *  \brief Imprime una matriz M en un archivo de imagen usando como intermedio un 
 *  archivo *.m de octave. Usando el comando imwrite en escala de grises.
 *
 *  <center>
 *  \image html pds_octave_imwrite_matrix.png "Usando la función imwrite() de Octave."
 *  </center>
 *  \param[in] M Matriz a imprimir. Esta matriz debe tener elements entre 0 y 1.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_imwrite_matrix(const PdsMatrix *M,const char *octfile,const char *imgfile)
{
	int i,j;
	FILE *fd=NULL;
	char *orden=NULL;

	if(M==NULL)		return FALSE;
	if(imgfile==NULL)	return FALSE;

    char *OCTAVEFILE=NULL;
    if(octfile==NULL)
    {
        OCTAVEFILE=(char*)calloc(64,sizeof(char));
        if(OCTAVEFILE==NULL)    return FALSE;
        sprintf(OCTAVEFILE,"temp%d.m",rand());
    }
    else
    {
        OCTAVEFILE=strdup(octfile);
        if(OCTAVEFILE==NULL)    return FALSE;
    }



	fd=fopen(OCTAVEFILE,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"M=[");
	for(i=0 ; i<M->Nlin ; i++)	
	{
		for(j=0 ; j<M->Ncol ; j++)
		{
			if(j!=M->Ncol-1)	fprintf(fd,"%e\t",M->M[i][j]);
			else			fprintf(fd,"%e\n",M->M[i][j]);
		}
	}
	fprintf(fd,"];\n");

    fprintf(fd,"imwrite(M,\'%s\');\n",imgfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,48+strlen(OCTAVEFILE));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave --no-gui -q %s",OCTAVEFILE);
	i=system(orden);

    if(octfile==NULL)
    {
        remove(OCTAVEFILE); 
    }
    free(OCTAVEFILE);
	return TRUE;
}


/** \fn int pds_octave_plot_vector(const PdsVector *V,const char *labelx,const char *labely,const char *octfile,const char *imgfile)
 *  \brief Imprime un vector V en un archivo de imagen usando como intermedio un 
 *  archivo *.m de octave.
 *
 *  <center>
 *  \image html pds_octave_plot_vector.png "Usando la función plot() de Octave."
 *  </center>
 *  \param[in] V Vector a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_plot_vector(const PdsVector *V,const char *labelx,const char *labely,const char *octfile,const char *imgfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(V==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(imgfile==NULL)	return FALSE;

    char *OCTAVEFILE=NULL;
    if(octfile==NULL)
    {
        OCTAVEFILE=(char*)calloc(64,sizeof(char));
        if(OCTAVEFILE==NULL)    return FALSE;
        sprintf(OCTAVEFILE,"temp%d.m",rand());
    }
    else
    {
        OCTAVEFILE=strdup(octfile);
        if(OCTAVEFILE==NULL)    return FALSE;
    }



	fd=fopen(OCTAVEFILE,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"V=[");
	for(i=0 ; i<V->Nel ; i++)	fprintf(fd,"%e\t",V->V[i]);
	fprintf(fd,"];\n");
	fprintf(fd,"N=length(V);\n");
	fprintf(fd,"X=[0:N-1];\n");
	fprintf(fd,"h=figure(1);\n");// h=figure('Visible', 'off');
	fprintf(fd,"plot(X,V,\'o-\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"hx=xlabel(\'%s\');\n",labelx);
	fprintf(fd,"hy=ylabel(\'%s\');\n",labely);
	fprintf(fd,"xlim([min(X),max(X)]);\n");
    fprintf(fd,"ha = gca(); \n");
    fprintf(fd,"FONTSIZE=%d;\n",PDS_OCTAVE_FONTSIZE);
    fprintf(fd,"set(hx,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(hy,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(ha,'fontsize',FONTSIZE);\n");

    fprintf(fd,"refresh();\n");
	fprintf(fd,"saveas (h,\'%s\');\n",imgfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,48+strlen(OCTAVEFILE));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave --no-gui -q %s",OCTAVEFILE);
	i=system(orden);

    if(octfile==NULL)
    {
        remove(OCTAVEFILE); 
    }
    free(OCTAVEFILE);
	return TRUE;
}


/** \fn int pds_octave_plot_vectors(const PdsVector *X,const PdsVector *Y,const char *labelx,const char *labely,const char *octfile,const char *imgfile)
 *  \brief Imprime un vector Y vs X en un archivo de imagen usando como intermedio un 
 *  archivo *.m de octave.
 *
 *  <center>
 *  \image html pds_octave_plot_vector.png "Usando la función plot() de Octave."
 *  </center>
 *  \param[in] X Eje X del vector a imprimir.
 *  Si X==NULL entonces se mostrara la gráfica usando el numero de muestras
 *  como eje X.
 *  \param[in] Y Eje Y del vector a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_plot_vectors(const PdsVector *X,const PdsVector *Y,const char *labelx,const char *labely,const char *octfile,const char *imgfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(Y==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(imgfile==NULL)	return FALSE;
    if(X!=NULL)	if(X->Nel!=Y->Nel)	return FALSE;


    char *OCTAVEFILE=NULL;
    if(octfile==NULL)
    {
        OCTAVEFILE=(char*)calloc(64,sizeof(char));
        if(OCTAVEFILE==NULL)    return FALSE;
        sprintf(OCTAVEFILE,"temp%d.m",rand());
    }
    else
    {
        OCTAVEFILE=strdup(octfile);
        if(OCTAVEFILE==NULL)    return FALSE;
    }



	fd=fopen(OCTAVEFILE,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"Y=[");
	for(i=0 ; i<Y->Nel ; i++)	fprintf(fd,"%e\t",Y->V[i]);
	fprintf(fd,"];\n");
    if(X==NULL)
    {
	    fprintf(fd,"N=length(Y);\n");
	    fprintf(fd,"X=[0:N-1];\n");
    }
    else
    {
	    fprintf(fd,"X=[");
    	for(i=0 ; i<X->Nel ; i++)	fprintf(fd,"%e\t",X->V[i]);
    	fprintf(fd,"];\n");
    }
	fprintf(fd,"h=figure(1);\n");// h=figure('Visible', 'off');
	fprintf(fd,"plot(X,Y,\'o-\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"hx=xlabel(\'%s\');\n",labelx);
	fprintf(fd,"hy=ylabel(\'%s\');\n",labely);
	fprintf(fd,"xlim([min(X),max(X)]);\n");
    fprintf(fd,"ha = gca(); \n");
    fprintf(fd,"FONTSIZE=%d;\n",PDS_OCTAVE_FONTSIZE);
    fprintf(fd,"set(hx,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(hy,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(ha,'fontsize',FONTSIZE);\n");

    fprintf(fd,"refresh();\n");
	fprintf(fd,"saveas (h,\'%s\');\n",imgfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,48+strlen(OCTAVEFILE));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave --no-gui -q %s",OCTAVEFILE);
	i=system(orden);


    if(octfile==NULL)
    {
        remove(OCTAVEFILE); 
    }
    free(OCTAVEFILE);
	return TRUE;
}


/** \fn int pds_octave_plot_compare_vectors(const PdsVector *X,const PdsVector *U,const PdsVector *V,const char *labelx,const char *labely,const char *legendu,const char *legendv,const char *octfile,const char *imgfile)
 *  \brief Imprime un vector U y V en un archivo de imagen usando como intermedio un 
 *  archivo *.m de octave.
 *
 *  <center>
 *  \image html pds_octave_plot_compare_vectors.png "Usando la función plot() de Octave."
 *  </center>
 *  \param[in] X Vector correspondiente a el eje X de los vectores U y V.
 *  Si X==NULL entonces se mostrara la comparación usando el numero de muestras
 *  como eje X.
 *  \param[in] U Vector a imprimir.
 *  \param[in] V Vector a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] legendu Legenda de U.
 *  \param[in] legendv Legenda de V.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_plot_compare_vectors(const PdsVector *X,const PdsVector *U,const PdsVector *V,const char *labelx,const char *labely,const char *legendu,const char *legendv,const char *octfile,const char *imgfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(U==NULL)		return FALSE;
	if(V==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(imgfile==NULL)	return FALSE;
	if(U->Nel!=V->Nel)	return FALSE;

    if(X!=NULL)
	if(X->Nel!=V->Nel)	return FALSE;

    char *OCTAVEFILE=NULL;
    if(octfile==NULL)
    {
        OCTAVEFILE=(char*)calloc(64,sizeof(char));
        if(OCTAVEFILE==NULL)    return FALSE;
        sprintf(OCTAVEFILE,"temp%d.m",rand());
    }
    else
    {
        OCTAVEFILE=strdup(octfile);
        if(OCTAVEFILE==NULL)    return FALSE;
    }



	fd=fopen(OCTAVEFILE,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"U=[");
	for(i=0 ; i<U->Nel ; i++)	fprintf(fd,"%e\t",U->V[i]);
	fprintf(fd,"];\n");
	fprintf(fd,"V=[");
	for(i=0 ; i<V->Nel ; i++)	fprintf(fd,"%e\t",V->V[i]);
	fprintf(fd,"];\n");
    if(X==NULL)
    {
	    fprintf(fd,"N=length(V);\n");
	    fprintf(fd,"X=[0:N-1];\n");
    }
    else
    {
	    fprintf(fd,"X=[");
    	for(i=0 ; i<X->Nel ; i++)	fprintf(fd,"%e\t",X->V[i]);
    	fprintf(fd,"];\n");
    }
	fprintf(fd,"h=figure(1);\n");// h=figure('Visible', 'off');
	fprintf(fd,"plot(X,U,\'o-\',X,V,\'s-\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"hx=xlabel(\'%s\');\n",labelx);
	fprintf(fd,"hy=ylabel(\'%s\');\n",labely);
	fprintf(fd,"xlim([min(X),max(X)]);\n");
	fprintf(fd,"hl = legend(\'%s\',\'%s\');\n",legendu,legendv);
    fprintf(fd,"legend (hl, 'location', 'north', 'orientation', 'horizontal');\n");
    fprintf(fd,"ha = gca(); \n");
    fprintf(fd,"FONTSIZE=%d;\n",PDS_OCTAVE_FONTSIZE);
    fprintf(fd,"set(hx,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(hy,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(ha,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(hl,'fontsize',FONTSIZE);\n");

    fprintf(fd,"refresh();\n");
	fprintf(fd,"saveas (h,\'%s\');\n",imgfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,48+strlen(OCTAVEFILE));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave --no-gui -q %s",OCTAVEFILE);
	i=system(orden);


    if(octfile==NULL)
    {
        remove(OCTAVEFILE); 
    }
    free(OCTAVEFILE);
	return TRUE;
}



/** \fn int pds_octave_plot_file_col(const char *datafile,int col,const char *labelx,const char *labely,const char *octfile,const char *imgfile)
 *  \brief Imprime una columna de un archivo a un archivo de imagen usando como intermedio un 
 *  archivo *.m de octave.
 *
 *  <center>
 *  \image html pds_octave_plot_vector.png "Usando la función plot() de Octave."
 *  </center>
 *  \param[in] datafile Archivo de datos.
 *  \param[in] col Columna a imprimir, inicia en 1.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_plot_file_col(const char *datafile,int col,const char *labelx,const char *labely,const char *octfile,const char *imgfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(datafile==NULL)	return FALSE;
	if(col<=0)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(imgfile==NULL)	return FALSE;

    char *OCTAVEFILE=NULL;
    if(octfile==NULL)
    {
        OCTAVEFILE=(char*)calloc(64,sizeof(char));
        if(OCTAVEFILE==NULL)    return FALSE;
        sprintf(OCTAVEFILE,"temp%d.m",rand());
    }
    else
    {
        OCTAVEFILE=strdup(octfile);
        if(OCTAVEFILE==NULL)    return FALSE;
    }



	fd=fopen(OCTAVEFILE,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"M=load(\'%s\');\n",datafile);
	fprintf(fd,"V=M(:,%d);\n",col);
	fprintf(fd,"\n");
	fprintf(fd,"N=length(V);\n");
	fprintf(fd,"X=[0:N-1];\n");
	fprintf(fd,"h=figure(1);\n");// h=figure('Visible', 'off');
	fprintf(fd,"plot(X,V,\'o-\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"hx=xlabel(\'%s\');\n",labelx);
	fprintf(fd,"hy=ylabel(\'%s\');\n",labely);
	fprintf(fd,"xlim([min(X),max(X)]);\n");
    fprintf(fd,"ha = gca(); \n");
    fprintf(fd,"FONTSIZE=%d;\n",PDS_OCTAVE_FONTSIZE);
    fprintf(fd,"set(hx,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(hy,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(ha,'fontsize',FONTSIZE);\n");

    fprintf(fd,"refresh();\n");
	fprintf(fd,"saveas (h,\'%s\');\n",imgfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,48+strlen(OCTAVEFILE));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave --no-gui -q %s",OCTAVEFILE);
	i=system(orden);


    if(octfile==NULL)
    {
        remove(OCTAVEFILE); 
    }
    free(OCTAVEFILE);
	return TRUE;
}


/** \fn int pds_octave_plot_file_cols(const char *datafile,int colx,int coly,const char *labelx,const char *labely,const char *octfile,const char *imgfile)
 *  \brief Imprime Y vs X de un par de columnas de un archivo a un archivo de imagen 
 *  usando como intermedio un archivo *.m de octave.
 *
 *  <center>
 *  \image html pds_octave_plot_vector.png "Usando la función plot() de Octave."
 *  </center>
 *  \param[in] datafile Archivo de datos.
 *  \param[in] colx Columna a imprimir como eje X, inicia en 1.
 *  \param[in] coly Columna a imprimir como eje Y, inicia en 1.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  Si ostfile==NULL entonces se creará un fichero con nombre aleatorio y se 
 *  borrará al terminar de crear el archivo de imagen.
 *  \param[in] imgfile Nombre del archivo de imagen generado. 
 *  El tipo de archivo es seleccionado por la extensión del nombre imgfile.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctaveGroup
 */
int pds_octave_plot_file_cols(const char *datafile,int colx,int coly,const char *labelx,const char *labely,const char *octfile,const char *imgfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(datafile==NULL)	return FALSE;
	if(colx<=0)		return FALSE;
	if(coly<=0)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(imgfile==NULL)	return FALSE;

    char *OCTAVEFILE=NULL;
    if(octfile==NULL)
    {
        OCTAVEFILE=(char*)calloc(64,sizeof(char));
        if(OCTAVEFILE==NULL)    return FALSE;
        sprintf(OCTAVEFILE,"temp%d.m",rand());
    }
    else
    {
        OCTAVEFILE=strdup(octfile);
        if(OCTAVEFILE==NULL)    return FALSE;
    }



	fd=fopen(OCTAVEFILE,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"M=load(\'%s\');\n",datafile);
	fprintf(fd,"X=M(:,%d);\n",colx);
	fprintf(fd,"Y=M(:,%d);\n",coly);
	fprintf(fd,"\n");
	fprintf(fd,"h=figure(1);\n");// h=figure('Visible', 'off');
	fprintf(fd,"plot(X,Y,\'o-\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"hx=xlabel(\'%s\');\n",labelx);
	fprintf(fd,"hy=ylabel(\'%s\');\n",labely);
	fprintf(fd,"xlim([min(X),max(X)]);\n");
    fprintf(fd,"ha = gca(); \n");
    fprintf(fd,"FONTSIZE=%d;\n",PDS_OCTAVE_FONTSIZE);
    fprintf(fd,"set(hx,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(hy,'fontsize',FONTSIZE);\n");
    fprintf(fd,"set(ha,'fontsize',FONTSIZE);\n");

    fprintf(fd,"refresh();\n");
	fprintf(fd,"saveas (h,\'%s\');\n",imgfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,48+strlen(OCTAVEFILE));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave --no-gui -q %s",OCTAVEFILE);
	i=system(orden);


    if(octfile==NULL)
    {
        remove(OCTAVEFILE); 
    }
    free(OCTAVEFILE);
	return TRUE;
}

