/*
 * pdsoctplotpng.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <stdlib.h>
#include <string.h>
#include <pds/pdsra.h>
#include <pds/pdsca.h>
#include <pds/pdsoctplotpng.h>


/** \fn int pds_octplot_surf_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile)
 *  \brief Imprime una matriz M en un archivo png usando como intermedio un 
 *  archivo *.m de octave. Usando el comando surf.
 *  \param[in] M Matriz a imprimir.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] labelz Etiqueta del eje Z: Altura.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_surf_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile)
{
	int i,j;
	FILE *fd=NULL;
	char *orden=NULL;

	if(M==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(labelz==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"M=[");
	for(i=0 ; i<M->Nlin ; i++)	
	{
		for(j=0 ; j<M->Ncol ; j++)
		{
			if(j!=M->Ncol-1)	fprintf(fd,"%e\t",M->M[i][j]);
			else			fprintf(fd,"%e\n",M->M[i][j]);
		}
	}
	fprintf(fd,"];\n");
	fprintf(fd,"surf(M);\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'%s\');\n",labelx);
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"zlabel(\'%s\');\n",labelz);
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}


/** \fn int pds_octplot_pcolor_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile)
 *  \brief Imprime una matriz M en un archivo png usando como intermedio un 
 *  archivo *.m de octave. Usando el comando pcolor.
 *  \param[in] M Matriz a imprimir.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] labelz Etiqueta del eje Z: Altura.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_pcolor_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile)
{
	int i,j;
	FILE *fd=NULL;
	char *orden=NULL;

	if(M==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(labelz==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"M=[");
	for(i=0 ; i<M->Nlin ; i++)	
	{
		for(j=0 ; j<M->Ncol ; j++)
		{
			if(j!=M->Ncol-1)	fprintf(fd,"%e\t",M->M[i][j]);
			else			fprintf(fd,"%e\n",M->M[i][j]);
		}
	}
	fprintf(fd,"];\n");
	fprintf(fd,"pcolor(M);\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'%s\');\n",labelx);
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"zlabel(\'%s\');\n",labelz);
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}


/** \fn int pds_octplot_image_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile)
 *  \brief Imprime una matriz M en un archivo png usando como intermedio un 
 *  archivo *.m de octave. Usando el comando image.
 *  \param[in] M Matriz a imprimir.
 *  \param[in] labelx Etiqueta del eje X: Columnas.
 *  \param[in] labely Etiqueta del eje Y: Lineas.
 *  \param[in] labelz Etiqueta del eje Z: Altura.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_image_matrix_in_png(const PdsMatrix *M,const char *labelx,const char *labely,const char *labelz,const char *octfile,const char *pngfile)
{
	int i,j;
	FILE *fd=NULL;
	char *orden=NULL;

	if(M==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(labelz==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"M=[");
	for(i=0 ; i<M->Nlin ; i++)	
	{
		for(j=0 ; j<M->Ncol ; j++)
		{
			if(j!=M->Ncol-1)	fprintf(fd,"%e\t",M->M[i][j]);
			else			fprintf(fd,"%e\n",M->M[i][j]);
		}
	}
	fprintf(fd,"];\n");
	fprintf(fd,"image(M);\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'%s\');\n",labelx);
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"zlabel(\'%s\');\n",labelz);
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}


/** \fn int pds_octplot_vector_in_png(const PdsVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime un vector V en un archivo png usando como intermedio un 
 *  archivo *.m de octave.
 *  \param[in] V Vector a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_vector_in_png(const PdsVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(V==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"V=[");
	for(i=0 ; i<V->Nel ; i++)	fprintf(fd,"%e\t",V->V[i]);
	fprintf(fd,"];\n");
	fprintf(fd,"N=length(V);\n");
	fprintf(fd,"EjeX=[0:N-1];\n");
	fprintf(fd,"plot(EjeX,V,\'o-\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'%s\');\n",labelx);
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}


/** \fn int pds_octplot_vectors_in_png(const PdsVector *X,const PdsVector *Y,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime un vector Y vs X en un archivo png usando como intermedio un 
 *  archivo *.m de octave.
 *  \param[in] X Eje X del vector a imprimir.
 *  \param[in] Y Eje Y del vector a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_vectors_in_png(const PdsVector *X,const PdsVector *Y,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(X==NULL)		return FALSE;
	if(Y==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;
	if(X->Nel!=Y->Nel)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"X=[");
	for(i=0 ; i<X->Nel ; i++)	fprintf(fd,"%e\t",X->V[i]);
	fprintf(fd,"];\n");
	fprintf(fd,"Y=[");
	for(i=0 ; i<Y->Nel ; i++)	fprintf(fd,"%e\t",Y->V[i]);
	fprintf(fd,"];\n");
	fprintf(fd,"plot(X,Y,\'o-\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'%s\');\n",labelx);
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}


/** \fn int pds_octplot_cvector_in_png(const PdsCVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime un vector complejo V en un archivo png usando como intermedio un 
 *  archivo *.m de octave.
 *  \param[in] V Vector complejo a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_cvector_in_png(const PdsCVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(V==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"Vreal=[");
	for(i=0 ; i<V->Nel ; i++)	fprintf(fd,"%e\t",V->V[i].Real);
	fprintf(fd,"];\n");
	fprintf(fd,"Nreal=length(Vreal);\n");
	fprintf(fd,"EjeXreal=[0:Nreal-1];\n");
	fprintf(fd,"Vimag=[");
	for(i=0 ; i<V->Nel ; i++)	fprintf(fd,"%e\t",V->V[i].Imag);
	fprintf(fd,"];\n");
	fprintf(fd,"Nimag=length(Vimag);\n");
	fprintf(fd,"EjeXimag=[0:Nimag-1];\n");
	fprintf(fd,"plot(EjeXreal,Vreal,\'o-\',EjeXimag,Vimag,\'o-\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'%s\');\n",labelx);
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"legend('Real','Imag');\n");
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}


/** \fn int pds_octplot_cvectors_in_png(const PdsVector *X,const PdsCVector *Y,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime un vector complejo Y vs el vector real X en un archivo png 
 *  usando como intermedio un archivo *.m de octave.
 *  \param[in] X Eje X del vector real a imprimir.
 *  \param[in] Y Eje Y del vector complejo a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_cvectors_in_png(const PdsVector *X,const PdsCVector *Y,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(X==NULL)		return FALSE;
	if(Y==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;
	if(X->Nel!=Y->Nel)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"X=[");
	for(i=0 ; i<X->Nel ; i++)	fprintf(fd,"%e\t",X->V[i]);
	fprintf(fd,"];\n");
	fprintf(fd,"Yreal=[");
	for(i=0 ; i<Y->Nel ; i++)	fprintf(fd,"%e\t",Y->V[i].Real);
	fprintf(fd,"];\n");
	fprintf(fd,"Yimag=[");
	for(i=0 ; i<Y->Nel ; i++)	fprintf(fd,"%e\t",Y->V[i].Imag);
	fprintf(fd,"];\n");
	fprintf(fd,"plot(X,Yreal,\'o-\',X,Yimag,\'o-\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'%s\');\n",labelx);
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"legend(\'Real\',\'Imag\');\n");
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}


/** \fn int pds_octplot_vector_compare_in_png(const PdsVector *U,const PdsVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime un vector U y V en un archivo png usando como intermedio un 
 *  archivo *.m de octave.
 *  \param[in] U Vector a imprimir.
 *  \param[in] V Vector a imprimir.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_vector_compare_in_png(const PdsVector *U,const PdsVector *V,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(V==NULL)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;
	//if(U->Nel!=V->Nel)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"U=[");
	for(i=0 ; i<U->Nel ; i++)	fprintf(fd,"%e\t",U->V[i]);
	fprintf(fd,"];\n");
	fprintf(fd,"Nu=length(U);\n");
	fprintf(fd,"EjeXu=[0:Nu-1];\n");
	fprintf(fd,"V=[");
	for(i=0 ; i<V->Nel ; i++)	fprintf(fd,"%e\t",V->V[i]);
	fprintf(fd,"];\n");
	fprintf(fd,"Nv=length(V);\n");
	fprintf(fd,"EjeXv=[0:Nv-1];\n");
	fprintf(fd,"plot(EjeXu,U,\'o-\',EjeXv,V,\'o-\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'%s\');\n",labelx);
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"legend(\'First Vector\',\'Second Vector\');\n");
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}


/** \fn int pds_octplot_afft_vector_in_png(const PdsVector *V,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime el valor absoluto de la FFT de un vector V en un archivo png 
 *  usando como intermedio un archivo *.m de octave.
 *  Imprime el valor absoluto de la FFT de V con 'o'.
 *  \param[in] V Vector a imprimir.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_afft_vector_in_png(const PdsVector *V,const char *labely,const char *octfile,const char *pngfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(V==NULL)		return FALSE;
	if(labely==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"Y1=[");
	for(i=0 ; i<V->Nel ; i++)	fprintf(fd,"%e\t",V->V[i]);
	fprintf(fd,"];\n");
	fprintf(fd,"N=%d;\n",V->Nel);
	fprintf(fd,"X1=[0:N-1]*2/N;\n");
	fprintf(fd,"F1=abs(fft(Y1));\n");
	fprintf(fd,"plot(X1,F1,\'o-\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'[0 -> 2*pi]\');\n");
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}


/** \fn int pds_octplot_afft_vector8_in_png(const PdsVector *V,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime el valor absoluto de la FFT de un vector V en un archivo png 
 *  usando como intermedio un archivo *.m de octave.
 *  Imprime el valor absoluto de la FFT de V con 'o' y además el valor absoluto 
 *  de la FFT de V8=[V,0*V,0*V,0*V,0*V,0*V,0*V,0*V] con '--'.
 *  \param[in] V Vector a imprimir.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_afft_vector8_in_png(const PdsVector *V,const char *labely,const char *octfile,const char *pngfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(V==NULL)		return FALSE;
	if(labely==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"Y1=[");
	for(i=0 ; i<V->Nel ; i++)	fprintf(fd,"%e\t",V->V[i]);
	fprintf(fd,"];\n");
	fprintf(fd,"Y8=[Y1,0*Y1,0*Y1,0*Y1,0*Y1,0*Y1,0*Y1,0*Y1];\n");
	fprintf(fd,"N=%d;\n",V->Nel);
	fprintf(fd,"X1=[0:N-1]*2/N;\n");
	fprintf(fd,"X8=[0:8*N-1]*2/(8*N);\n");
	fprintf(fd,"F1=abs(fft(Y1));\n");
	fprintf(fd,"F8=abs(fft(Y8));\n");
	fprintf(fd,"plot(X1,F1,\'o\',X8,F8,\'--\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'[0 -> 2*pi]\');\n");
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}



/** \fn int pds_octplot_file_col_in_png(const char *datafile,int col,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime una columna de un archivo a un archivo png usando como intermedio un 
 *  archivo *.m de octave.
 *  \param[in] datafile Archivo de datos.
 *  \param[in] col Columna a imprimir, inicia en 1.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_file_col_in_png(const char *datafile,int col,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(datafile==NULL)	return FALSE;
	if(col<=0)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"M=load(\'%s\');\n",datafile);
	fprintf(fd,"V=M(:,%d);\n",col);
	fprintf(fd,"\n");
	fprintf(fd,"N=length(V)\n");
	fprintf(fd,"EjeX=[0:N-1]\n");
	fprintf(fd,"plot(EjeX,V,\'o-\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'%s\');\n",labelx);
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}


/** \fn int pds_octplot_file_cols_in_png(const char *datafile,int colx,int coly,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
 *  \brief Imprime Y vs X de un par de columnas de un archivo a un archivo png 
 *  usando como intermedio un archivo *.m de octave.
 *  \param[in] datafile Archivo de datos.
 *  \param[in] colx Columna a imprimir como eje X, inicia en 1.
 *  \param[in] coly Columna a imprimir como eje Y, inicia en 1.
 *  \param[in] labelx Etiqueta del eje X.
 *  \param[in] labely Etiqueta del eje Y.
 *  \param[in] octfile Nombre del archivo *.m de octave que se usará de intermediario.
 *  \param[in] pngfile Nombre del archivo *.png de salida.
 *  \return TRUE si todo fue bien o FALSE si no, ejemplo punteros igual a NULL.
 *  \ingroup PdsOctPlotPngGroup
 */
int pds_octplot_file_cols_in_png(const char *datafile,int colx,int coly,const char *labelx,const char *labely,const char *octfile,const char *pngfile)
{
	int i;
	FILE *fd=NULL;
	char *orden=NULL;

	if(datafile==NULL)	return FALSE;
	if(colx<=0)		return FALSE;
	if(coly<=0)		return FALSE;
	if(labelx==NULL)	return FALSE;
	if(labely==NULL)	return FALSE;
	if(octfile==NULL)	return FALSE;
	if(pngfile==NULL)	return FALSE;

	fd=fopen(octfile,"w");
	if(fd==NULL)		return FALSE;
	fprintf(fd,"M=load(\'%s\');\n",datafile);
	fprintf(fd,"X=M(:,%d);\n",colx);
	fprintf(fd,"Y=M(:,%d);\n",coly);
	fprintf(fd,"\n");
	fprintf(fd,"plot(X,Y,\'o-\');\n");
	fprintf(fd,"grid on;\n");
	fprintf(fd,"xlabel(\'%s\');\n",labelx);
	fprintf(fd,"ylabel(\'%s\');\n",labely);
	fprintf(fd,"print(\'%s\',\'-dpng\',\'-Farial:20\',\'-S800,600\');\n",pngfile);
	fprintf(fd,"\n");
	fprintf(fd,"\n");
	fclose(fd);

	orden=(char *)calloc(1,16+strlen(octfile));
	if(orden==NULL)	return FALSE;
	sprintf(orden,"octave -q %s",octfile);
	i=system(orden);

	return TRUE;
}

