/*
 * pdsstring.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <stdarg.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <pds/pdsstring.h>
#include <pds/pdscstring.h>
#include <pds/pdsfilesfunc.h>

/** \fn int isintnumber(const char *str)
 *  \brief Esta función verifica si la cadena contiene un número entero. La función
 *  solo desconsidera espacios en blanco iniciales y finales
 *  \param[in] str Cadena a anlizar.
 *  \return Retorna PDS_OK si es un numero y PDS_WRONG quando no lo es o str==NULL
 *  o str es una cadena vacia o hubo un error de allocacion de memoria.
 *  \ingroup PdsStringGroup
 */
int isintnumber(const char *str)
{ 
    int N;
    int i;
    char *dump=NULL;

    if(str==NULL)   return PDS_WRONG;

    N=strlen(str);
    if(N==0)        return PDS_WRONG;

    dump=(char *)calloc(N+1,sizeof(char));
    if(dump==NULL)  return  PDS_WRONG;
    
    strcpy(dump,str);
    pds_string_clean_beginning(dump,' ');
    pds_string_clean_ending(dump,' ');

    N=strlen(dump);
    for(i=0;i<N;i++)
    {
        if(i==0)
        {
            if((isdigit(dump[i])!=0)||(dump[i]=='+')||(dump[i]=='-'))
            {
                i=i;
            }
            else
            {
                free(dump);
                return PDS_WRONG;
            } 
            
        }
        else
        {
            if(isdigit(dump[i])==0)
            {
                free(dump);
                return PDS_WRONG;
            }
        }
    }

    return PDS_OK;
}
 

/** \fn char *pds_itoa(int num,int base)
 *  \brief Esta función es similar a la función itoa() pero retorna una nueva
 *  cadena de texto con el número num en la base base.
 *  \param[in] num Número a escribir.
 *  \param[in] base Base del número a escribir.
 *  \return Retorna una nueva cadena con el numero num en la base base..
 *  Retorna NULL si hubo problemas de reserva de memoria o base <=0 .
 *  \ingroup PdsStringGroup
 */
char *pds_itoa(int num,int base)
{   
    char *buffer=NULL;
    int N,tmp,i;
    int signo=0;

    if(base<=0) return NULL;

    if(num<0)
    {
        signo=1;
        num=-num;
    }

    if(signo==1)    N=0;
    else            N=1;

    tmp=num;
    do{
        tmp=(tmp/base);
        N++;
    }while (tmp>0);

    buffer=(char*)calloc(N+1,sizeof(char));
    if(buffer!=NULL)
    {        
        i=0;
        do{
            tmp=num%base;
            num=num/base;
            buffer[N-1-i]=tmp;
            i++;
        }while(num>0);

        if(signo==1) buffer[0]='-';
    }
    return buffer;
}

/** \fn char *pds_toupper(const char *str)
 *  \brief Esta función es similar a la función toupper() pero transforma una  
 *  cadena de texto (y no un caracter) a letras mayúsculas.
 *  \param[in] str Cadena a convetir en mayúsculas.
 *  \return Retorna una nueva cadena con el resultado de convertir en mayúsculas.
 *  Retorna NULL si str es NULL o si hubo problemas de reserva de memoria.
 *  \ingroup PdsStringGroup
 */
char *pds_toupper(const char *str)
{
    char *dat=NULL;
    int i;
    
    if(str==NULL)   return  NULL;
    
    int N=strlen(str);
    dat=(char *)calloc(N+1,sizeof(char));
    if(dat==NULL)   return  NULL;

    for(i=0;i<N;i++) dat[i]=toupper(str[i]);
    
    return dat;
}

/** \fn int pds_inupper(char *str)
 *  \brief Esta función  modifica una cadena de texto a letras mayúsculas.
 *  \param[in] str Cadena a convetir en mayúsculas.
 *  \return Retorna PDS_OK si todo fue bien o PDS_WRONG si str es NULL.
 *  \ingroup PdsStringGroup
 */
int pds_inupper(char *str)
{
    int i;
    
    if(str==NULL)   return  PDS_WRONG;
    
    int N=strlen(str);

    for(i=0;i<N;i++) str[i]=toupper(str[i]);
    
    return PDS_OK;
}

/** \fn char *pds_tolower(const char *str)
 *  \brief Esta función es similar a la función tolower() pero transforma una  
 *  cadena de texto (y no un caracter) a letras minúsculas.
 *  \param[in] str Cadena a convetir en minúsculas.
 *  \return Retorna una nueva cadena con el resultado de convertir en minúsculas.
 *  Retorna NULL si str es NULL o si hubo problemas de reserva de memoria.
 *  \ingroup PdsStringGroup
 */
char *pds_tolower(const char *str)
{
    char *dat=NULL;
    int i;
    
    if(str==NULL)   return  NULL;
    
    int N=strlen(str);
    dat=(char *)calloc(N+1,sizeof(char));
    if(dat==NULL)   return  NULL;

    for(i=0;i<N;i++) dat[i]=tolower(str[i]);
    
    return dat;
}


/** \fn int pds_inlower(char *str)
 *  \brief Esta función  modifica una cadena de texto a letras minúsculas.
 *  \param[in] str Cadena a convetir en minúsculas.
 *  \return Retorna PDS_OK si todo fue bien o PDS_WRONG si str es NULL.
 *  \ingroup PdsStringGroup
 */
int pds_inlower(char *str)
{
    int i;
    
    if(str==NULL)   return  PDS_WRONG;
    
    int N=strlen(str);

    for(i=0;i<N;i++) str[i]=tolower(str[i]);
    
    return PDS_OK;
}    

/** \fn char* pds_sprintf(const char *format, ...)
 *  \brief Similar a sprintf pero esta retorna un puntero a una nueva cadena con 
 *  el texto.
 *
 *  \param[in] format Formato de texto de salida.
 *  \param[in] ... Otros argumentos.
 *  \return Retorna un puntero a una nueva cadena con el texto. O NULL en caso de error.
 *  \ingroup PdsStringGroup
 */
char* pds_sprintf(const char *format, ...)
{
    char *char_string=NULL;
    va_list args;               /* points to each unnamed arg in turn */
    int N;

    va_start( args, format );   /* make args point to 1st unnamed arg */
    N=vsnprintf(NULL,0,format, args);
    va_end( args );             /* clean up when done */

    char_string=(char*)calloc(N+1,sizeof(char));
    if(char_string==NULL)
    {
        return NULL;
    }

    va_start( args, format );   /* make args point to 1st unnamed arg */
    vsprintf( char_string, format, args );
    va_end( args );             /* clean up when done */

    return char_string;
}

/* strsep.c
   Copyright (C) 1992-2015 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
 */
char *pds_strsep (char **stringp, const char *delim)
{
  char *begin, *end;

  begin = *stringp;
  if (begin == NULL)
    return NULL;

  /* A frequent case is when the delimiter string contains only one
     character.  Here we don't need to call the expensive `strpbrk'
     function and instead work using `strchr'.  */
  if (delim[0] == '\0' || delim[1] == '\0')
    {
      char ch = delim[0];

      if (ch == '\0')
	end = NULL;
      else
	{
	  if (*begin == ch)
	    end = begin;
	  else if (*begin == '\0')
	    end = NULL;
	  else
	    end = strchr (begin + 1, ch);
	}
    }
  else
    /* Find the end of the token.  */
    end = strpbrk (begin, delim);

  if (end)
    {
      /* Terminate the token and set *STRINGP past NUL character.  */
      *end++ = '\0';
      *stringp = end;
    }
  else
    /* No more delimiters; this is the last token.  */
    *stringp = NULL;

  return begin;
}

/** \fn PdsCellString *pds_strsep_cell_string(const char *str, const char *delim)
 *  \brief Esta función es similar a la función strsep, donde la cadena str es 
 *  dividida usando algunos de los delimitadores especificados en delim. Los 
 *  pedazos son cargados en una estructura PdsCellString* retornada por la función.
 *  \param[in] str Cadena a dividir, esta cadena no sera modificada.
 *  \param[in] delim Delimitadores a usar. Si delim es NULL la función retorna NULL.
 *  Si delim es una cadena vacia es retornada la cadena entera dentro de una PdsCellString.
 *  \return Retorna un puntero a una nueva estructura de tipo PdsCellString *,
 *  con los tokens de la división.
 *  \ingroup PdsStringGroup
 */
PdsCellString *pds_strsep_cell_string(const char *str, const char *delim)
{
    char *str0 =NULL; 
    char *token=NULL;
    char *str1=NULL;    
    PdsCellString *Tok=NULL;
    int dat;

    if(str==NULL)   return NULL;
    if(delim==NULL) return NULL;

    str0=strdup(str);
    if(str0==NULL)  return NULL;

    Tok=pds_cell_string_new(0);
    if(Tok==NULL)
    {
        return NULL;
    }

    str1=str0;
    do{
        token = pds_strsep(&str0, delim);
        if(token!=NULL)
        {
            //if(strlen(token)>0) 
            {
                dat=pds_cell_string_add(Tok,token);
                if(dat==PDS_WRONG)   fprintf(stderr,"ERROR: Adding values to the PdsCellString\n");
            }
            
        }
    }while(token!=NULL);

    free(str1);

    return Tok;
}


/** \fn int pds_strcicmp(const char *str0,const char *str1)
 *  \brief Esta función es similar a la función strcmp pero no distinguiendo 
 *  mayúsculas y minúsculas (Case Insensitive), donde si las cadenas son iguales
 *  la funcion retorna 0.
 *  \param[in] str0 Primera cadena a comparar.
 *  \param[in] str1 Segunda cadena a comparar.
 *  \return Retorna 0 si las cadenas son iguales (Case Insensitive).
 *  Retorna < 0 si str0 is less than str1.
 *  Retorna > 0 si str1 is less than str0.
 *  \ingroup PdsStringGroup
 */
int pds_strcicmp(const char *str0,const char *str1)
{
    int d;

    for (;; str0++, str1++) 
    {
        d = tolower(*str0) - tolower(*str1);
        if( (d != 0) || (*str0==0) )
            return d;
    }
}

/** \fn int pds_strlexcmp(const char *str0,const char *str1)
 *  \brief Esta función es similar a la función strcmp pero compara lexicograficamente
 *  las cadenas, donde si las cadenas son iguales
 *  la funcion retorna 0.
 *  \param[in] str0 Primera cadena a comparar.
 *  \param[in] str1 Segunda cadena a comparar.
 *  \return Retorna 0 si las cadenas son iguales (Case Insensitive).
 *  Retorna < 0 si str0 is less than str1.
 *  Retorna > 0 si str1 is less than str0.
 *  \ingroup PdsStringGroup
 */
int pds_strlexcmp(const char *str0,const char *str1)
{
    int d;
    int a0,a1;


    for (;; str0++, str1++) 
    {
        if( isdigit(*str0) && isdigit(*str1) )
        {
            a0=atoi(str0);
            a1=atoi(str1);
            //printf("[%d %d]\n",a0,a1);
            if(a0<a1)   return -1;
            if(a1<a0)   return +1;

        }
        else
        {
            //printf("<%c %c>\n",*str0,*str1);
            d = tolower(*str0) - tolower(*str1);
            if( (d != 0) || (*str0==0) )
                return d;
        }
    }

}



/** \fn int pds_string_multi_clean_beginning(char *str,const char *pat)
 *  \brief Esta función limpia los primeros caracteres de la cadena str que 
 *  contengan algun caracter en pat, la cadena no es realocada simplemente los datos son
 *  movidos de lugar.
 *  \param[in] str Cadena a limpiar.
 *  \param[in] pat Caracteres a limpiar.
 *  \return Retorna PDS_OK si las cadena fue limpiada correctamente o retorna PDS_WRONG si
 *  str o pat es igual a NULL;
 *  \ingroup PdsStringGroup
 */
int pds_string_multi_clean_beginning(char *str,const char *pat)
{
    int len,i,n,j;
    int ntrash;
    int found;

    if(str==NULL)   return PDS_WRONG;
    if(pat==NULL)   return PDS_WRONG;

    ntrash=0;
    len=strlen(str);
    n=strlen(pat);

    for(i=0;i<len;i++)
    {
        found=PDS_WRONG;
        for(j=0;j<n;j++)
        {
            if(str[i]==pat[j])   found=PDS_OK;
        }
        
        if(found==PDS_OK) ntrash++;
        else            break;
    }
    
    if(ntrash>0)
    {
        for(i=ntrash;i<len;i++)
        {
            str[i-ntrash]=str[i];
        }
        for(i=len-ntrash;i<len;i++) str[i]=0;
    }
    
    return PDS_OK;
}


/** \fn int pds_string_multi_clean_ending(char *str,const char *pat)
 *  \brief Esta función limpia los ultimos caracteres de la cadena str que 
 *  contengan algun caracter en pat, la cadena no es realocada simplemente los datos son
 *  cerados.
 *  \param[in] str Cadena a limpiar.
 *  \param[in] pat Caracteres a limpiar.
 *  \return Retorna PDS_OK si las cadena fue limpiada correctamente o retorna PDS_WRONG si
 *  str o pat es igual a NULL;
 *  \ingroup PdsStringGroup
 */
int pds_string_multi_clean_ending(char *str,const char *pat)
{
    int len,i,n,j;
    int found;

    if(str==NULL)   return PDS_WRONG;
    if(pat==NULL)   return PDS_WRONG;

    len=strlen(str);
    n=strlen(pat);

    for(i=len-1;i>=0;i--)
    {
        found=PDS_WRONG;
        for(j=0;j<n;j++)
        {
            if(str[i]==pat[j])   found=PDS_OK;
        }

        if(found==PDS_OK)   str[i]=0;
        else break;
    }
    
    return PDS_OK;
}


/** \fn int pds_string_clean_beginning(char *str,char c)
 *  \brief Esta función limpia los primeros caracteres de la cadena str que 
 *  contengan el caracter c, la cadena no es realocada simplemente los datos son
 *  movidos de lugar.
 *  \param[in] str Cadena a limpiar.
 *  \param[in] c Caracter a limpiar.
 *  \return Retorna PDS_OK si las cadena fue limpiada correctamente o retorna PDS_WRONG si
 *  str es igual a NULL;
 *  \ingroup PdsStringGroup
 */
int pds_string_clean_beginning(char *str,char c)
{
    int len,i;
    int ntrash;

    if(str==NULL)   return PDS_WRONG;

    ntrash=0;
    len=strlen(str);

    for(i=0;i<len;i++)
    {
        if(str[i]==c)   ntrash++;
        else break;
    }
    
    if(ntrash>0)
    {
        for(i=ntrash;i<len;i++)
        {
            str[i-ntrash]=str[i];
        }
        for(i=len-ntrash;i<len;i++) str[i]=0;
    }
    
    return PDS_OK;
}



/** \fn int pds_string_clean_ending(char *str,char c)
 *  \brief Esta función limpia los ultimos caracteres de la cadena str que 
 *  contengan el caracter c, la cadena no es realocada simplemente los datos son
 *  cerados.
 *  \param[in] str Cadena a limpiar.
 *  \param[in] c Caracter a limpiar.
 *  \return Retorna PDS_OK si las cadena fue limpiada correctamente o retorna PDS_WRONG si
 *  str es igual a NULL;
 *  \ingroup PdsStringGroup
 */
int pds_string_clean_ending(char *str,char c)
{
    int len,i;

    if(str==NULL)   return PDS_WRONG;

    len=strlen(str);

    for(i=len-1;i>=0;i--)
    {
        if(str[i]==c)   str[i]=0;
        else break;
    }
    
    return PDS_OK;
}


/** \fn char * pds_string_new_from_file(const char *path)
 *  \brief Esta función retorna una cadena de caracteres con el contenido de un archivo.
 *  \param[in] path El archivo a leer.
 *  \return Retorna una cadena de caracteres con el contenido de un archivo. O
 *  NULL en caso de error.
 *  \ingroup PdsStringGroup
 */
char * pds_string_new_from_file(const char *path)
{
    char *str=NULL;
    long N;
    FILE *fd=NULL;
    long i;
    int c;

    if(pds_exist_file(path)==PDS_WRONG) return NULL;

    N=pds_get_number_of_chars(path);

    fd=fopen(path,"r");
    if(fd==NULL)    return NULL;

    str=(char *)calloc(N+1,sizeof(char));
    if(str==NULL)
    {
        fclose(fd);
        return NULL;
    }

    i=0;
    while( 1 )
    {
        c=fgetc(fd);
        if(i >= N) break;
        if(c == EOF) break;
        str[i]=(char)c;
        i++;
    }

    fclose(fd);
    return str;
}


/** \fn char *pds_strcat_cell_string(const PdsCellString *celldata,const char *separator)
 *  \brief Esta función retorna una cadena de caracteres con el contenido concatenado
 *  usando como separador separator.
 *
 *  "str1"<b>separator</b>"str2"<b>separator</b> ... <b>separator</b>"strN"
 *
 *  \param[in] celldata La estructura a concatenar.
 *  \param[in] separator Separador usado entre celulas.
 *  \return Retorna una nueva cadena de caracteres con el contenido concatenado.
 *  O NULL en caso de error.
 *  \see pds_cell_string_joint()
 *  \see pds_cell_string_joint_with_end()
 *  \ingroup PdsStringGroup
 */
char *pds_strcat_cell_string(const PdsCellString *celldata,const char *separator)
{
    int N,Nc,Ns,i,Nnulls;
    char *str=NULL;
    char *ptr;
    
    if(celldata==NULL)  return NULL;
    //if(separator==NULL)  return NULL;

    Nc=pds_cell_string_get_ncell(celldata);
    if(Nc<=0)   return NULL;

    if(separator!=NULL) Ns=strlen(separator);
    else                Ns=0;

    N=1+(Nc-1)*Ns;
    for(i=0;i<Nc;i++)
    {
        if(celldata->data[i]!=NULL)
        N=N+strlen(celldata->data[i]);
    }

    str=(char*)calloc(N,sizeof(char));
    if(str==NULL)
    {
        fprintf(stderr,"ERROR while alloc memory.");
        return NULL;
    }
    
    Nnulls=0;
    while(celldata->data[Nnulls]==NULL)  Nnulls++;

    sprintf(str,"%s",celldata->data[Nnulls]);
    ptr=str+strlen(celldata->data[Nnulls]);

    for(i=Nnulls+1;i<Nc;i++)
    {
        if(celldata->data[i]!=NULL)
        {
            if(separator!=NULL) sprintf(ptr,"%s%s",separator,celldata->data[i]);
            else                sprintf(ptr,"%s",celldata->data[i]);
            ptr=ptr+Ns+strlen(celldata->data[i]);
        }
    }

    return str;

}

/** \fn int pds_string_get_number_of_patterns(const char *str,const char *pat)
 *  \brief Esta función cuenta la cantidad de coincidencias de la cadena pat
 *  en la cadena str.
 *
 *  Cuando encuentra una coincidencia en la posición p0 de str, continua la
 *  siguiente búsqueda en p0+strlen(path).
 *  \param[in] str Cadena donde se realizará la búsqueda.
 *  \param[in] pat Patrona buscar.
 *  \return Retorna el número de coincidencias o -1 en caso de error.
 *  \ingroup PdsStringGroup
 */
int pds_string_get_number_of_patterns(const char *str,const char *pat)
{
    long N;
    long L;
    int count,i;

    if(str==NULL)   return -1;
    if(pat==NULL)   return -1;

    L=strlen(pat);
    N=strlen(str);

    i=0;
    count=0;
    while(1)
    {
        if(i>=N)  break;

        if( strncmp((str+i),pat,L)==0 )
        {
            count++;
            i=i+L;
        }
        else    i++;
    }
    return count;
}


/** \fn char *pds_string_pattern_replacement(const char *str,const char *pat,const char *rep)
 *  \brief Esta función busca en una cadena str un patrón pat y los remplaza con 
 *  el contenido de la cadena rep, el resultado es retornado en una nueva cadena;
 *
 *  Cuando encuentra una coincidencia en la posición p0 de str, continua la
 *  siguiente búsqueda en p0+strlen(pat).
 *  \param[in] str Cadena donde se realizará la búsqueda.
 *  \param[in] pat Patrón a buscar.
 *  \param[in] rep Patrón a de remplazo.
 *  \return Retorna una nueva cadena con el resultado de remplazar pat por rep en str.
 *  \ingroup PdsStringGroup
 */
char *pds_string_pattern_replacement(const char *str,const char *pat,const char *rep)
{
    int n,Nstr,Npat,Nrep;
    int is,jn;
    char *newstr=NULL;

    if(str==NULL)   return NULL;
    if(pat==NULL)   return NULL;
    if(rep==NULL)   return NULL;

    n=pds_string_get_number_of_patterns(str,pat);
    Nstr=strlen(str);
    Npat=strlen(pat);
    Nrep=strlen(rep);

    newstr=(char *)calloc(Nstr-n*Npat+n*Nrep+1,sizeof(char));
    if(newstr==NULL)    return NULL;

    is=0;
    jn=0;
    while(1)
    {
        if(is>=Nstr)  break;

        if( strncmp(str+is,pat,Npat)==0 )
        {
            strncpy(newstr+jn,rep,Nrep);
            jn=jn+Nrep;
            is=is+Npat;
        }
        else
        {
            newstr[jn]=str[is];
            jn++;
            is++;
        }

    }
    return newstr;

}


/** \fn char *pds_string_get_match_rawdata(const char *str,const char *pat_open,const char *pat_close,char **str_ptr)
 *  \brief Esta función busca en una cadena str, los datos entre un patrón 
 *  pat_open y uno pat_close. Retorna los datos en una nueva cadena de texto.
 *
 *  Cuando encuentra una coincidencia la posición inmediata después de pat_close
 *  es cargada en str_ptr (solo si esa variable str_ptr existe).
 *
 *  Ejemplos : Busca la coincidencia y además es cargado str_ptr con la dirección
 *  inmediata a la coincidencia con pat_close.
\code{.c}
char str[]="hola <b> Fernando\n</b> como estas";
char *str_ptr=NULL;
char *rawdata=pds_string_get_match_rawdata(str,"<b>","</b>",&str_ptr);
\endcode
 *  Ejemplos : Busca la coincidencia en str.
\code{.c}
char str[]="hola <b> Fernando\n</b> como estas";
char *rawdata=pds_string_get_match_rawdata(str,"<b>","</b>",NULL);
\endcode
 *  \param[in] str Cadena donde se realizará la búsqueda.
 *  \param[in] pat_open Patrón de apertura.
 *  \param[in] pat_close Patrón de finalización.
 *  \param[out] str_ptr puntero en cuyo contenido se grabará la dirección de la
 *  dirección de memoria en str después de pat_close.
 *  Si se retorna un str_ptr==NULL significa que se llegó al final de la cadena.
 *  Este solo es una dirección de memoria de str y no debe ser liberada.
 *  \return Retorna una nueva cadena con el contenido entre pat_open y pat_close.
 *  \ingroup PdsStringGroup
 */
char *pds_string_get_match_rawdata(const char *str,const char *pat_open,const char *pat_close,char **str_ptr)
{
    long i;
    long offset_open;//inicio del patron open
    long offset_close;//inicio del patron close
    long N,Nopen,Nclose;

    char *str0=NULL;
    char *str_open=NULL;
    char *str_close=NULL;

    char *data=NULL;
    long data_length;


    if( (str==NULL)||(pat_open==NULL)||(pat_close==NULL) )
    {
        if( (*str_ptr)!=NULL )  (*str_ptr)=NULL;   
        return NULL;
    }

    str0=(char*)str;

    N      = strlen(str0);
    Nopen  = strlen(pat_open);
    Nclose = strlen(pat_close);

    //Buscamos el patrón de apertura
    i=0;
    while(1)
    {
        if( i>(N-Nopen-Nclose) )  break;

        if( strncmp(str0+i,pat_open,Nopen)==0 )
        {
            str_open=str0+i;
            offset_open=i;
            break;
        }

        i++;
    }

    if(str_open==NULL)
    {
        if( (*str_ptr)!=NULL )  (*str_ptr)=NULL;   
        return NULL;
    }

    //Buscamos el patrón de finalización
    i=offset_open+Nopen;

    while(1)
    {
        if( i>(N-Nclose) )  break;

        if( strncmp(str0+i,pat_close,Nclose)==0 )
        {

            str_close=str0+i;
            offset_close=i;
            break;
        }

        i++;
    }

    if(str_close==NULL)
    {
        if( (*str_ptr)!=NULL )  (*str_ptr)=NULL;   
        return NULL;
    }



    //Creando data
    data_length=offset_close-(offset_open+Nopen);
    
    data=(char*)calloc(data_length+1,sizeof(char));
    if(data==NULL)
    {
        if( (*str_ptr)!=NULL )  (*str_ptr)=NULL;   
        return NULL;
    }

    for(i=0;i<data_length;i++)
    data[i]=str[offset_open+Nopen+i];

    if( (offset_close+Nclose)<=N)
        *str_ptr=str0+(offset_close+Nclose);
    else
        *str_ptr=NULL;


    return data;
}


/** \fn char *pds_string_content_replacement( const char *str_in, const char *pat_open,const char *pat_close, char *(*func)(const char *))
 *  \brief Esta función busca en una cadena de texto str_in, el contenido
 *  entre un patrón pat_open y pat_close y lo remplaza  por el valor devuelto de 
 *  la función func, el resultado de todo esto es retornado en una nueva cadena 
 *  de texto.
 *
 *  La función func debe ser de la forma :
\code{.c}
char *func(const char* content);
\endcode
 *
\code{.c}
str_out=pds_string_content_replacement(str_in,"<b>","</b>",func);
\endcode
 *  \param[in] str_in Cadena de entrada donde se realizará la búsqueda.
 *  \param[in] pat_open Patrón inicial a buscar.
 *  \param[in] pat_close Patrón final a buscar.
 *  \param[in] func Es una función que se usará para remplazar el contenido 
 *  encontrado. Por ejemplo si entre pat_open y pat_close es encontrado content
\code{.c}
replace=func(content);
\endcode
 *  este será remplazado por el contenido de la cadena replace, luego la cadena 
 *  replace será liberada internamente con free(replace).
 *  Si la funcion func retorna NULL, la busqueda en str_in es detenida y
 *  se retorna NULL.
 *  \return Retorna una nueva cadena de texto si todo fue bien o NULL si no.
 *  \ingroup PdsStringGroup
 */
char *pds_string_content_replacement( const char *str_in,
                                     const char *pat_open,const char *pat_close,
                                     char *(*func)(const char *))
{
    char *content=NULL;
    char *rep=NULL;
    char *str_ptr=NULL;
    char *str_last=NULL;
    long Nclose,Ncontent,i,n;

    PdsCellString *Cout=NULL;
    char *str_out=NULL;

    if(str_in==NULL)    return NULL;
    if(pat_open==NULL)  return NULL;
    if(pat_close==NULL) return NULL;
    if((*func)==NULL)   return NULL;


    str_ptr=(char *)str_in;
    Nclose=strlen(pat_close);


    Cout=pds_cell_string_new(0);
    if(Cout==NULL)      return NULL;


    while(1)
    {
        str_last=str_ptr;

        content=pds_string_get_match_rawdata(str_ptr,pat_open,pat_close,&str_ptr);
        
        if(content!=NULL)
        {
            Ncontent=strlen(content);
            n=( (long)(str_ptr-str_last) )-Ncontent-Nclose;

            rep=calloc(n+1,sizeof(char));
            if(rep==NULL)
            {
                free(content);
                pds_cell_string_free(Cout);
                return NULL;
            }
            
            for(i=0;i<n;i++)    rep[i]=str_last[i];
            n=pds_cell_string_add (Cout,rep);
    

            free(rep);
            if(n==PDS_WRONG)
            {
                free(content);
                pds_cell_string_free(Cout);
                return NULL;
            }


            rep=(*func)(content);
            if(rep==NULL)
            {
                free(content);
                pds_cell_string_free(Cout);
                return NULL;
            }

            pds_cell_string_add (Cout,rep);
            pds_cell_string_add (Cout,pat_close);

          
            free(content);
            free(rep);
        }
        else
        {
            pds_cell_string_add (Cout,str_last);
            break;
        }

    }

    str_out=pds_cell_string_joint(Cout);
    pds_cell_string_free(Cout);

    return str_out;
}


