/*
 * pdsfilesfunc.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <pds/pdsfilesfunc.h>
#include <pds/pdscstring.h>
#include <pds/pdsstring.h>

#include <fcntl.h>           /* Definition of AT_* constants */
#include <unistd.h>
#include <dirent.h>
#include <pds/pdsfnmatch.h>

#include <sys/types.h>
#include <sys/stat.h>

#include <time.h>

#include <errno.h>

#define __BUFFER_SIZE__ 4096
#if defined(_WINDOWS_) || defined(_WIN32) || defined(WIN32)
    #include <windows.h>
    #include <tchar.h>
#endif


/** \fn PdsCellString *pds_get_arquives_new(const char *dirpath,ArqType Type,const char* pattern_include,const char* pattern_exclude)
 *  \brief Retorna los archivos dentro de un directorio (siguiendo algunos criterios).
 *
 *  Retorna una estructura PdsCellString con un arreglo de células de nombres de arquivo.
 *  A partir de aqui pueden ser leidos los siguientes con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] Type El tipo de arquivo a leer, pueden ser usados los valores :
 <ul>
    <li><b>TYPE_DIR</b> para directorios, </li>
    <li><b>TYPE_DIRTOP</b> para directorios incluyendo la raiz o </li>
    <li><b>TYPE_FILE</b>  para fiheros.</li> 
    <li>Cualquier otro valor es equivalente a usar TYPE_FILE.</li>
 </ul>
 *  \param[in] pattern_include Patrón de archivos a incluir solamente.
 *  \param[in] pattern_exclude Patrón de archivos a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  nombres de arquivo.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_arquives_new(const char *dirpath,ArqType Type,const char* pattern_include,const char* pattern_exclude)
{
    char *dir_path=NULL;
    int  search_type;
    char *PatternInc=NULL;
    char *PatternExc=NULL;
    struct stat s;

    DIR *dir=NULL;
    struct dirent *lsdir=NULL;
    char *TEMPORAL_STRING=NULL;
    PdsCellString *fullnames=NULL;

    if (dirpath==NULL)      return NULL;
    if (strlen(dirpath)==0) return NULL;


    dir_path=pds_get_realpath(dirpath);
    if (dir_path==NULL)      return NULL;    

    dir = opendir(dir_path);
    if (dir==NULL)
    {
        TEMPORAL_STRING=(char *)calloc(strlen(dir_path)+32,sizeof(char));
        sprintf(TEMPORAL_STRING,"ERROR opening directory %s",dir_path);
        perror(TEMPORAL_STRING);
        free(TEMPORAL_STRING);
        return NULL;
    }


    fullnames=pds_cell_string_new(1);
    if(fullnames==NULL) return NULL;

    if(Type==TYPE_DIR)          search_type=S_IFDIR;
    else if(Type==TYPE_DIRTOP)  
    {
        search_type=S_IFDIR;
        pds_cell_string_add(fullnames,dir_path);
    }
    else                        search_type=S_IFREG;


    if (pattern_include==NULL)
    {
        PatternInc=(char*)calloc(2,sizeof(char));
        PatternInc[0]='*';
    }
    else if (strlen(pattern_include)==0)
    {
        PatternInc=(char*)calloc(2,sizeof(char));
        PatternInc[0]='*';
    }
    else
    {
        PatternInc=(char*)calloc(strlen(pattern_include)+1,sizeof(char));
        sprintf(PatternInc,"%s",pattern_include);
    }


    if (pattern_exclude==NULL)
    {
        PatternExc=(char*)calloc(2,sizeof(char));
        PatternExc[0]=0;
    }
    else if (strlen(pattern_exclude)==0)
    {
        PatternExc=(char*)calloc(2,sizeof(char));
        PatternExc[0]=0;
    }
    else
    {
        PatternExc=(char*)calloc(strlen(pattern_exclude)+1,sizeof(char));
        sprintf(PatternExc,"%s",pattern_exclude);
    }


    while ( ( lsdir = readdir(dir) ) != NULL )
    {
        TEMPORAL_STRING=(char *)calloc(strlen(dir_path)+4+strlen(lsdir->d_name),sizeof(char));
        sprintf(TEMPORAL_STRING,"%s%c%s",dir_path,pds_filesep(),lsdir->d_name);
        stat(TEMPORAL_STRING, &s);
        //printf("<<%s>>::%x::%x::%x\n",TEMPORAL_STRING,s.st_mode,S_IFREG,S_IFDIR);

        if((s.st_mode & S_IFMT)==search_type)
        if( (strcmp(lsdir->d_name,".")!=0)&&(strcmp(lsdir->d_name,"..")!=0) )
        if(pds_fnmatch(PatternInc, lsdir->d_name, FNM_NOESCAPE)==0)
        if(pds_fnmatch(PatternExc, lsdir->d_name, FNM_NOESCAPE)!=0)
        {
            pds_cell_string_add(fullnames,TEMPORAL_STRING);
        }
        free(TEMPORAL_STRING);
    }

    closedir(dir);
    free(dir_path);
    free(PatternInc);
    free(PatternExc);

    if(pds_cell_string_get_last_empty(fullnames)==0)
    {
        pds_cell_string_destroy(&fullnames);
	    return NULL;
    }
    else
    {
	    return fullnames;
    }
}

/** \fn char *pds_get_homedir(void)
 *  \brief Retorna una nueva cadena de texto con la dirección del directorio de 
 *  usuario. HOME en gnu-linux y {HOMEDRIVE,HOMEPATH} en Windows.
 *  \return Retorna el directorio de usuario o NULL en caso de problemas en la
 *  reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_homedir(void)
{
    char homedir[__BUFFER_SIZE__];
    memset(homedir, 0,__BUFFER_SIZE__);

    #if defined(_WINDOWS_) || defined(_WIN32) || defined(WIN32)
    snprintf(homedir, __BUFFER_SIZE__, "%s%s", getenv("HOMEDRIVE"), getenv("HOMEPATH"));
    #elif defined(__linux__)
    snprintf(homedir, __BUFFER_SIZE__, "%s", getenv("HOME"));
    #endif
    return strdup(homedir);
}

/** \fn char pds_filesep(void)
 *  \brief Retorna um caracter con el separador de archivo para el sistema.
 *  \return Retorna um caracter con el separador de archivo para el sistema.
 *  En caso de error de dedución, retorna '/' por defecto.
 *  \ingroup PdsFilesFuncGroup
 */
char pds_filesep(void)
{
    char c='/';

    char *string_data=NULL;

    string_data=(char*)getcwd(0,0);

    if(string_data==NULL)
    {
        perror("ERROR deducing the file separator assuming: /");
        return c;
    }

    if (strlen(string_data)>=2)
    if ( ((string_data[0]>=65)&&(string_data[0]<=90)) || ((string_data[0]>=97)&&(string_data[0]<=122)) )
    if (string_data[1]==':')
    {   
        c='\\';
    }
    
    free(string_data);
    return c;
}



/** \fn PdsCellString *pds_get_all_subdirs_new(const char *dirpath,const char* pattern_include,const char* pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células con 
 *  nombres de subdirectorios incluyendo la raiz y excluyendo los simbolicos.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células com 
 *  nombres de subdirectorios incluyendo la raiz y excluyendo los simbólicos.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_all_subdirs_new(const char *dirpath,const char* pattern_include,const char* pattern_exclude)
{
    int dat;
    char *dir_path=NULL;

    PdsCellString *C0=NULL;
    PdsCellString *Ca=NULL;
    PdsCellString *Cb=NULL;

    if(dirpath==NULL)    return NULL;

    C0=pds_cell_string_new(0);
    if(C0==NULL)    return NULL;

    Ca=pds_cell_string_new(1);
    if(Ca==NULL)    {   pds_cell_string_free(C0);
                        return NULL;}


    dir_path=pds_get_realpath(dirpath);
    if(dir_path==NULL)    return NULL;

    dat=pds_cell_string_add(Ca,dir_path);
    if(dat==PDS_WRONG)  {   free(dir_path);
                        pds_cell_string_free(Ca);
                        pds_cell_string_free(C0);
                        return NULL;}


    dat=pds_cell_string_add_cell_string(C0,Ca);
    if(dat==PDS_WRONG)  {   free(dir_path);
                        pds_cell_string_free(Ca);
                        pds_cell_string_free(C0);
                        return NULL;}
    
    do{

        Cb=pds_get_subdirs_new(Ca,pattern_include,pattern_exclude);
        //printf("directorios:\n");
        //if(Ca!=NULL)  pds_cell_string_all_printf(Cb);
        //printf("____________\n");

        dat=pds_cell_string_add_cell_string(C0,Cb);
        if(dat==PDS_WRONG)  {   free(dir_path);
                            pds_cell_string_free(Ca);
                            pds_cell_string_free(Cb);
                            pds_cell_string_free(C0);
                            return NULL;}
        
        pds_cell_string_free(Ca);
        Ca=Cb;

    }while(Cb!=NULL);

    free(dir_path);    
    pds_cell_string_free(Ca);
    pds_cell_string_free(Cb);

    return C0;
}

/** \fn PdsCellString *pds_get_subdirs_new(const PdsCellString *C0,const char* pattern_include,const char* pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células, con  
 *  los nombres de los subdirectorios del siguiente nivel de los directorios del 
 *  arreglo de células C0 y excluyendo los simbólicos. 
 *  El arreglo de células de salida no contiene los directorios de entrada dentro de C0.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] C0 Un arreglo de células con los directorios a buscar.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  los nombres de los subdirectorios del siguiente nivel (solamente un nivel)
 *  y excluyendo los simbólicos O NULL en caso de error.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_subdirs_new(const PdsCellString *C0,const char* pattern_include,const char* pattern_exclude)
{
    int i,dat;
    PdsCellString *Cout=NULL;
    PdsCellString *Ctmp=NULL;

    if(C0==NULL)    return NULL;

    Cout=pds_cell_string_new(0);
    if(Cout==NULL)    return NULL;


    for(i=0;i<C0->Ncell;i++)
    {
        Ctmp=pds_get_arquives_new(C0->data[i],TYPE_DIR,pattern_include,pattern_exclude);
        dat=pds_cell_string_add_cell_string(Cout,Ctmp);
        if(dat==PDS_WRONG)
        {
            pds_cell_string_free(Ctmp);
            pds_cell_string_free(Cout);
            return NULL;
        }
    }

    pds_cell_string_free(Ctmp);

    if(pds_cell_string_get_last_empty(Cout)==0)
    {
        pds_cell_string_free(Cout);
        return NULL;
    }

    return Cout;
}




/** \fn PdsCellString *pds_get_all_files_new(const char *dirpath,   const char* pattern_include,const char* pattern_exclude,const char* file_pattern_include,const char* file_pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células con los 
 *  nombres de archivos. No son buscados archivos en los directorios simbólicos.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \param[in] file_pattern_include Patrón de archivos a incluir solamente.
 *  \param[in] file_pattern_exclude Patrón de archivos a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  los nombres de todos los archivos.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_all_files_new(const char *dirpath,   const char* pattern_include,
                                                            const char* pattern_exclude,
                                                            const char* file_pattern_include,
                                                            const char* file_pattern_exclude)
{
    PdsCellString *DIRECTORY=NULL;
    PdsCellString *TmpFileCell=NULL;
    PdsCellString *FileCell=NULL;
    int i,dat;

    if(dirpath==NULL)   return NULL;
    
    FileCell=pds_cell_string_new(0);
    if(FileCell==NULL)   return NULL;    

    DIRECTORY=pds_get_all_subdirs_new(dirpath,pattern_include,pattern_exclude);
    if(DIRECTORY==NULL)  { pds_cell_string_free(FileCell); return NULL;}

    for(i=0;i<DIRECTORY->Ncell;i++)
    {
        TmpFileCell=pds_get_arquives_new(   DIRECTORY->data[i],TYPE_FILE,
                                            file_pattern_include,
                                            file_pattern_exclude);
        dat=pds_cell_string_add_cell_string(FileCell,TmpFileCell);
        if(dat==PDS_WRONG)
        {
            pds_cell_string_free(FileCell);
            pds_cell_string_free(TmpFileCell);
            pds_cell_string_free(DIRECTORY);
            return NULL;
        }
    }


    pds_cell_string_free(TmpFileCell);
    pds_cell_string_free(DIRECTORY);

    if(pds_cell_string_get_last_empty(FileCell)==0)
    {
        pds_cell_string_free(FileCell);
        return NULL;
    }

    return FileCell;
}

/** \fn int pds_cat(const char* filepath)
 *  \brief Imprime en pantalla el contenido de un archivo de texto apuntado por
 *   filepath.
 *
 *  \param[in] filepath Ruta del fichero a analizar. 
 *  \return Retorna PDS_OK si todo fue bien o PDS_WRONG en el caso contrario.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_cat(const char* filepath)
{
    FILE *fd=NULL;
    char c;

    fd=fopen(filepath,"r");
    if(fd==NULL)    return PDS_WRONG;

    while ((c = getc(fd)) != EOF)   putchar(c);

    fclose(fd);

    return PDS_OK;
}

/** \fn char *pds_get_realpath(const char* filepath)
 *  \brief Expande todos los enlaces simbolicos y resuleve referencia a /./,
 *  /../ y caracteres extras '/'  en filepath, asi es producido un pathname 
 *  canonizado y absoluto. El resultado es cargado en una cadena nueva, 
 *  retornada por la funcion. El path resultante no tiene enlaces simbolicos, 
 *  /./ o /../ componentes. No expande '~/'.
 *
 *  Si la ruta de fichero es: ./Makefile
 *                            src/../Makefile
 *                            src/../////Makefile
 *                            Makefile 
 *  La función retorna:       /path_of_file/Makefile
 *  \param[in] filepath Ruta del fichero a analizar. Este fichero tiene que existir,
 *  caso contrario la función retornará NULL.
 *  \return Retorna una nueva cadena de texto con el path absoluto y canonizado, 
 *  o NULL en caso de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_realpath(const char* filepath)
{
    char *file_path=NULL;

    #if defined(_WINDOWS_) || defined(_WIN32) || defined(WIN32)
        int i;
        TCHAR  buffer[__BUFFER_SIZE__]=TEXT(""); 
        DWORD retval = GetFullPathName(filepath,__BUFFER_SIZE__,buffer,NULL);
        file_path=(char*)calloc(strlen(buffer)+1,sizeof(char));
        for(i=0;i<strlen(buffer);i++)
        {
            wchar_t wc = buffer[i];
            char c = (char)wc;
            file_path[i] = c;
        }
    #elif defined(__linux__)
        file_path=realpath(filepath,NULL);
    #else
        file_path=realpath(filepath,NULL);
        
    #endif

    return file_path;

}

/** \fn char *pds_get_basename(const char * file_path)
 *  \brief Obtén el nombre base desde una ruta de fichero.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       filename.c
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con el nombre base, o NULL en caso 
 *  de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_basename(const char * file_path)
{
    char *ptr = NULL;

    ptr=strrchr(file_path,pds_filesep());

    if(ptr==NULL)   return strdup(file_path);
    else            return strdup(ptr + 1);
}


/** \fn char *pds_get_filename_extension(const char * file_path)
 *  \brief Obtén la extención de fichero desde una ruta de fichero.
 *  Esta función tendrá problemas con archivos ocultos ".abcd", pues
 *  entenderá "abcd" como una extención de fihero.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       c
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con la extención de fichero, o 
 *  NULL en caso de error de reserva de memoria o fihero sin extención.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_filename_extension(const char * file_path)
{
    char *ptr = NULL;
    char *ptrout = NULL;
    char *base_name=NULL;

    base_name=pds_get_basename(file_path);
    if(base_name==NULL)  return NULL;

    ptr=strrchr(base_name,'.');

    if(ptr==NULL)
    {
        free(base_name); 
        return NULL;
    }
    else
    {
        ptrout=strdup(ptr + 1);
        free(base_name);
        return ptrout;
    }
}


/** \fn char *pds_get_filename(const char * file_path)
 *  \brief Obtén el nombre de fichero desde una ruta de fichero.
 *  Esta función tendrá problemas con archivos ocultos ".abcd", pues
 *  retornará una cadena vacía.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       filename
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con el nombre de fichero, o NULL 
 *  en caso de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_filename(const char * file_path)
{
    char *base_name=NULL;
    char *file_name=NULL;
    char *ptr=NULL;
    int file_name_size;
    int i;

    base_name=pds_get_basename(file_path);
    if(base_name==NULL)  return NULL;

    ptr=strrchr(base_name,'.');

    if(ptr==NULL)
    {
        file_name=strdup(base_name);
    }
    else
    {
        file_name_size=(int)(ptr-base_name);
        file_name=(char*) calloc(file_name_size+1,sizeof(char));
        if(file_name!=NULL)
        {
            for(i=0;i<file_name_size;i++) file_name[i]=base_name[i];
        }
    }

    free(base_name);
    return file_name;
}


/** \fn char *pds_get_dirname(const char * file_path)
 *  \brief Obtén el directorio madre desde una ruta de fichero.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       /path_to_file
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \return Retorna una nueva cadena de texto con el directorio madre, o 
 *  NULL en caso de error de reserva de memoria o fihero sin directorio madre.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_dirname(const char * file_path)
{
    char *dir_path=NULL;
    char *ptr=NULL;
    int dir_path_size;
    int i;

    ptr=strrchr(file_path,pds_filesep());

    if(ptr==NULL)
    {
        dir_path=NULL;
    }
    else
    {
        dir_path_size=(int)(ptr-file_path);
        dir_path=(char*) calloc(dir_path_size+1,sizeof(char));
        if(dir_path!=NULL)
        {
            for(i=0;i<dir_path_size;i++) dir_path[i]=file_path[i];
        }
    }

    return dir_path; 
}


/** \fn char *pds_get_filepath_with_extension(const char *file_path,const char *extension)
 *  \brief Obtén un filepath con el mismo filename pero con la extención cambiada 
 *  a extension.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       /path_to_file/filename.[extension]
 *  \param[in] file_path Ruta del fichero a analizar.
 *  \param[in] extension Extención a usar.
 *  \return Retorna una nueva cadena de texto con el filepath nuevo, o 
 *  NULL en caso de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_filepath_with_extension(const char *file_path,const char *extension)
{
    char *cad=NULL;
    char *dirname=NULL;
    char *filename=NULL;

    dirname=pds_get_dirname(file_path);

    filename=pds_get_filename(file_path);
    if(filename==NULL)
    {
        free(dirname);
        return NULL;
    }

    if(dirname==NULL)
        cad=pds_sprintf("%s.%s",filename,extension);
    else
        cad=pds_sprintf("%s%c%s.%s",dirname,pds_filesep(),filename,extension);

    free(dirname);
    free(filename);
    return cad;
}

/** \fn char *pds_get_temporal_filepath(const char *filepath,const char *groupname)
 *  \brief Obtén un nombre de archivo con la misma extención y directorio pero con nombre de
 *  archivo diferente y aleatoriamente diseñado.
 *
 *  Si la ruta de fichero es: /path_to_file/filename.c
 *  La función retorna:       /path_to_file/filename_[groupname][clock()][rand()].c
 *  
 *  \param[in] filepath Ruta del fichero a analizar. Esta ruta no precisa existir.
 *  \param[in] groupname nombre que se agregará al nombre de archivo retornado.
 *  \return Retorna una nueva cadena de texto con el filepath nuevo, o 
 *  NULL en caso de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_temporal_filepath(const char *filepath,const char *groupname)
{
    char *cad=NULL;
    char *ext=NULL;
    char *filename=NULL;
    char *directory=NULL;

    ext=pds_get_filename_extension(filepath);
    if(ext==NULL)  return NULL;

    filename=pds_get_filename(filepath);
    if(filename==NULL)  
    {
        free(ext);
        return NULL;
    }

    directory=pds_get_dirname(filepath);


    if(directory==NULL)
        if(groupname==NULL)
            cad=pds_sprintf("%s_%ld_%d.%s",filename,clock(),rand(),ext);
        else
            cad=pds_sprintf("%s_%s%ld_%d.%s",filename,groupname,clock(),rand(),ext);
    else
        if(groupname==NULL)
            cad=pds_sprintf("%s%c%s_%ld_%d.%s",directory,pds_filesep(),filename,clock(),rand(),ext);
        else
            cad=pds_sprintf("%s%c%s_%s%ld_%d.%s",directory,pds_filesep(),filename,groupname,clock(),rand(),ext);

    free(ext);
    free(filename);
    free(directory);
    return cad;
}



/** \fn char* pds_get_absolute_dirname(void)
 *  \brief Obtén el directorio base del programa que invoca esta función.
 *
 *  Si la ruta del programa es: /path_to_file/programfilename
 *  La función retorna:       /path_to_file
 *  \return Retorna una nueva cadena de texto con el nombre del directorio base, 
 *  o NULL en caso de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char* pds_get_absolute_dirname(void)
{
    char exepath[__BUFFER_SIZE__];
    memset(exepath, 0,__BUFFER_SIZE__);

    #if defined(_WINDOWS_) || defined(_WIN32) || defined(WIN32)
        int i;
        TCHAR buffer[__BUFFER_SIZE__];
        GetModuleFileName(NULL,buffer, __BUFFER_SIZE__) ;
        for(i = 0; buffer[i]; ++i) 
        {
          wchar_t wc = buffer[i];
          char c = (char)wc;
          exepath[i] = c;
        }
    #elif defined(__linux__) 
        int ID=readlink("/proc/self/exe", exepath,__BUFFER_SIZE__-1);
        ID++;
    #else
        printf("The function pds_get_absolute_dirname() is not suported in this operating system.")
        exepath[0]='.'
    #endif

    char *CMD=(char *)calloc(strlen(exepath)+1,sizeof(char));
    sprintf(CMD,"%s",pds_get_dirname(exepath));

    return CMD;
}


/** \fn char* pds_get_absolute_programpath(void)
 *  \brief Obtén la ruta del programa que invoca esta función.
 *
 *  Si la ruta del programa es: /path_to_file/programfilename
 *  La función retorna:       /path_to_file/programfilename
 *  \return Retorna una nueva cadena de texto con el nombre de la ruta base, 
 *  o NULL en caso de error de reserva de memoria.
 *  \ingroup PdsFilesFuncGroup
 */
char* pds_get_absolute_programpath(void)
{
    char exepath[__BUFFER_SIZE__];
    memset(exepath, 0,__BUFFER_SIZE__);

    #if defined(_WINDOWS_) || defined(_WIN32) || defined(WIN32)
        int i;
        TCHAR buffer[__BUFFER_SIZE__];
        GetModuleFileName(NULL,buffer, __BUFFER_SIZE__) ;
        for(i = 0; buffer[i]; ++i) 
        {
          wchar_t wc = buffer[i];
          char c = (char)wc;
          exepath[i] = c;
        }
    #elif defined(__linux__)
        int ID=readlink("/proc/self/exe", exepath,__BUFFER_SIZE__-1);
        ID++;
    #else
        printf("The function pds_get_absolute_programpath() is not suported in this operating system.")
        exepath[0]='.'
    #endif

    char *CMD=(char *)calloc(strlen(exepath)+1,sizeof(char));
    sprintf(CMD,"%s",exepath);

    return CMD;
}


/** \fn PdsCellString *pds_get_all_filename_extensions_in_cell(const PdsCellString *FileCell)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células con las 
 *  extenciones de archivos en el PdsCellString. 
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] FileCell Celula donde se buscará etensiones.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  los nombres de todas las extenciones de archivos. En caso de no existir extensiones
 *  retorna una estructura vacia, en caso de error retorna NULL. Si FileCell==NULL
 *  se retorna NULL.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_all_filename_extensions_in_cell(const PdsCellString *FileCell)
{
    PdsCellString *ExtCell=NULL;

    char *strPtr=NULL;
    int i,dat;

    if(FileCell==NULL)  return NULL;

    
    ExtCell=pds_cell_string_new(0);
    if(ExtCell==NULL)   {return NULL;}

    for(i=0;i<FileCell->Ncell;i++)
    {
        strPtr=pds_get_filename_extension(FileCell->data[i]);
        
        if(strPtr!=NULL)
        if(strlen(strPtr)>0)
        if(pds_cell_string_find(ExtCell,strPtr)<0)
        {
            dat=pds_cell_string_add(ExtCell,strPtr);
            if(dat==PDS_WRONG)
            {
                pds_cell_string_free(ExtCell);
                return NULL;
            }
        }
    }

    return ExtCell;
}

/** \fn PdsCellString *pds_get_all_filename_extensions(const char *dirpath,   const char* pattern_include,const char* pattern_exclude,const char* file_pattern_include,const char* file_pattern_exclude)
 *  \brief Retorna una estructura PdsCellString con un arreglo de células con las 
 *  extenciones de archivos en el directorio. No son buscados archivos en los directorios simbólicos.
 *  A partir de aquí pueden ser leidos los datos con la función pds_cell_string_read().
 *  \param[in] dirpath Camino del directorio en consulta.
 *  \param[in] pattern_include Patrón de directorios a incluir solamente.
 *  \param[in] pattern_exclude Patrón de directorios a excluir.
 *  \param[in] file_pattern_include Patrón de archivos a incluir solamente.
 *  \param[in] file_pattern_exclude Patrón de archivos a excluir.
 *  \return Retorna una estructura PdsCellString con un arreglo de células con 
 *  los nombres de todas las extenciones de archivos. En caso de error retorna NULL.
 *  Si dirpath==NULL se retorna NULL. En caso de no existir extensiones
 *  retorna una estructura vacia
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_all_filename_extensions(const char *dirpath,   const char* pattern_include,
                                                            const char* pattern_exclude,
                                                            const char* file_pattern_include,
                                                            const char* file_pattern_exclude)
{
    PdsCellString *ExtCell=NULL;
    PdsCellString *FileCell=NULL;

    if(dirpath==NULL)   return NULL;

    FileCell=pds_get_all_files_new(dirpath,pattern_include,pattern_exclude,file_pattern_include,file_pattern_exclude);
    if(FileCell==NULL)  return NULL;

    ExtCell=pds_get_all_filename_extensions_in_cell(FileCell);
    if(ExtCell==NULL)  return NULL;

    pds_cell_string_free(FileCell);

    return ExtCell;
}


/** \fn char *pds_fgets(FILE *fd)
 *  \brief Retorna una linea leida desde un desriptor de fichero de texto.
 *
 *  \param[in] fd Descriptor de fichero de texto a analizar.
 *  \return Retorna una nueva cadena de texto con una copia de la linea leida 
 *  incluyendo los caracteres de control finales exepto EOF.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_fgets(FILE *fd)
{
    char *line=NULL;
    int i;
    fpos_t init;
    int count,N;
    char c;

    if(fd==NULL)    return NULL;

    fgetpos(fd,&init);

    count=0;
    c=0;
    while( (c!=10)&&(c!=EOF) )  /* \n */
    {
        c=fgetc(fd);
        if(c!=EOF) count++;
    }

    if(c!=EOF)
    {
        c=fgetc(fd);
        /* \r */
        if(c==13) count++;
    }
    fsetpos(fd,&init);

    

    N=count;


    line=(char*)calloc(N+1,sizeof(char));
    if (line==NULL) return NULL;

    for(i=0;i<N;i++)
    {    
        line[i]=fgetc(fd);
    }

    if(c==EOF)  
    {
        fseek( fd, 0, SEEK_END );
        fgetc(fd);
    }

    return line;
}


/** \fn char *pds_read_line(FILE *fd)
 *  \brief Retorna una linea leida desde un desriptor de fichero de texto.
 *
 *  \param[in] fd Descriptor de fichero de texto a analizar.
 *  \return Retorna una nueva cadena de texto con una copia de la linea leida 
 *  descartando salto de linea '\\n'.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_read_line(FILE *fd)
{
    char *data=NULL;
    char *tmp=NULL;
    int i,N;

    if(fd==NULL)    return NULL;

    tmp=pds_fgets(fd);
    if(tmp==NULL)   return NULL;

    N=strlen(tmp);

    for(i=0;i<strlen(tmp);i++)
    {
        if(tmp[i]==10) /* \n */
        {
            N=i;
            break;
        }
    }

    data=(char*)calloc(N+1,sizeof(char));
    if(data==NULL)
    {
        free(tmp);
        return NULL;
    }
    
    for(i=0;i<N;i++)    data[i]=tmp[i];

    free(tmp);
    return data;
}

/** \fn char *pds_get_filedata_string(const char *filepath,const char* pattern, const char* reject,const char* default_value)
 *  \brief La funcion analiza linea por linea el archivo y retorna
 *  el texto despues del primer patrón pattern encontrado.
 *  <ul>
 *  <li>Es desconsiderado al final de cada linea el salto de linea. </li>
 *  <li>Son desconsiderados al inicio de cada linea los espacios y tabuladores. </li>
 *  <li>Son desconsiderados al final del dato los espacios y tabuladores. </li>
 *  <li>Son desconsideradas al inicio del dato qualquier caracter en reject. </li>
 *  </ul>
 *
 *  Por ejemplo para un archivo mydata.ini con
 * \code{.c}
 *  MY_FIRST_DATA  = 1000
 *  MY_SECOND_DATA = Cadena de texto
 *  MY_THIRD_DATA  = TRUE
 *  MY_FOURTH_DATA = dat1, dat2  , dat3
 * \endcode
 *
 *  Para leer la cadena "Cadena de texto" usaremos:
 * \code{.c}
 * DATA=pds_get_filedata_string("mydata.ini","MY_SECOND_DATA","= ","valor por defecto");
 * \endcode
 *
 *  \param[in] filepath Fichero de texto a analizar. Si filepath==NULL se retorna NULL.
 *  \param[in] pattern Patrón a buscar. Si pattern==NULL se retorna NULL.
 *  \param[in] reject Caracteres a eliminar al inicio del dato encontrado.
 *  Si reject==NULL o vacio, no se elimina ningun caracter al inicio del dato encontrado.
 *  Si la variable default_value es usada entonces reject no es aplicado sobre el dato retornado.
 *  \param[in] default_value Es el valor por defecto que será retornado si pattern
 *  no es encontrado, o si el valor encontrado es una cadena vacia.
 *  Si default_value==NULL y no es encontrado ningun dato o este esta vacio entonces
 *  NULL es retornando.
 *  \return Retorna una nueva cadena de texto con el dato encontrado,
 *  o NULL si se encontró algun error.
 *  \ingroup PdsFilesFuncGroup
 */
char *pds_get_filedata_string(const char *filepath,const char* pattern, const char* reject,const char* default_value)
{
    FILE *fd=NULL;
    char *LINE=NULL;
    char *DATA=NULL;
    int N,Nl,i;
    char *PATTERN=NULL;

    if(filepath==NULL)
    {
        //fprintf(stdout,"WARNING(pds_get_filedata_string): First parameter equal to NULL.\n")
        return NULL;
    }
    if(pattern ==NULL)
    {
        //fprintf(stdout,"WARNING(pds_get_filedata_string): Second parameter equal to NULL.\n")
        return NULL;
    }


    PATTERN=(char*)calloc(strlen(pattern)+1,sizeof(char));
    if(PATTERN==NULL)   return NULL;
    strcpy(PATTERN,pattern);
    pds_string_multi_clean_beginning(PATTERN,"\t ");
    pds_string_multi_clean_ending (PATTERN,"\t ");

    N=strlen(PATTERN);
    

    fd=fopen(filepath,"r");
    if(fd==NULL)
    {
        fprintf(stderr,"ERROR opening the file:%s\n",filepath);
        return NULL;
    }


    while(feof(fd)==0)
    {
        free(LINE);
        LINE=pds_read_line(fd);
        if(LINE!=NULL)
        {

            pds_string_multi_clean_beginning (LINE,"\t ");

            Nl=strlen(LINE);
            if(Nl>=N)
            if(strncmp(PATTERN,LINE,N)==0)
            {
                DATA=(char*)calloc(Nl-N+1,sizeof(char));
                if(DATA==NULL)
                {
                    fprintf(stderr,"ERROR while alloc memory.\n");
                    fclose(fd);
                    free(PATTERN);
                    free(LINE);
                    return NULL;
                }
                for(i=N;i<Nl;i++)   DATA[i-N]=LINE[i];

                break;
            }
        }
    }

    if(DATA==NULL)
    {
        if(default_value!=NULL)
        {
            DATA=(char*)calloc(strlen(default_value)+1,sizeof(char));
            if(DATA!=NULL)
            strcpy(DATA,default_value);
        }
    }
    else
    {
        if(reject!=NULL)
        pds_string_multi_clean_beginning (DATA,reject);
        pds_string_multi_clean_ending (DATA,"\t ");

        if(strlen(DATA)==0)
        {
            free(DATA);DATA=NULL;

            if(default_value!=NULL)
            {
                DATA=(char*)calloc(strlen(default_value)+1,sizeof(char));
                if(DATA!=NULL)
                strcpy(DATA,default_value);
            }
        }

    }

    fclose(fd);
    free(LINE);
    free(PATTERN);

    return DATA;
}


/** \fn int pds_get_filedata_int(const char *filepath,const char* pattern, const char* reject,int default_value)
 *  \brief La funcion analiza linea por linea el archivo y retorna
 *  el valor entero despues del primer patrón pattern encontrado.
 *  <ul>
 *  <li>Es desconsiderado al final de cada linea el salto de linea. </li>
 *  <li>Son desconsiderados al inicio de cada linea los espacios y tabuladores. </li>
 *  <li>Son desconsiderados al final del dato los espacios y tabuladores. </li>
 *  <li>Son desconsideradas al inicio del dato qualquier caracter en reject. </li>
 *  </ul>
 *
 *  Por ejemplo para un archivo mydata.ini con
 * \code{.c}
 *  MY_FIRST_DATA  = 1000
 *  MY_SECOND_DATA = Cadena de texto
 *  MY_THIRD_DATA  = TRUE
 *  MY_FOURTH_DATA = dat1, dat2  , dat3
 * \endcode
 *
 *  Para leer el valor 1000 usaremos:
 * \code{.c}
 * DATA=pds_get_filedata_int("mydata.ini","MY_FIRST_DATA","= ",-1);
 * \endcode
 *
 *  \param[in] filepath Fichero de texto a analizar. Si filepath==NULL se retorna default_value.
 *  \param[in] pattern Patrón a buscar. Si pattern==NULL se retorna default_value.
 *  \param[in] reject Caracteres a eliminar al inicio del dato encontrado.
 *  Si reject==NULL o vacio, no se elimina ningun caracter al inicio del dato encontrado.
 *  \param[in] default_value Es el valor por defecto que será retornado si pattern
 *  no es encontrado, o si el valor encontrado es una cadena vacia o en caso de algun error.
 *  \return Retorna el dato encontrado, o default_value si se encontró algun error.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_get_filedata_int(const char *filepath,const char* pattern, const char* reject,int default_value)
{
    char *str=NULL;
    int d;
    char *buffer=NULL;

    if(filepath==NULL)
    {
        fprintf(stdout,"WARNING(pds_get_filedata_int): First parameter equal to NULL. Returning 0.\n");
        return default_value;
    }
    if(pattern ==NULL)
    {
        fprintf(stdout,"WARNING(pds_get_filedata_int): Second parameter equal to NULL. Returning 0.\n");
        return default_value;
    }
    
    buffer=pds_itoa(default_value,10);
    if(buffer==NULL)    
    {
        fprintf(stdout,"WARNING(pds_get_filedata_int): Memory alloc problem. Returning 0.\n");
        return default_value;
    }

    str=pds_get_filedata_string(filepath,pattern,reject,buffer);
    free(buffer);

    if(str==NULL)
    {
        fprintf(stdout,"WARNING(pds_get_filedata_int): Searching unsuccessful. Returning 0.\n");
        return default_value;
    }
    
    d=atoi(str);

    free(str);

    return d;
}

/** \fn int pds_get_filedata_strcmp(const char *filepath,const char* pattern,const char* reject,const char* dat)
 *  \brief La funcion analiza linea por linea el archivo y retorna
 *  PDS_OK si el valor despues de el primer patrón pattern encontrado es dat.
 *  <ul>
 *  <li>Es desconsiderado al final de cada linea el salto de linea. </li>
 *  <li>Son desconsiderados al inicio de cada linea los espacios y tabuladores. </li>
 *  <li>Son desconsiderados al final del dato los espacios y tabuladores. </li>
 *  <li>Son desconsideradas al inicio del dato qualquier caracter en reject. </li>
 *  </ul>
 *
 *  Por ejemplo para un archivo mydata.ini con
 * \code{.c}
 *  MY_FIRST_DATA  = 1000
 *  MY_SECOND_DATA = Cadena de texto
 *  MY_THIRD_DATA  = TRUE
 *  MY_FOURTH_DATA = dat1, dat2  , dat3
 * \endcode
 *
 *  Para comparar el valor TRUE usaremos:
 * \code{.c}
 * d=pds_get_filedata_strcmp("mydata.ini","MY_THIRD_DATA","= ",TRUE);
 * \endcode
 *
 *  \param[in] filepath Fichero de texto a analizar. Si filepath==NULL se retorna PDS_WRONG.
 *  \param[in] pattern Patrón a buscar. Si pattern==NULL se retorna PDS_WRONG.
 *  \param[in] reject Caracteres a eliminar al inicio del dato encontrado.
 *  Si reject==NULL o vacio, no se elimina ningun caracter al inicio del dato encontrado.
 *  \param[in] dat Es dato a comparar. Si dat==NULL se retornará PDS_WRONG.
 *  no es encontrado, o si el valor encontrado es una cadena vacia o en caso de algun error.
 *  \return Retorna PDS_OK si el valor dat es encontrado despues del primer 
 *  patrón pattern. En caso de error es retornado PDS_WRONG.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_get_filedata_strcmp(const char *filepath,const char* pattern,const char* reject,const char* dat)
{
    char *str=NULL;
    int d;


    if(filepath==NULL)
    {
        fprintf(stdout,"WARNING(pds_get_filedata_strcmp): First parameter equal to NULL. Returning PDS_WRONG.\n");
        return PDS_WRONG;
    }
    if(pattern ==NULL)
    {
        fprintf(stdout,"WARNING(pds_get_filedata_strcmp): Second parameter equal to NULL. Returning PDS_WRONG.\n");
        return PDS_WRONG;
    }
    if(dat     ==NULL)
    {
        fprintf(stdout,"WARNING(pds_get_filedata_strcmp): Fourth parameter equal to NULL. Returning PDS_WRONG.\n");
        return PDS_WRONG;
    }

    str=pds_get_filedata_string(filepath,pattern,reject,"");
    if(str==NULL)
    {
        //fprintf(stdout,"WARNING(pds_get_filedata_int): Searching unsuccessful. Returning PDS_WRONG.\n");
        return PDS_WRONG;
    }

    if(strcmp(dat,str)==0)  d=PDS_OK;
    else                    d=PDS_WRONG;

    free(str);

    return d;
}

/** \fn PdsCellString *pds_get_filedata_cell_string(const char *filepath,const char* pattern,const char *reject,const char *default_value, const char *separator)
 *  \brief La funcion analiza linea por linea el archivo y retorna
 *  el texto despues de el primer patrón pattern encontrado, pero estos datos son
 *  separados usando algunos de los separadores usados en separator y ordenandos en una PdsCellString.
 *  <ul>
 *  <li>Es desconsiderado al final de cada linea el salto de linea. </li>
 *  <li>Son desconsiderados al inicio de cada linea los espacios y tabuladores. </li>
 *  <li>Son desconsiderados al final del dato los espacios y tabuladores. </li>
 *  <li>Son desconsideradas al inicio del dato qualquier caracter en reject. </li>
 *  </ul>
 *
 *  Por ejemplo para un archivo mydata.ini con
 * \code{.c}
 *  MY_FIRST_DATA  = 1000
 *  MY_SECOND_DATA = Cadena de texto
 *  MY_THIRD_DATA  = TRUE
 *  MY_FOURTH_DATA = dat1, dat2  , dat3
 * \endcode
 *
 *  Para leer llos datos "dat1, dat2  , dat3" usaremos:
 * \code{.c}
 * Cell=pds_get_filedata_cell_string("mydata.ini","MY_FOURTH_DATA","= ","valor por defecto",";,");
 * \endcode
 *
 *  \param[in] filepath Fichero de texto a analizar. Si filepath==NULL se retorna NULL.
 *  \param[in] pattern Patrón a buscar. Si pattern==NULL se retorna NULL.
 *  \param[in] reject Caracteres a eliminar al inicio del dato encontrado.
 *  Si reject==NULL o vacio, no se elimina ningun caracter al inicio del dato encontrado.
 *  Si la variable default_value es usada entonces reject no es aplicado sobre el dato retornado.
 *  \param[in] default_value Es el valor por defecto que será retornado si pattern
 *  no es encontrado, o si el valor encontrado es una cadena vacia.
 *  Si default_value==NULL y no es encontrado ningun dato o este esta vacio entonces
 *  NULL es retornando.
 *  \param[in] separator Separador de texto. Si separator es igual a NULL o vacia la 
 *  función retorna la cadena sem dividir.
 *  \return Retorna una nueva estructurs PdsCellString con los datos encontrado.
 *  Si no se encontró nada retorna una estructura vacia o NULL
 *  si se encontró algun error.
 *  \ingroup PdsFilesFuncGroup
 */
PdsCellString *pds_get_filedata_cell_string(const char *filepath,const char* pattern,
                                            const char *reject,const char *default_value, 
                                            const char *separator)
{
    char *DATA=NULL;
    PdsCellString *CELL=NULL;
    int i;
    
    if(filepath==NULL)  return NULL;
    if(pattern==NULL)   return NULL;

    DATA=pds_get_filedata_string(filepath,pattern,reject,default_value);
    if(DATA==NULL)  return  NULL;

    if(separator!=NULL) CELL=pds_strsep_cell_string (DATA,separator);
    else                CELL=pds_strsep_cell_string (DATA,"");

    for(i=0;i<CELL->Ncell;i++)
    {
        pds_string_multi_clean_beginning(CELL->data[i],"\t ");
        pds_string_multi_clean_ending(CELL->data[i],"\t ");
    }

    free(DATA);
    return CELL;
}

/** \fn int pds_is_file(const char *path)
 *  \brief PDS_OK si es un fichero (simbólico o real) o PDS_WRONG si no.
 *
 *  \param[in] path Dirección de fichero a testar.
 *  \return Retorna PDS_OK is es un fichero o PDS_WRONG si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_is_file(const char *path)
{
    struct stat stat_result;
    stat(path, &stat_result);

    if( access( path, F_OK ) == -1 ) return PDS_WRONG;

    if(S_ISREG(stat_result.st_mode)==0) return PDS_WRONG;
    else                                return PDS_OK;
}


/** \fn int pds_is_dir(const char *path)
 *  \brief PDS_OK is es un directorio (simbólico o real) o PDS_WRONG si no.
 *
 *  \param[in] path Dirección de directorio a testar.
 *  \return Retorna PDS_OK is es un directorio o PDS_WRONG si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_is_dir(const char *path)
{
    struct stat stat_result;
    stat(path, &stat_result);

    if( access( path, F_OK ) == -1 ) return PDS_WRONG;

    if(S_ISDIR(stat_result.st_mode))    return PDS_OK;
    else                                return PDS_WRONG;
}


/** \fn long pds_get_file_size(const char *path)
 *  \brief Retorna el número de bytes del archivo.
 *
 *  \param[in] path Dirección de archivo a testar.
 *  \return Retorna el número de bytes del archivo.
 *  \ingroup PdsFilesFuncGroup
 */
long pds_get_file_size(const char *path)
{
    FILE *fd=NULL;
    long sz;

    if(!pds_is_file(path))  return -1;

    fd=fopen(path,"rb");
    if(fd==NULL)    return -1;

    fseek(fd, 0L, SEEK_END);
    sz = ftell(fd);

    fclose(fd);
    return sz;
}




/** \fn long pds_get_number_of_chars(const char *path)
 *  \brief Retorna el número de caracteres de un archivo.
 *  Se asume que es un archivo de texto.
 *
 *  \param[in] path Dirección de archivo a testar.
 *  \return Retorna el número de caracteres de un archivo
 *  o negativo en caso de error
 *  \ingroup PdsFilesFuncGroup
 */
long pds_get_number_of_chars(const char *path)
{
    FILE *fd=NULL;
    long count;
    int c;

    if(!pds_is_file(path))  return -1;


    fd=fopen(path,"r");
    if(fd==NULL)    return -1;

    count=0;
    while((c=fgetc(fd))) 
    {
        if(c == EOF) break;
        count++;
    }
    fclose(fd);

    return count;
}

/** \fn int pds_exist_file(const char *path)
 *  \brief Retorna PDS_OK si path es un fichero real o simbólico (no un directorio) 
 *  y es accesible.
 *
 *  \param[in] path Dirección de fichero a testar.
 *  \return Retorna PDS_OK si el archivo existe o PDS_WRONG si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_exist_file(const char *path)
{
    if(pds_archive_is_accessible(path))
    {
        if(pds_is_file(path))   return PDS_OK;
        else                    return PDS_WRONG;
    }
    else    return PDS_WRONG;
    
}


/** \fn int pds_archive_is_accessible(const char *path)
 *  \brief Retorna PDS_OK si el archivo simbólico o real (Directorio o fichero) 
 *  es accesible.
 *
 *  \param[in] path Dirección de archivo a testar.
 *  \return Retorna PDS_OK si el archivo existe o PDS_WRONG si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_archive_is_accessible(const char *path)
{
    if( access( path, F_OK ) != -1 ) 
    {
        return PDS_OK;
    } 
    else
    {
        return PDS_WRONG;
    }
}



/** \fn int pds_file_pattern_replacement(const char *filename_out,const char *filename_in,const char *pat,const char *rep)
 *  \brief Esta función busca en un archivo de texto filename_in, un patron pat y los 
 *  remplaza con el contenido de la cadena rep, el resultado es retornado en un 
 *  archivo filename_out.
 *
 *  Cuando encuentra una coincidencia en la posición p0 del archivo, continua la
 *  siguiente busqueda en p0+strlen(pat).
 *  \param[in] filename_out Archivo de salida donde se retornará el resultado del remplazo.
 *  \param[in] filename_in Archivo de entrada donde se realizará la búsqueda.
 *  \param[in] pat Patron a buscar.
 *  \param[in] rep Patron a de remplazo.
 *  \return Retorna PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_file_pattern_replacement(const char *filename_out,const char *filename_in,const char *pat,const char *rep)
{
    char *str_in=NULL;
    char *str_out=NULL;
    FILE *fd=NULL;

    if(filename_out==NULL)   return PDS_WRONG;
    if(filename_in==NULL)    return PDS_WRONG;
    if(pat==NULL)           return PDS_WRONG;
    if(rep==NULL)           return PDS_WRONG;


    str_in=pds_string_new_from_file(filename_in);
    if(str_in==NULL)    return PDS_WRONG;

    str_out=pds_string_pattern_replacement(str_in,pat,rep);
    if(str_out==NULL)
    {
        free(str_in);
        return PDS_WRONG;
    }

    fd=fopen(filename_out,"w");
    if(fd==NULL)
    {
        free(str_in);
        free(str_out);
        return PDS_WRONG;
    }
    fprintf(fd,"%s",str_out);
    fclose(fd);

    free(str_in);
    free(str_out);

    return PDS_OK;
}


/** \fn int pds_file_content_replacement(   const char *filename_out,const char *filename_in,const char *pat_open,const char *pat_close,char *(*func)(const char *))
 *  \brief Esta función busca en un archivo de texto filename_in, el contenido
 *  entre un patrón pat_open y pat_close y lo remplaza  por el valor devuelto de 
 *  la función func, el resultado de todo esto es escrito en un archivo filename_out.
 *
 *  La función func debe ser de la forma :
\code{.c}
char *func(const char* content);
\endcode
 *
\code{.c}
pds_file_content_replacement("output.html","input.html","<b>","</b>",func);
\endcode
 *  \param[in] filename_out Archivo de salida donde se retornará el resultado del remplazo.
 *  \param[in] filename_in Archivo de entrada donde se realizará la búsqueda.
 *  \param[in] pat_open Patrón inicial a buscar.
 *  \param[in] pat_close Patrón final a buscar.
 *  \param[in] func Es una función que se usará para remplazar el contenido 
 *  encontrado. Por ejemplo si entre pat_open y pat_close es encontrado content
\code{.c}
replace=func(content);
\endcode
 *  este será remplazado por el contenido de la cadena replace, luego la cadena 
 *  replace será liberada internamente con free(replace).
 *  Si la funcion func retorna NULL, la escritura en filename_out es detenida y
 *  se retorna PDS_WRONG.
 *  \return Retorna PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsFilesFuncGroup
 */
int pds_file_content_replacement(   const char *filename_out,const char *filename_in,
                                    const char *pat_open,const char *pat_close,
                                    char *(*func)(const char *))
{
    FILE *fd=NULL;
    char *str_in=NULL;


    char *content=NULL;
    char *rep=NULL;
    char *str_ptr=NULL;
    char *str_last=NULL;
    long Nclose,Ncontent,i,n;

    if(filename_out==NULL)   return PDS_WRONG;
    if(filename_in==NULL)    return PDS_WRONG;
    if(pat_open==NULL)       return PDS_WRONG;
    if(pat_close==NULL)      return PDS_WRONG;
    if((*func)==NULL)        return PDS_WRONG;

    str_in= pds_string_new_from_file(filename_in);
    if(str_in==NULL)    return PDS_WRONG;

    str_ptr=str_in;
    Nclose=strlen(pat_close);

    fd=fopen(filename_out,"w");
    if(fd==NULL)
    {
        free(str_in);
        return PDS_WRONG;
    }


    while(1)
    {
        str_last=str_ptr;

        content=pds_string_get_match_rawdata(str_ptr,pat_open,pat_close,&str_ptr);
        
        if(content!=NULL)
        {
            Ncontent=strlen(content);
            n=( (long)(str_ptr-str_last) )-Ncontent-Nclose;
            for(i=0;i<n;i++)    fputc(str_last[i],fd);

            rep=(*func)(content);
            if(rep==NULL)
            {
                free(str_in);
                free(content);
                fclose(fd);
                return PDS_WRONG;
            }

            fputs(rep,fd);
            fputs(pat_close,fd);
            
            free(content);
            free(rep);
        }
        else
        {
            
            fputs(str_last,fd);
            break;
        }

    }
    fclose(fd); 
  
    return PDS_OK;
}





