/*
 * pdsstring.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <pds/pdsstring.h>
#include <pds/pdscstring.h>
#include <pds/pdsfilesfunc.h>

/** \fn char* pds_sprintf(const char *format, ...)
 *  \brief Similar a sprintf pero esta retorna un puntero a una nueva cadena con 
 *  el texto.
 *
 *  \param[in] format Formato de texto de salida.
 *  \param[in] ... Otros argumentos.
 *  \return Retorna un puntero a una nueva cadena con el texto.
 *  \ingroup PdsStringGroup
 */
char* pds_sprintf(const char *format, ...)
{
    char *char_string=NULL;
    char *p=NULL;
    int N=-1,n;
    int len,i;
    char buffer[32];
    int val;
    va_list args;               /* points to each unnamed arg in turn */

    va_start( args, format );   /* make args point to 1st unnamed arg */
    N=strlen(format);
    for (p = (char*)format; *p; p++) 
    {
        if (*p == '%') 
        {
            p++;

            i=0;
            while(isalpha((int)(*p))==0)
            {
                buffer[i]=(char)(*p); 
                i++;
                p++;
            }
            buffer[i]=0;
            len=atoi(buffer);

            if(*p=='s')         {n=strlen(va_arg(args, char *));}
            else if(*p=='c')    {val = va_arg(args, int); n=1;}
            else                {val = va_arg(args, int); n=64;}//<----Problemas aqui si es entregado %64s
            if(len>n)   N=N+len;
            else        N=N+n;
        }
    }
    va_end( args );             /* clean up when done */

    if(N==-1)
    {
        return NULL;
    }

    char_string=(char*)calloc(N+1,sizeof(char));
    if(char_string==NULL)
    {
        return NULL;
    }


    va_start( args, format );   /* make args point to 1st unnamed arg */
    vsprintf( char_string, format, args );
    va_end( args );             /* clean up when done */


    char_string=(char*)realloc(char_string,strlen(char_string)+1);

    return char_string;
}

/* strsep.c
   Copyright (C) 1992-2015 Free Software Foundation, Inc.
   This file is part of the GNU C Library.
 */
char *pds_strsep (char **stringp, const char *delim)
{
  char *begin, *end;

  begin = *stringp;
  if (begin == NULL)
    return NULL;

  /* A frequent case is when the delimiter string contains only one
     character.  Here we don't need to call the expensive `strpbrk'
     function and instead work using `strchr'.  */
  if (delim[0] == '\0' || delim[1] == '\0')
    {
      char ch = delim[0];

      if (ch == '\0')
	end = NULL;
      else
	{
	  if (*begin == ch)
	    end = begin;
	  else if (*begin == '\0')
	    end = NULL;
	  else
	    end = strchr (begin + 1, ch);
	}
    }
  else
    /* Find the end of the token.  */
    end = strpbrk (begin, delim);

  if (end)
    {
      /* Terminate the token and set *STRINGP past NUL character.  */
      *end++ = '\0';
      *stringp = end;
    }
  else
    /* No more delimiters; this is the last token.  */
    *stringp = NULL;

  return begin;
}

/** \fn PdsCellString *pds_strsep_cell_string(const char *str, const char *delim)
 *  \brief Esta función es similar a la función strsep, donde la cadena str es 
 *  dividida usando algunos de los delimitadores especificados en delim. Los 
 *  pedazos son cargados en una estructura PdsCellString* retornada por la función.
 *  \param[in] str Cadena a dividir, esta cadena no sera modificada.
 *  \param[in] delim Delimitadores a usar. Si delim es NULL la función retorna NULL.
 *  Si delim es una cadena vacia es retornada la cadena entera dentro de una PdsCellString.
 *  \return Retorna un puntero a una nueva estructura de tipo PdsCellString *,
 *  con los tokens de la división.
 *  \ingroup PdsStringGroup
 */
PdsCellString *pds_strsep_cell_string(const char *str, const char *delim)
{
    char *str0 =NULL; 
    char *token=NULL;
    char *str1=NULL;    
    PdsCellString *Tok=NULL;
    int dat;

    if(str==NULL)   return NULL;
    if(delim==NULL) return NULL;

    str0=strdup(str);
    if(str0==NULL)  return NULL;

    Tok=pds_cell_string_new(0);
    if(Tok==NULL)
    {
        return NULL;
    }

    str1=str0;
    do{
        token = pds_strsep(&str0, delim);
        if(token!=NULL)
        {
            //if(strlen(token)>0) 
            {
                dat=pds_cell_string_add(Tok,token);
            }
            
        }
    }while(token!=NULL);

    free(str1);

    return Tok;
}


/** \fn int pds_strcicmp(const char *str0,const char *str1)
 *  \brief Esta función es similar a la función strcmp pero no distinguiendo 
 *  mayúsculas y minúsculas (Case Insensitive), donde si las cadenas son iguales
 *  la funcion retorna 0.
 *  \param[in] str0 Primera cadena a comparar.
 *  \param[in] str1 Segunda cadena a comparar.
 *  \return Retorna 0 si las cadenas son iguales (Case Insensitive).
 *  \ingroup PdsStringGroup
 */
int pds_strcicmp(const char *str0,const char *str1)
{
    int d;

    for (;; str0++, str1++) 
    {
        d = tolower(*str0) - tolower(*str1);
        if( (d != 0) || (*str0==0) )
            return d;
    }
}


/** \fn char * pds_string_new_from_file(const char *path)
 *  \brief Esta función retorna una cadena de caracteres con el contenido de un archivo.
 *  \param[in] path El archivo a leer.
 *  \return Retorna una cadena de caracteres con el contenido de un archivo. O
 *  NULL en caso de error.
 *  \ingroup PdsStringGroup
 */
char * pds_string_new_from_file(const char *path)
{
    char *str=NULL;
    long N;
    FILE *fd=NULL;
    long i;
    int c;

    if(pds_exist_file(path)==FALSE) return NULL;

    N=pds_get_number_of_chars(path);

    fd=fopen(path,"r");
    if(fd==NULL)    return NULL;

    str=(char *)calloc(N+1,sizeof(char));
    if(str==NULL)
    {
        fclose(fd);
        return NULL;
    }

    i=0;
    while( 1 )
    {
        c=fgetc(fd);
        if(i >= N) break;
        if(c == EOF) break;
        str[i]=(char)c;
        i++;
    }

    fclose(fd);
    return str;
}


/** \fn char *pds_get_concatenated_cell_string(const PdsCellString *celldata,const char *separator)
 *  \brief Esta función retorna una cadena de caracteres con el contenido concatenado
 *  usando como separador separator.
 *  \param[in] celldata La estructura a concatenar.
 *  \param[in] separator Separador usado entre celulas.
 *  \return Retorna una nueva cadena de caracteres con el contenido concatenado.
 *  O NULL en caso de error.
 *  \ingroup PdsStringGroup
 */
char *pds_get_concatenated_cell_string(const PdsCellString *celldata,const char *separator)
{
    int N,Nc,Ns,i;
    char *str=NULL;
    char *ptr;
    
    if(celldata==NULL)  return NULL;
    if(separator==NULL)  return NULL;

    Nc=pds_cell_string_get_ncell(celldata);
    if(Nc<=0)   return NULL;
    Ns=strlen(separator);

    N=1+(Nc-1)*strlen(separator);
    for(i=0;i<Nc;i++)
    {
        N=N+strlen(celldata->data[i]);
    }

    str=(char*)calloc(N,sizeof(char));
    if(str==NULL)
    {
        fprintf(stderr,"ERROR while alloc memory.");
        return NULL;
    }


    sprintf(str,"%s",celldata->data[0]);
    ptr=str+strlen(celldata->data[0]);

    for(i=1;i<Nc;i++)
    {
        //printf("Nc=%d\tstr=%p\tptr=%p\tlaslen=%d\t%s\n",Nc,str,ptr,strlen(celldata->data[i-1]),str);
        //getchar();
        sprintf(ptr,"%s%s",separator,celldata->data[i]);
        ptr=ptr+Ns+strlen(celldata->data[i]);
        
    }

    return str;

}

/** \fn int pds_string_how_many_times(const char *str,const char *pat)
 *  \brief Esta función cuenta la cantidad de coincidencias de la cadena pat
 *  en la cadena str.
 *
 *  Cuando encuentra una coincidencia en la posición p0 de str, continua la
 *  siguiente búsqueda en p0+strlen(path).
 *  \param[in] str Cadena donde se realizará la búsqueda.
 *  \param[in] pat Patrona buscar.
 *  \return Retorna el número de coincidencias o -1 en caso de error.
 *  \ingroup PdsStringGroup
 */
int pds_string_how_many_times(const char *str,const char *pat)
{
    long N;
    long L;
    int count,i;

    if(str==NULL)   return -1;
    if(pat==NULL)   return -1;

    L=strlen(pat);
    N=strlen(str);

    i=0;
    count=0;
    while(1)
    {
        if(i>=N)  break;

        if( strncmp((str+i),pat,L)==0 )
        {
            count++;
            i=i+L;
        }
        else    i++;
    }
    return count;
}


/** \fn char *pds_string_new_with_replacement(const char *str,const char *pat,const char *rep)
 *  \brief Esta función busca en una cadena str un patrón pat y los remplaza con 
 *  el contenido de la cadena rep, el resultado es retornado en una nueva cadena;
 *
 *  Cuando encuentra una coincidencia en la posición p0 de str, continua la
 *  siguiente búsqueda en p0+strlen(pat).
 *  \param[in] str Cadena donde se realizará la búsqueda.
 *  \param[in] pat Patrón a buscar.
 *  \param[in] rep Patrón a de remplazo.
 *  \return Retorna una nueva cadena con el resultado de remplazar pat por rep en str.
 *  \ingroup PdsStringGroup
 */
char *pds_string_new_with_replacement(const char *str,const char *pat,const char *rep)
{
    int n,Nstr,Npat,Nrep;
    int is,jn;
    char *newstr=NULL;

    if(str==NULL)   return NULL;
    if(pat==NULL)   return NULL;
    if(rep==NULL)   return NULL;

    n=pds_string_how_many_times(str,pat);
    Nstr=strlen(str);
    Npat=strlen(pat);
    Nrep=strlen(rep);

    newstr=(char *)calloc(Nstr-n*Npat+n*Nrep+1,sizeof(char));
    if(newstr==NULL)    return NULL;

    is=0;
    jn=0;
    while(1)
    {
        if(is>=Nstr)  break;

        if( strncmp(str+is,pat,Npat)==0 )
        {
            strncpy(newstr+jn,rep,Nrep);
            jn=jn+Nrep;
            is=is+Npat;
        }
        else
        {
            newstr[jn]=str[is];
            jn++;
            is++;
        }

    }
    return newstr;

}


/** \fn char *pds_string_new_rawdata_after_match(const char *str,const char *pat_open,const char *pat_close,char **str_ptr)
 *  \brief Esta función busca en una cadena str, los datos entre un patrón 
 *  pat_open y uno pat_close. Retorna los datos en una nueva cadena de texto.
 *
 *  Cuando encuentra una coincidencia la posición inmediata después de pat_close
 *  es cargada en str_ptr (solo si esa variable str_ptr existe).
 *
 *  Ejemplos : Busca la coincidencia y además es cargado str_ptr con la dirección
 *  inmediata a la coincidencia con pat_close.
\code{.c}
char str[]="hola <b> Fernando\n</b> como estas";
char *str_ptr=NULL;
char *rawdata=pds_string_new_rawdata_after_match(str,"<b>","</b>",&str_ptr);
\endcode
 *  Ejemplos : Busca la coincidencia en str.
\code{.c}
char str[]="hola <b> Fernando\n</b> como estas";
char *rawdata=pds_string_new_rawdata_after_match(str,"<b>","</b>",NULL);
\endcode
 *  \param[in] str Cadena donde se realizará la búsqueda.
 *  \param[in] pat_open Patrón de apertura.
 *  \param[in] pat_close Patrón de finalización.
 *  \param[out] str_ptr puntero en cuyo contenido se grabará la dirección de la
 *  dirección de memoria en str después de pat_close.
 *  Si se retorna un str_ptr==NULL significa que se llegó al final de la cadena.
 *  Este solo es una dirección de memoria de str y no debe ser liberada.
 *  \return Retorna una nueva cadena con el contenido entre pat_open y pat_close.
 *  \ingroup PdsStringGroup
 */
char *pds_string_new_rawdata_after_match(const char *str,const char *pat_open,const char *pat_close,char **str_ptr)
{
    long i;
    long offset_open;
    long offset_close;
    long N,Nopen,Nclose;

    char *str0=NULL;
    char *str_open=NULL;
    char *str_close=NULL;

    char *data=NULL;
    long data_length;


    if( (str==NULL)||(pat_open==NULL)||(pat_close==NULL) )
    {
        if( (*str_ptr)!=NULL )  (*str_ptr)=NULL;   
        return NULL;
    }

    str0=(char*)str;

    N      = strlen(str0);
    Nopen  = strlen(pat_open);
    Nclose = strlen(pat_close);

    //Buscamos el patrón de apertura
    i=0;
    while(1)
    {
        if( i>(N-Nopen-Nclose) )  break;

        if( strncmp(str0+i,pat_open,Nopen)==0 )
        {
            str_open=str0+i;
            offset_open=i;
            break;
        }

        i++;
    }

    if(str_open==NULL)
    {
        if( (*str_ptr)!=NULL )  (*str_ptr)=NULL;   
        return NULL;
    }

    //Buscamos el patrón de finalización
    i=offset_open+Nopen;

    while(1)
    {
        if( i>(N-Nclose) )  break;

        if( strncmp(str0+i,pat_close,Nclose)==0 )
        {

            str_close=str0+i;
            offset_close=i;
            break;
        }

        i++;
    }

    if(str_close==NULL)
    {
        if( (*str_ptr)!=NULL )  (*str_ptr)=NULL;   
        return NULL;
    }



    //Creando data
    data_length=offset_close-(offset_open+Nopen);
    
    data=(char*)calloc(data_length+1,sizeof(char));
    if(data==NULL)
    {
        if( (*str_ptr)!=NULL )  (*str_ptr)=NULL;   
        return NULL;
    }

    for(i=0;i<data_length;i++)
    data[i]=str[offset_open+Nopen+i];

    if( (offset_close+Nclose)<N)
        *str_ptr=str0+(offset_close+Nclose);
    else
        *str_ptr=NULL;


    return data;
}

