/*
 * pdslstring.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdslstring.h
 *  \author Fernando Pujaico Rivera
 *  \date 30-07-2016
 *  \brief Funciones que trabajan con listas doblemente enlazadas que almacenan 
 *  cadenas de texto.
 *   
 */

#ifndef __PDSLSTRING_H__
#define __PDSLSTRING_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsListStringGroup Funciones del módulo PdsListString.
 *
 *  Estas funciones manipulan una lista con elementos de tipo cadena.
 *  <br>
 *  Funciones.
 * @{
 */



/*! \struct PdsListString
 *  \brief Una estructura tipo  PdsListString .
 *  Esta estructura genera una doblemente enlazada lista de cadenas te texto.
 *  <br>
 *
 *  \ingroup PdsListStringGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	// Dirección del nodo anterios de la lista.
	struct PdsListString *AddPrev;
	// Strings real asociado; dentro;del nodo.
	char *S;
	// Direccion del nodo siguiente en la lista.
	struct PdsListString *AddNext;
}PdsListString;




/** \fn PdsListString *pds_list_string_new(void)
 *  \brief Crea una lista de tipo PdsListString vacia.
 *  \return Un puntero a la cadena de tipo PdsListString.
 *  \ingroup PdsListStringGroup
 */
PdsListString *pds_list_string_new(void);


/** \fn int pds_list_string_push(PdsListString **Lprev,const char* string_data)
 *  \brief Agrega un elemento a la cima de la lista.
 *  \param[in,out] Lprev El nodo de la cima de la lista.
 *  \param[in] string_data La cadena a escrever.
 *  \return TRUE si todo fue bien o FALSE si no. Si no le das el último nodo da error
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_push(PdsListString **Lprev,const char* string_data);


/** \fn int pds_list_string_pop(PdsListString **L)
 *  \brief Quita un elemento de la lista. Si no hay elementos retorna FALSE.
 *  \param[in,out] L El nodo de la cima de la lista.
 *  \return TRUE si todo fue bien o FALSE si no. Si no le das el último nodo da error
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_pop(PdsListString **L);


/** \fn int pds_list_string_shift(PdsListString **L)
 *  \brief Quita el elemento inicial;mas antiguo; de la lista, si no hay elementos retorna FALSE.
 *  \param[in,out] L El primer nodo de la lista.
 *  \return TRUE si todo fue bien o FALSE si no. Si no le das el primer nodo da error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_shift(PdsListString **L);


/** \fn int pds_list_string_unshift(PdsListString **Lnext,const char* string_data)
 *  \brief Agrega un elemento al inicio de la lista.
 *  \param[in,out] Lnext El primer nodo de la lista.
 *  \param[in] string_data La cadena a escrever.
 *  \return TRUE si todo fue bien o FALSE si no. Si no le das el primer nodo da error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_unshift(PdsListString **Lnext,const char* string_data);


/** \fn int pds_list_string_top(PdsListString **L)
 *  \brief Busca el elemento final; superior; de la lista.
 *  \param[in,out] L Un nodo de la lista, en donde se cargará el último nodo.
 *  \return TRUE si todo fue bien o FALSE si no. 
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_top(PdsListString **L);


/** \fn int pds_list_string_bottom(PdsListString **L)
 *  \brief Busca el elemento inicial; inferior; de la lista.
 *  \param[in,out] L Un nodo de la lista, en donde se cargará el primer nodo.
 *  \return TRUE si todo fue bien o FALSE si no. 
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_bottom(PdsListString **L);


/** \fn int pds_list_string_printf(const PdsListString *L)
 *  \brief Imprime en pantalla los datos de un nodo de tipo puntero PdsListString.
 *  \param[in] L El nodo a imprimir en pantalla.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_printf(const PdsListString *L);


/** \fn int pds_list_string_all_printf(const PdsListString *L)
 *  \brief Imprime en pantalla todos los datos de la lista.
 *  \param[in] L Un nodo de la lista a imprimir en pantalla.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_all_printf(const PdsListString *L);


/** \fn void pds_list_string_free(PdsListString *L)
 *  \brief Libera una lista entera de tipo puntero PdsListString. 
 *  \param[in,out] L La lista a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsListStringGroup
 */
void pds_list_string_free(PdsListString *L);


/** \fn void pds_list_string_destroy(PdsListString **L)
 *  \brief Libera una lista de tipo puntero PdsListString, y limpia el puntero con NULL.
 *  \param[in,out] L La lista a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsListStringGroup
 */
void pds_list_string_destroy(PdsListString **L);




/** \fn int pds_list_string_add(PdsListString **Lprev,const char* string_data)
 *  \brief Agrega un elemento a la cima de la lista. Si la lista no esta en la cima,
 *  la lista es llevada a la cima y es dejada luego de agregar en esta posición.
 *  \param[in,out] Lprev Un nodo cualquiera de la lista.
 *  \param[in] string_data La cadena a escrever.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_add(PdsListString **Lprev,const char* string_data);


/** \fn int pds_list_string_read(PdsListString **Lprev,char** string_data)
 *  \brief Lee una cadena de texto, en la lista actual y se coloca en el siguiente 
 *  nodo de la lista, si está ya en el último nodo no se mueve al siguiente nodo.
 *  \param[in,out] Lprev Un nodo cualquiera de la lista.
 *  \param[out] string_data La dirección de la cadena a leer (NO liberar esta 
 *  dirección de memoria).
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_read(PdsListString **Lprev,char** string_data);


/** \fn int pds_list_string_is_top(const PdsListString *L)
 *  \brief Retorna TRUE si L apunta el nodo que esta en el top de la lista.
 *  \param[in] L Un nodo cualquiera de la lista.
 *  \return TRUE si L apunta al final de la lista o FALSE si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_is_top(const PdsListString *L);


/** \fn int pds_list_string_is_bottom(const PdsListString *L)
 *  \brief Retorna TRUE si L apunta el nodo que esta en el bottom de la lista.
 *  \param[in] L Un nodo cualquiera de la lista.
 *  \return TRUE si L apunta al inicio de la lista o FALSE si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_is_bottom(const PdsListString *L);


/** \fn int pds_list_string_get_length(const PdsListString *L)
 *  \brief Cuenta la cantidad de nodos de la lista.
 *  Es considerada una lista con cero nodos cuando no exiten nodos antes, nodos 
 *  despues y el valor actual de la lista NULL.
 *  \param[in] L Un nodo cualquiera de la lista a consultar.
 *  \return Retorna la cantidad de nodos de la lista o un número menor que cero 
 *  en caso de error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_get_length(const PdsListString *L);


/** \fn int pds_list_string_is_empty(const PdsListString *L)
 *  \brief Indica si la lista está vacía.
 *  \param[in] L Un nodo cualquiera de la lista a consultar.
 *  \return Retorna TRUE si L==NULL o si todos los elementos del nodo son nulos.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_is_empty(const PdsListString *L);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSLSTRING_H__ */ 

