/*
 * pdsstring.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsstring.h
 *  \author Fernando Pujaico Rivera
 *  \date 30-07-2016
 *  \brief Funciones que manipulan archivos y caminos.
 *   
 */

#ifndef __PDSSTRING_H__
#define __PDSSTRING_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>
#include <pds/pdscstring.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsStringGroup Funciones del módulo PdsString.
 *
 *  <br>
 *  Funciones.
 * @{
 */


/** \fn char* pds_sprintf(const char *format, ...)
 *  \brief Similar a sprintf pero esta retorna un puntero a una nueva cadena con 
 *  el texto.
 *
 *  \param[in] format Formato de texto de salida.
 *  \param[in] ... Otros argumentos.
 *  \return Retorna un puntero a una nueva cadena con el texto.
 *  \ingroup PdsStringGroup
 */
char* pds_sprintf(const char *format, ...);


/** \fn PdsCellString *pds_strsep(const char *str, const char *delim)
 *  \brief Esta función es similar a la función strsep, donde la cadena str es 
 *  dividida usando algunos de los delimitadores especificados en delim. Los 
 *  pedazos son cargados en una estructura PdsCellString* retornada por la función.
 *  \param[in] str Cadena a dividir, esta cadena no sera modificada.
 *  \param[in] delim Delimitadores a usar.
 *  \return Retorna un puntero a una nueva estructura de tipo PdsCellString *,
 *  con los tokens de la división.
 *  \ingroup PdsStringGroup
 */
PdsCellString *pds_strsep(const char *str, const char *delim);


/** \fn int pds_strcicmp(const char *str0,const char *str1)
 *  \brief Esta función es similar a la función strcmp pero no distinguiendo 
 *  mayúsculas y minúsculas (Case Insensitive), donde si las cadenas son iguales
 *  la funcion retorna 0.
 *  \param[in] str0 Primera cadena a comparar.
 *  \param[in] str1 Segunda cadena a comparar.
 *  \return Retorna 0 si las cadenas son iguales (Case Insensitive).
 *  \ingroup PdsStringGroup
 */
int pds_strcicmp(const char *str0,const char *str1);

/** \fn char * pds_string_new_from_file(const char *path)
 *  \brief Esta función retorna una cadena de caracteres con el contenido de un archivo.
 *  \param[in] path El archivo a leer.
 *  \return Retorna una cadena de caracteres con el contenido de un archivo. O
 *  NULL en caso de error.
 *  \ingroup PdsStringGroup
 */
char * pds_string_new_from_file(const char *path);


/** \fn char *pds_get_concatenated_cell_string(const PdsCellString *celldata,const char *separator)
 *  \brief Esta función retorna una cadena de caracteres con el contenido concatenado
 *  usando como separador separator.
 *  \param[in] celldata La estructura a concatenar.
 *  \param[in] separator Separador usado entre celulas.
 *  \return Retorna una nueva cadena de caracteres con el contenido concatenado.
 *  O NULL en caso de error.
 *  \ingroup PdsStringGroup
 */
char *pds_get_concatenated_cell_string(const PdsCellString *celldata,const char *separator);


/** \fn int pds_string_how_many_times(const char *str,const char *pat)
 *  \brief Esta función cuenta la cantidad de coincidencias de la cadena pat
 *  en la cadena str.
 *
 *  Cuando encuentra una coincidencia en la posición p0 de str, continua la
 *  siguiente búsqueda en p0+strlen(path).
 *  \param[in] str Cadena donde se realizará la búsqueda.
 *  \param[in] pat Patrona buscar.
 *  \return Retorna el número de coincidencias o -1 en caso de error.
 *  \ingroup PdsStringGroup
 */
int pds_string_how_many_times(const char *str,const char *pat);


/** \fn char *pds_string_new_with_replacement(const char *str,const char *pat,const char *rep)
 *  \brief Esta función busca en una cadena str un patrón pat y los remplaza con 
 *  el contenido de la cadena rep, el resultado es retornado en una nueva cadena;
 *
 *  Cuando encuentra una coincidencia en la posición p0 de str, continua la
 *  siguiente búsqueda en p0+strlen(pat).
 *  \param[in] str Cadena donde se realizará la búsqueda.
 *  \param[in] pat Patrón a buscar.
 *  \param[in] rep Patrón a de remplazo.
 *  \return Retorna una nueva cadena con el resultado de remplazar pat por rep en str.
 *  \ingroup PdsStringGroup
 */
char *pds_string_new_with_replacement(const char *str,const char *pat,const char *rep);


/** \fn char *pds_string_new_rawdata_after_match(const char *str,const char *pat_open,const char *pat_close,char **str_ptr)
 *  \brief Esta función busca en una cadena str, los datos entre un patrón 
 *  pat_open y uno pat_close. Retorna los datos en una nueva cadena de texto.
 *
 *  Cuando encuentra una coincidencia la posición inmediata después de pat_close
 *  es cargada en str_ptr (solo si esa variable str_ptr existe).
 *
 *  Ejemplos : Busca la coincidencia y además es cargado str_ptr con la dirección
 *  inmediata a la coincidencia con pat_close.
\code{.c}
char str[]="hola <b> Fernando\n</b> como estas";
char *str_ptr=NULL;
char *rawdata=pds_string_new_rawdata_after_match(str,"<b>","</b>",&str_ptr);
\endcode
 *  Ejemplos : Busca la coincidencia en str.
\code{.c}
char str[]="hola <b> Fernando\n</b> como estas";
char *rawdata=pds_string_new_rawdata_after_match(str,"<b>","</b>",NULL);
\endcode
 *  \param[in] str Cadena donde se realizará la búsqueda.
 *  \param[in] pat_open Patrón de apertura.
 *  \param[in] pat_close Patrón de finalización.
 *  \param[out] str_ptr puntero en cuyo contenido se grabará la dirección de la
 *  dirección de memoria en str después de pat_close.
 *  Si se retorna un str_ptr==NULL significa que se llegó al final de la cadena.
 *  Este solo es una dirección de memoria de str y no debe ser liberada.
 *  \return Retorna una nueva cadena con el contenido entre pat_open y pat_close.
 *  \ingroup PdsStringGroup
 */
char *pds_string_new_rawdata_after_match(const char *str,const char *pat_open,const char *pat_close,char **str_ptr);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSSTRING_H__ */ 

