#
# This file is part of OpenClone.
#
# Copyright (C) 2009  David Gnedt
#
# OpenClone is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenClone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with OpenClone.  If not, see <http://www.gnu.org/licenses/>.
#

from sqlalchemy import *
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.orm import relation, backref

Base = declarative_base()

class Partitiontable(Base):
    __tablename__ = 'partitiontable'
    id = Column('id', Integer, primary_key=True)
    type = Column('type', String(3))
    __mapper_args__ = {'polymorphic_on': type}

class MBRPartitiontable(Partitiontable):
    __tablename__ = 'partitiontable_mbr'
    __mapper_args__ = {'polymorphic_identity': 'mbr'}
    id = Column('id', Integer, ForeignKey('partitiontable.id'), primary_key=True)
    disk_signature = Column('disk_signature', Binary, nullable=False)
    bootloader = Column('bootloader', Binary, nullable=False)
    partitions = Column('partitions', Binary, nullable=False)
    extended = Column('extended', Binary)
    unused = Column('unused', Binary)

class Partition(Base):
    __tablename__ = 'partition'
    id = Column('id', Integer, primary_key=True)
    partitiontable_id = Column('partitiontable_id', Integer, ForeignKey('partitiontable.id'), nullable=False)
    partitiontable = relation('Partitiontable', backref='parts')
    type = Column('type', String(3), nullable=False)
    fs = Column('fs', String(255))
    os = Column('os', String(255))
    __mapper_args__ = {'polymorphic_on': type}

class MBRPartition(Partition):
    __tablename__ = 'partition_mbr'
    __mapper_args__ = {'polymorphic_identity': 'mbr'}
    id = Column('id', Integer, ForeignKey('partition.id'), primary_key=True)
    no = Column('no', Integer, nullable=False)
    record_type = Column('record_type', String(8), nullable=False)
    bootable = Column('bootable', Boolean, nullable=False)
    partition_type = Column('partition_type', Integer, nullable=False)
    start_lba = Column('start_lba', Integer, nullable=False)
    sectors = Column('sectors', Integer, nullable=False)
    start_head = Column('start_head', Integer, nullable=False)
    start_sector = Column('start_sector', Integer, nullable=False)
    start_cylinder = Column('start_cylinder', Integer, nullable=False)
    end_head = Column('end_head', Integer, nullable=False)
    end_sector = Column('end_sector', Integer, nullable=False)
    end_cylinder = Column('end_cylinder', Integer, nullable=False)

class Group(Base):
    __tablename__ = 'group'
    id = Column('id', Integer, primary_key=True)
    name = Column('name', String(255))

class Host(Base):
    __tablename__ = 'host'
    id = Column('id', Integer, primary_key=True)
    group_id = Column('group_id', Integer, ForeignKey('group.id'))
    group = relation('Group', backref='hosts')
    mac = Column('mac', String(12), nullable=False)
    ip = Column('ip', String(15))
    hostname = Column('hostname', String(255))
    serial_no = Column('serial_no', String(255))
    running = Column('running', Boolean, nullable=False)
    last_boot_time = Column('last_boot_time', DateTime)

class Device(Base):
    __tablename__ = 'device'
    id = Column('id', Integer, primary_key=True)
    host_id = Column('host_id', Integer, ForeignKey('host.id'), nullable=False)
    host = relation('Host', backref='devices')
    type = Column('type', String(8), nullable=False)
    __mapper_args__ = {'polymorphic_on': type}

class CPUDevice(Device):
    __tablename__ = 'device_cpu'
    __mapper_args__ = {'polymorphic_identity': 'cpu'}
    id = Column('id', Integer, ForeignKey('device.id'), primary_key=True)
    processor_no = Column('processor_no', Integer, nullable=False)
    vendor_id = Column('vendor_id', String(255))
    model_name = Column('model_name', String(255))
    mhz = Column('mhz', Float)
    bogomips = Column('bogomips', Float)
    cache = Column('cache', Integer)
    core_id = Column('core_id', Integer)
    cpu_cores = Column('cpu_cores', Integer)

class RAMDevice(Device):
    __tablename__ = 'device_ram'
    __mapper_args__ = {'polymorphic_identity': 'ram'}
    id = Column('id', Integer, ForeignKey('device.id'), primary_key=True)
    size = Column('size')

class HardDiskDevice(Device):
    __tablename__ = 'device_harddisk'
    __mapper_args__ = {'polymorphic_identity': 'harddisk'}
    id = Column('id', Integer, ForeignKey('device.id'), primary_key=True)
    address = Column('address', String(255), nullable=False)
    partitiontable_id = Column('partitiontable_id', Integer, ForeignKey('partitiontable.id'))
    partitiontable = relation('Partitiontable', uselist=False, backref=backref('harddisk'))
    size = Column('size', Integer, nullable=False)
    sector_size = Column('sector_size', Integer, nullable=False)
    model_no = Column('model_no', String(255))
    serial_no = Column('serial_no', String(255))
    firmware_rev = Column('firmware_rev', String(255))
    wwn = Column('wwn', Integer)
    cylinder = Column('cylinder', Integer)
    heads = Column('heads', Integer)
    sectors = Column('sectors', Integer)

class Image(Base):
    __tablename__ = 'image'
    id = Column('id', Integer, primary_key=True)
    name = Column('name', String(255))

class DiskImage(Base):
    __tablename__ = 'image_disk'
    id = Column('id', Integer, primary_key=True)
    image_id = Column('image_id', Integer, ForeignKey('image.id'), nullable=False)
    image = relation('Image', backref='disks')
    partitiontable_id = Column('partitiontable_id', Integer, ForeignKey('partitiontable.id'))
    partitiontable = relation('Partitiontable')
    address = Column('address', String(255))

class PartitionImage(Base):
    __tablename__ = 'image_partition'
    id = Column('id', Integer, primary_key=True)
    image_disk_id = Column('image_disk_id', Integer, ForeignKey('image_disk.id'), nullable=False)
    disk = relation('DiskImage', backref='partitions')
    partition_id = Column('partition_id', Integer, ForeignKey('partition.id'), nullable=False)
    partition = relation('Partition')
    format = Column('format', String(10), nullable=False)
    compression = Column('compression', String(5), nullable=False)
    size = Column('size', Integer)
    minimum_size = Column('minimum_size', Integer)
    image_size = Column('image_size', Integer)

class Task(Base):
    __tablename__ = 'task'
    id = Column('id', Integer, primary_key=True)
    sequence = Column('sequence', Integer, nullable=False)
    description = Column('description', String(255))

class Operation(Base):
    __tablename__ = 'operation'
    #UniqueConstraint(['task_id', 'sequence'])
    id = Column('id', Integer, primary_key=True)
    task_id = Column('task_id', Integer, ForeignKey('task.id'), nullable=False)
    task = relation('Task', backref='operations')
    sequence = Column('sequence', Integer, nullable=False)
    type = Column('type', String(8), nullable=False)
    __mapper_args__ = {'polymorphic_on': type, 'order_by': [task_id, sequence]}

class PartitionOperation(Operation):
    __tablename__ = 'operation_partition'
    __mapper_args__ = {'polymorphic_identity': 'partition'}
    id = Column('id', Integer, ForeignKey('operation.id'), primary_key=True)
    partitiontable_id = Column('partitiontable_id', Integer, ForeignKey('partitiontable.id'), nullable=False)
    partitiontable = relation('Partitiontable')
    program = Column('program', String(6), nullable=False)
    address = Column('address', String(255), nullable=False)
    restore_bootloader = Column('restore_bootloader', Boolean, nullable=False)
    restore_unused = Column('restore_unused', Boolean, nullable=False)

class ImageOperation(Operation):
    __tablename__ = 'operation_image'
    __mapper_args__ = {'polymorphic_identity': 'image'}
    id = Column('id', Integer, ForeignKey('operation.id'), primary_key=True)
    mode = Column('mode', String(7), nullable=False)
    image_partition_id = Column('image_partition_id', Integer, ForeignKey('image_partition.id'), nullable=False)
    image_partition = relation('PartitionImage')
    address = Column('address', String(255), nullable=False)
    transfer_mode = Column('transfer_mode', String(9), nullable=False)

class ShutdownOperation(Operation):
    __tablename__ = 'operation_shutdown'
    __mapper_args__ = {'polymorphic_identity': 'shutdown'}
    id = Column('id', Integer, ForeignKey('operation.id'), primary_key=True)
    mode = Column('mode', String(8), nullable=False)

class HostTask(Base):
    __table__ = Table('host_task', Base.metadata,
        Column('host_id', Integer, ForeignKey('host.id'), primary_key=True),
        Column('task_id', Integer, ForeignKey('task.id'), primary_key=True),
        Column('status', String(8), nullable=False),
        Column('cur_operation_id', Integer),
        Column('next_operation_id', Integer),
        Column('percentage', Float),
        Column('speed', String(255)),
        ForeignKeyConstraint(['task_id', 'cur_operation_id'],
                             ['operation.task_id', 'operation.id']),
        ForeignKeyConstraint(['task_id', 'next_operation_id'],
                             ['operation.task_id', 'operation.id'])
    )
    host = relation('Host', backref='hosttasks')
    task = relation('Task', backref='taskhosts')

HostTask.cur_operation = relation('Operation', primaryjoin=and_(HostTask.task_id==Operation.task_id, HostTask.cur_operation_id==Operation.id))
HostTask.next_operation = relation('Operation', primaryjoin=and_(HostTask.task_id==Operation.task_id, HostTask.next_operation_id==Operation.id))
