#!/bin/sh
#
# This file is part of OpenClone.
#
# Copyright (C) 2009  David Gnedt
#
# OpenClone is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenClone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with OpenClone.  If not, see <http://www.gnu.org/licenses/>.
#

# TODO: Use random temp directory names

HG_REPO="http://hg.savannah.nongnu.org/hgweb/openclone/"
OPENCLONE_VER="rel"

OPENCLONE_MAJOR="0"
OPENCLONE_MINOR="1"
OPENCLONE_PATCH="1"
OPENCLONE_REV="109:c97c1623ab27"
OPENCLONE_DEV="false"

DIST_SUITE="jaunty"
DIST_MIRROR="http://de.archive.ubuntu.com/ubuntu"
CLIENT_HOSTNAME="OpenCloneClient"
UDPCAST_VER="20081213"
SQLALCHEMY_VER="0.4.8-1"
KEYRING_VER="2008.03.04"
ZSI_VER="2.0-2ubuntu4"

status_text() {
	echo -n " * $1 "
}

status_ok() {
	if [ -z "$1" ]; then
		echo "done"
	else
		echo "$1"
	fi
}

status_failed() {
	if [ -z "$1" ]; then
		echo "*FAILED*"
	else
		echo "$1"
	fi
}

status_error() {
	echo "ERROR: $1"
	exit 1
}

question() {
	echo -n "   $1 "
	read QUESTION_RET
}

question_yn() {
	while true; do
		question "$1 [y/n]"
		if [ "$QUESTION_RET" = "y" -o "$QUESTION_RET" = "Y" -o "$QUESTION_RET" = "yes" -o "$QUESTION_RET" = "Yes" ]; then
			QUESTION_RET="y"
			return
		elif [ "$QUESTION_RET" = "n" -o "$QUESTION_RET" = "N" -o "$QUESTION_RET" = "no" -o "$QUESTION_RET" = "No" ]; then
			QUESTION_RET="n"
			return
		else
			echo "Invalid input, please try again"
		fi
	done
}

STEP2=""

for ARG in "$@"; do
	case "$ARG" in
		--dev)
			OPENCLONE_VER="tip"
			;;
		--step2)
			STEP2="true"
			;;
	esac
done

#status_text "Checking for new installer..."
#
#if which wget; then
#	LATEST_VERSION="`wget -q -O - http://openclone.nongnu.org/latest.php?mode=check&version=$OPENCLONE_VER&major=$OPENCLONE_MAJOR&minor=$OPENCLONE_MINOR&patch=$OPENCLONE_PATCH&rev=$OPENCLONE_REV&dev=$OPENCLONE_DEV`"
#	if [ $? -ne 0 ]; then
#		LATEST_VERSION=""
#		status_failed "wget *FAILED*"
#	fi
#elif which curl; then
#	LATEST_VERSION="`curl -s -f http://openclone.nongnu.org/latest.php?version=$OPENCLONE_VER`"
#	if [ $? -ne 0 ]; then
#		LATEST_VERSION=""
#		status_failed "curl *FAILED*"
#	fi
#elif which lwp-request; then
#	LATEST_VERSION="`lwp-request http://openclone.nongnu.org/latest.php?version=$OPENCLONE_VER`"
#	if [ $? -ne 0 ]; then
#		LATEST_VERSION=""
#		status_failed "lwp-request *FAILED*"
#	fi
#else
#	status_failed "wget, curl or lwp-request is required"
#fi

status_text "Checking for root previleges..."

if [ "`id --user`" = "0" ]; then
	status_ok
else
	status_failed
	status_error "Please execute this script with root privileges (e.g. with sudo)"
fi

status_text "Detecting installation sources..."
echo ""
echo ""

SOURCE=""

which hg > /dev/null
if [ $? -eq 0 ]; then
	SOURCE="hg"
	echo "   [m] Mercurial repository (online)"
fi

for DIR in "`dirname $0`" "`dirname $0`/.." "." ".."; do
	if [ -d "$DIR/openclone-database" -a -d "$DIR/openclone-engine" -a -d "$DIR/openclone-webinterface" -a -d "$DIR/openclone-webservice" ]; then
		SOURCE="local"
		echo "   [l] Local files (offline)"
		break
	fi
done

if [ -z "$SOURCE" ]; then
	status_error "No installation sources found. If you want to perform an online installation you need Mercurial else you have to extract the OpenClone tar package and execute the included installer."
fi

if [ -z "$STEP2" ]; then
	echo ""
	question "What installation source do you want to use?"
else
	QUESTION_RET="m"
fi

SOURCE=""

if [ "$QUESTION_RET" = "m" -o "$QUESTION_RET" = "M" ]; then
	if [ -d /var/cache/openclone ]; then
		(cd /var/cache/openclone && hg verify)
		if [ $? -eq 0 ]; then
			if [ -z "$STEP2" ]; then
				(cd /var/cache/openclone && hg revert -a)
				if [ $? -ne 0 ]; then
					status_error "Unexpected Mercurial error"
				fi
				(cd /var/cache/openclone && hg pull "$HG_REPO")
				if [ $? -ne 0 ]; then
					status_error "Unexpected Mercurial error"
				fi
			fi
		else
			status_error "Unexpected cache files found. Please remove /var/cache/openclone"
		fi
	else
		if [ -z "$STEP2" ]; then
			hg clone "$HG_REPO" /var/cache/openclone
			if [ $? -ne 0 ]; then
				status_error "Unexpected Mercurial error"
			fi
		else
			status_error "Mercurial repository not found (Step 2 failed)"
		fi
	fi
	TAG_LINE="`(cd /var/cache/openclone && hg tags) | grep "^$OPENCLONE_VER" | head -n 1`"
	TAG="`echo $TAG_LINE | awk '{ print $1 }'`"
	REV="`echo $TAG_LINE | awk '{ print $2 }'`"
	if [ -z "$TAG" ]; then
		status_error "No release found"
	fi
	if [ -z "$STEP2" ]; then
		(cd /var/cache/openclone && hg update -r "$TAG")
		if [ $? -ne 0 ]; then
			status_error "Unexpected Mercurial error"
		fi
		INSTALLER_FOUND=""
		status_text "Updating installer..."
		for INSTALLER in "$0" "`which $0`"; do
			if [ -f "$INSTALLER" ]; then
				INSTALLER_FOUND="true"
				if ! cmp -s /var/cache/openclone/openclone-installer/install.sh "$INSTALLER"; then
					cp /var/cache/openclone/openclone-installer/install.sh "$INSTALLER"
					status_ok
					exec sh "$INSTALLER" "$@" --step2
				else
					status_ok "Already up to date"
					break
				fi
			fi
		done
		if [ -z "$INSTALLER_FOUND" ]; then
			status_error "Installer not found"
		fi
	fi
	if [ "$OPENCLONE_VER" = "tip" ]; then
		TAG_LINE="`(cd /var/cache/openclone && hg tags) | grep "^rel" | head -n 1`"
		TAG="`echo $TAG_LINE | awk '{ print $1 }'`"
		OPENCLONE_DEV="true"
	else
		OPENCLONE_DEV="false"
	fi
	OPENCLONE_MAJOR="`echo $TAG | cut -b4- | cut -d "." -f 1`"
	OPENCLONE_MINOR="`echo $TAG | cut -d "." -f 2`"
	OPENCLONE_PATCH="`echo $TAG | cut -d "." -f 3`"
	OPENCLONE_REV="$REV"
	SOURCE="/var/cache/openclone"
elif [ "$QUESTION_RET" = "l" -o "$QUESTION_RET" = "L" ]; then
	# TODO: Fix offline installation
	echo "WARNING: Currently a real offline installation isn't possible, you still require Internet connectivity"
	SOURCE="$DIR"
fi

status_text "Checking for your distribution..."

if [ -f /etc/lsb-release ]; then
	DISTRIB_ID="`sed -n "s/^DISTRIB_ID=\(.*\)\$/\1/p" /etc/lsb-release`"
	DISTRIB_RELEASE="`sed -n "s/^DISTRIB_RELEASE=\(.*\)\$/\1/p" /etc/lsb-release`"
elif [ -f /etc/debian_version ]; then
	DISTRIB_ID="Debian"
	DISTRIB_RELEASE="`cat /etc/debian_version`"
else
	status_error "Couldn't detect your distribution."
fi

echo "$DISTRIB_ID $DISTRIB_RELEASE found"
case "$DISTRIB_ID" in
	Debian)
		case "$DISTRIB_RELEASE" in
			5.0.1|5.0.2)
				;;

			*)
				if [ "`echo "$DISTRIB_RELEASE" | tr -d "."`" -lt 501 ]; then
					status_error "Sorry your distribution version is not supported. Please consider an upgrade."
				else
					status_error "Sorry your distribution version is not supported."
				fi
				;;
		esac
		;;

	Ubuntu)
		case "$DISTRIB_RELEASE" in
			8.04)
			# TODO: Test script on 8.10, 9.04
				;;

			*)
				if [ "`echo "$DISTRIB_RELEASE" | tr -d "."`" -lt 804 ]; then
					status_error "Sorry your distribution version is not supported. Please consider an upgrade."
				else
					status_error "Sorry your distribution version is not supported."
				fi
				;;
		esac
		;;

	*)
		status_error "Sorry your distribution is not supported by this installer."
		;;
esac

status_text "Checking for previous OpenClone installation..."

# TODO: Optimize OpenClone installation detection
if [ -d /etc/openclone ]; then
	echo "found"
	# TODO: Check if new version is available
	#if [ "$OPENCLONE_DEV" = "true" ]; then
	#	OPENCLONE_VER_STR="$OPENCLONE_MAJOR.$OPENCLONE_MINOR.$OPENCLONE_PATCH+dev (rev $OPENCLONE_REV)"
	#else
	#	OPENCLONE_VER_STR="$OPENCLONE_MAJOR.$OPENCLONE_MINOR.$OPENCLONE_PATCH (rev $OPENCLONE_REV)"
	#fi
	question_yn "Do you want to UPDATE your OpenClone installation?"
	if [ "$QUESTION_RET" = "y" ]; then
		MODE="update"
	else
		exit 0
	fi
else
	echo "not found"
	question_yn "Do you want to INSTALL OpenClone?"
	if [ "$QUESTION_RET" = "y" ]; then
		MODE="install"
	else
		exit 0
	fi
fi

ask_network() {
	echo ""
	echo "== Network configuration =="
	# TODO: Ask for Ubuntu mirror
	# TODO: Display available network interfaces
	question "What network interface is connected to the clients? [eth0]"
	if [ -z "$QUESTION_RET" ]; then
		CLIENT_INTERFACE="eth0"
	else
		CLIENT_INTERFACE="$QUESTION_RET"
	fi
	# TODO: Check if network interface is available
	# TODO: Get server IP
	question "Server IP? []"
	SERVER_IP="$QUESTION_RET"
	# TODO: Check server IP
	# TODO: Get network address from interface
	question "Network address: []"
	CLIENT_SUBNET="$QUESTION_RET"
	# TODO: Check network address
	# TODO: Get netmask from interface
	question "Network mask: []"
	CLIENT_NETMASK="$QUESTION_RET"
	# TODO: Check netmask
	# TODO: Get netmask bits
	question "Network mask bits: []"
	CLIENT_BITS="$QUESTION_RET"
	# TODO: Check netmask bits
	# TODO: Get first IP
	question "First IP address for dynamic clients: []"
	CLIENT_START="$QUESTION_RET"
	# TODO: Check first IP
	# TODO: Get last IP
	question "Last IP address for dynamic clients: []"
	CLIENT_END="$QUESTION_RET"
	# TODO: Check last IP
}

ask_storage() {
	echo ""
	echo "== Storage configuration =="
	question "TFTP boot file storage path? [/tftpboot]"
	if [ -z "$QUESTION_RET" ]; then
		TFTPBOOT="/tftpboot"
	else
		TFTPBOOT="$QUESTION_RET"
	fi
	# TODO: Verify tftpboot
	question "NFS root path? [/nfsroot]"
	if [ -z "$QUESTION_RET" ]; then
		NFSROOT="/nfsroot"
	else
		NFSROOT="$QUESTION_RET"
	fi
	# TODO: Verify nfsroot
	question "Image storage path? [/data]"
	if [ -z "$QUESTION_RET" ]; then
		DATAPATH="/data"
	else
		DATAPATH="$QUESTION_RET"
	fi
	# TODO: Verify data path
	echo ""
}

save_options() {
	if [ ! -d /etc/openclone ]; then
		mkdir -p /etc/openclone
		if [ $? -ne 0 ]; then
			status_error "Couldn't create openclone configuration directory"
		fi
	fi
	cat << _EOF_ > /etc/openclone/installer.conf
CLIENT_INTERFACE=$CLIENT_INTERFACE
SERVER_IP=$SERVER_IP
CLIENT_SUBNET=$CLIENT_SUBNET
CLIENT_NETMASK=$CLIENT_NETMASK
CLIENT_BITS=$CLIENT_BITS
CLIENT_START=$CLIENT_START
CLIENT_END=$CLIENT_END
TFTPBOOT=$TFTPBOOT
NFSROOT=$NFSROOT
DATAPATH=$DATAPATH
_EOF_
	if [ $? -ne 0 ]; then
		status_error "Couldn't write installer configuration"
	fi
}

read_options() {
	. /etc/openclone/installer.conf
}

deb_update() {
	status_text "Updating your system"
	echo ""
	aptitude update
	if [ $? -eq 0 ]; then
		aptitude dist-upgrade
	else
		apt-get update
		if [ $? -eq 0 ]; then
			apt-get dist-upgrade
		else
			status_error "No apt package management tool found"
		fi
	fi
}

deb_install() {
	status_text "Installing packages"
	echo ""
	which aptitude > /dev/null
	if [ $? -eq 0 ]; then
		aptitude install "$@"
		if [ $? -ne 0 ]; then
			status_error "Aptitude failed"
		fi
	else
		which apt-get > /dev/null
		if [ $? -eq 0 ]; then
			apt-get install "$@"
			if [ $? -ne 0 ]; then
				status_error "Apt-get failed"
			fi
		else
			status_error "No apt package management tool found"
		fi
	fi
}

configure_dhcpd() {
	status_text "Configuring dhcpd..."
	# Configure DHCP Server
	cat << _EOF_ > /etc/dhcp3/dhcpd-openclone.conf
allow booting;
allow bootp;

shared-network lan {
	subnet $CLIENT_SUBNET netmask $CLIENT_NETMASK {
		range $CLIENT_START $CLIENT_END;
		filename "pxelinux.0";
	}
}
_EOF_
	if [ $? -ne 0 ]; then
		status_error "Couldn't write OpenClone dhcp configuration"
	fi
	# TODO: Only add line if not already there
	echo "include \"/etc/dhcp3/dhcpd-openclone.conf\";" >> /etc/dhcp3/dhcpd.conf
	if [ $? -ne 0 ]; then
		status_error "Couldn't write dhcp configuration"
	fi

	status_ok
	/etc/init.d/dhcp3-server restart
}

configure_tftpd() {
	status_text "Configuring tftpd..."
	sed -i -e "s/^\(tftp\s\+dgram\s\+udp\s\+wait\s\+root\s\+\/usr\/sbin\/in\.tftpd \/usr\/sbin\/in\.tftpd -s \)[^\w]\+\$/\1\/tftpboot/" /etc/inetd.conf
	status_ok
	/etc/init.d/openbsd-inetd restart
}

configure_nfs() {
	status_text "Configuring nfs export..."
	# Configure NFS export for root file system
	# TODO: Check if already exported
	echo "$NFSROOT $CLIENT_SUBNET/$CLIENT_BITS(ro,no_root_squash,async)" >> /etc/exports
	if [ $? -ne 0 ]; then
		status_error "Couldn't write export configuration"
	fi
	exportfs -rv
	if [ $? -ne 0 ]; then
		status_error "Couldn't export nfsroot"
	fi
	status_ok
}

setup_tftpboot() {
	status_text "Setting up tftp boot..."
	if [ ! -d "$TFTPBOOT" ]; then
		mkdir -p "$TFTPBOOT"
		if [ $? -ne 0 ]; then
			status_error "Couldn't create tftpboot directory"
		fi
	fi
	cp /usr/lib/syslinux/pxelinux.0 "$TFTPBOOT/pxelinux.0"
	if [ $? -ne 0 ]; then
		status_error "Couldn't copy pxelinux"
	fi
	cp /usr/lib/syslinux/menu.c32 "$TFTPBOOT/menu.c32"
	if [ $? -ne 0 ]; then
		status_error "Couldn't copy pxelinux"
	fi
	# Configure pxelinux
	if [ ! -d "$TFTPBOOT/pxelinux.cfg" ]; then
		mkdir "$TFTPBOOT/pxelinux.cfg"
		if [ $? -ne 0 ]; then
			status_error "Couldn't create pxelinux configuration directory"
		fi
	fi
	cat << _EOF_ > "$TFTPBOOT/pxelinux.cfg/default"
DEFAULT menu.c32
#DEFAULT openclone
PROMPT 0
MENU TITLE OpenClone BootMenu
LABEL openclone
  MENU DEFAULT
  MENU LABEL ^OpenClone
  KERNEL ubuntu/vmlinuz
  APPEND initrd=ubuntu/initrd.img root=/dev/nfs nfsroot=$SERVER_IP:$NFSROOT ip=dhcp ocws=http://$SERVER_IP:18080/OpenCloneEngineService/
_EOF_
	if [ $? -ne 0 ]; then
		status_error "Couldn't write default pxelinux configuration"
	fi
	status_ok
}

install_engine() {
	# Install openclone-engine
	cp -a "$SOURCE/openclone-engine" "$NFSROOT/tmp/openclone-engine"
	if [ $? -ne 0 ]; then
		status_error "Couldn't copy OpenClone-Engine"
	fi
	if [ "$OPENCLONE_DEV" = "true" ]; then
		DEV="True"
	else
		DEV="False"
	fi
	sed -i -e "s/^major = .*\$/major = $OPENCLONE_MAJOR/;s/^minor = .*\$/minor = $OPENCLONE_MINOR/;s/^patch = .*\$/patch = $OPENCLONE_PATCH/;s/^rev = .*\$/rev = '$OPENCLONE_REV'/;s/^dev = .*\$/dev = $DEV/" "$NFSROOT/tmp/openclone-engine/src/openclone/common/version.py"
	chroot "$NFSROOT" sh -c "cd /tmp/openclone-engine && python setup.py install"
	if [ $? -ne 0 ]; then
		status_error "Couldn't install OpenClone-Engine"
	fi
	rm -rf "$NFSROOT/tmp/openclone-engine"
}

setup_nfsroot() {
	status_text "Setting up nfs root..."
	if [ ! -d "$NFSROOT" ]; then
		mkdir -p "$NFSROOT"
		if [ $? -ne 0 ]; then
			status_error "Couldn't create nfsroot directory"
		fi
	fi
	status_ok

	echo "Installing Ubuntu system in nfsroot..."
	# Install base ubuntu system
	if [ ! -e "/usr/share/debootstrap/scripts/$DIST_SUITE" ]; then
		case "$DIST_SUITE" in
			hardy|intrepid|jaunty)
				if [ -e "/usr/share/debootstrap/scripts/gutsy" ]; then
					ln -s gutsy "/usr/share/debootstrap/scripts/$DIST_SUITE"
					if [ $? -ne 0 ]; then
						status_exit "Couldn't create new debootstrap scripts"
					fi
				else
					status_exit "Couldn't find necessary debootstrap scripts"
				fi
				;;
			*)
				status_exit "Unknown suite for debootstrap"
				;;
		esac
	fi
	OPTS="--include gpgv"
	if [ ! -f /usr/share/keyrings/ubuntu-archive-keyring.gpg ]; then
		wget -P /tmp "$DIST_MIRROR/pool/main/u/ubuntu-keyring/ubuntu-keyring_${KEYRING_VER}_all.deb"
		if [ $? -ne 0 ]; then
			status_error "Couldn't download Ubuntu-keyring"
		fi
		dpkg -i "/tmp/ubuntu-keyring_${KEYRING_VER}_all.deb"
		if [ $? -ne 0 ]; then
			status_error "Couldn't install Ubuntu-keyring"
		fi
		rm -f "/tmp/ubuntu-keyring_${KEYRING_VER}_all.deb"
	fi
	if [ -r /usr/share/keyrings/ubuntu-archive-keyring.gpg ]; then
		OPTS="$OPTS --keyring /usr/share/keyrings/ubuntu-archive-keyring.gpg"
	fi
	debootstrap --variant=minbase $OPTS "$DIST_SUITE" "$NFSROOT" "$DIST_MIRROR"
	if [ $? -ne 0 ]; then
		status_error "Couldn't install base system"
	fi

	# Configure /etc/hostname
	echo "$CLIENT_HOSTNAME" > "$NFSROOT/etc/hostname"
	if [ $? -ne 0 ]; then
		status_error "Couldn't set hostname"
	fi

	# Correct mtab
	rm -f "$NFSROOT/etc/mtab"
	ln -s ../proc/mounts "$NFSROOT/etc/mtab"
	if [ $? -ne 0 ]; then
		status_error "Couldn't create mtab"
	fi

	# Configure /etc/fstab
	cat << _EOF_ > "$NFSROOT/etc/fstab"
# /etc/fstab: static file system information.
#
# <file system> <mount point>   <type>  <options>       <dump>  <pass>
#proc            /proc           proc    defaults        0       0
#sys             /sys            sysfs   defaults        0       0
#/dev/nfs        /               nfs     defaults        0       1
#none            /var/run        tmpfs   defaults        0       0
#none            /var/lock       tmpfs   defaults        0       0
none            /tmp            tmpfs   defaults        0       0
none            /var/log        tmpfs   defaults        0       0
none            /var/tmp        tmpfs   defaults        0       0
_EOF_
	if [ $? -ne 0 ]; then
		status_error "Couldn't write fstab"
	fi

	# Update system
	cat << _EOF_ > "$NFSROOT/etc/apt/sources.list"
deb $DIST_MIRROR $DIST_SUITE main universe
deb $DIST_MIRROR $DIST_SUITE-updates main universe
deb http://security.ubuntu.com/ubuntu $DIST_SUITE-security main universe
_EOF_
	if [ $? -ne 0 ]; then
		status_error "Couldn't write sources.list"
	fi
	chroot "$NFSROOT" apt-get update
	chroot "$NFSROOT" apt-get -y dist-upgrade

	# Install kernel
	chroot "$NFSROOT" apt-get -y install linux-image
	if [ $? -ne 0 ]; then
		status_error "Couldn't install linux kernel"
	fi
	sed -i -e 's/BOOT=.*/BOOT=nfs/' "$NFSROOT/etc/initramfs-tools/initramfs.conf"
	if [ $? -ne 0 ]; then
		status_error "Couldn't modify initrd configuration"
	fi
	chroot "$NFSROOT" update-initramfs -k all -c
	if [ $? -ne 0 ]; then
		status_error "Couldn't update initrd"
	fi
	if [ ! -d "$TFTPBOOT/ubuntu" ]; then
		mkdir "$TFTPBOOT/ubuntu"
		if [ $? -ne 0 ]; then
			status_error "Couldn't create ubuntu directory"
		fi
	fi
	cp `ls -1 "$NFSROOT"/boot/vmlinuz-*-generic | tail -1` "$TFTPBOOT/ubuntu/vmlinuz"
	if [ $? -ne 0 ]; then
		status_error "Couldn't copy kernel"
	fi
	cp `ls -1 "$NFSROOT"/boot/initrd.img-*-generic | tail -1` "$TFTPBOOT/ubuntu/initrd.img"
	if [ $? -ne 0 ]; then
		status_error "Couldn't copy initrd image"
	fi

	# Clear root password
	chroot "$NFSROOT" passwd -d root

	# TODO: Install nfs-common?

	# TODO: Swapfile?
	#Package: dphys-swapfile
	#losetup /dev/loop0 /var/swap
	#swapon /dev/loop0

	# Install openclone-engine dependencies
	chroot "$NFSROOT" apt-get -y install python python-zsi net-tools hdparm partimage dmidecode lzop
	if [ $? -ne 0 ]; then
		status_error "Couldn't install OpenClone-Engine dependencies"
	fi
	wget -P "$NFSROOT/tmp" "http://udpcast.linux.lu/download/udpcast_${UDPCAST_VER}_i386.deb"
	if [ $? -ne 0 ]; then
		status_error "Couldn't download UDPcast"
	fi
	chroot "$NFSROOT" dpkg -i "/tmp/udpcast_${UDPCAST_VER}_i386.deb"
	if [ $? -ne 0 ]; then
		status_error "Couldn't install UDPcast"
	fi
	rm -f "$NFSROOT/tmp/udpcast_${UDPCAST_VER}_i386.deb"
	#wget -P /nfsroot/tmp http://ovh.dl.sourceforge.net/sourceforge/directio/directio-1.0.2.tgz
	#python setup.py build
	#chroot /nfsroot sh -c cd /tmp/directio-1.0.2 && python setup.py install
	#wget -P "$NFSROOT/tmp" http://free.nchc.org.tw/drbl-core/pool/drbl/stable/p/partclone/partclone_0.0.9-4_i386.deb
	#chroot "$NFSROOT" dpkg -i /tmp/partclone_0.0.9-4_i386.deb
	# TODO: Install hdparm form source?
	# TODO: Install partimage from source?

	install_engine

	if [ -x "$NFSROOT/usr/bin/openclone-engine" ]; then
		OCE="\/usr\/bin\/openclone-engine"
	elif [ -x "$NFSROOT/usr/local/bin/openclone-engine" ]; then
		OCE="\/usr\/local\/bin\/openclone-engine"
	else
		status_error "Couldn't detect openclone-engine path"
	fi

	sed -i -e "s/^exec \(.*\)\$/exec $OCE/" "$NFSROOT/etc/event.d/tty1"
	grep -q "^console owner$" "$NFSROOT/etc/event.d/tty1"
	if [ $? -ne 0 ]; then
		echo "console owner" >> "$NFSROOT/etc/event.d/tty1"
	fi

	# Cleanup
	chroot "$NFSROOT" apt-get clean
}

update_nfsroot() {
	chroot "$NFSROOT" apt-get update
	chroot "$NFSROOT" apt-get -y dist-upgrade
	cp `ls -1 "$NFSROOT"/boot/vmlinuz-*-generic | tail -1` "$TFTPBOOT/ubuntu/vmlinuz"
	if [ $? -ne 0 ]; then
		status_error "Couldn't copy kernel"
	fi
	cp `ls -1 "$NFSROOT"/boot/initrd.img-*-generic | tail -1` "$TFTPBOOT/ubuntu/initrd.img"
	if [ $? -ne 0 ]; then
		status_error "Couldn't copy initrd image"
	fi

	# Update openclone-engine dependencies
	# TODO: Check if update is needed
	wget -P "$NFSROOT/tmp" "http://udpcast.linux.lu/download/udpcast_${UDPCAST_VER}_i386.deb"
	if [ $? -ne 0 ]; then
		status_error "Couldn't download UDPcast"
	fi
	chroot "$NFSROOT" dpkg -i "/tmp/udpcast_${UDPCAST_VER}_i386.deb"
	if [ $? -ne 0 ]; then
		status_error "Couldn't install UDPcast"
	fi
	rm -f "$NFSROOT/tmp/udpcast_${UDPCAST_VER}_i386.deb"

	install_engine

	# Cleanup
	chroot "$NFSROOT" apt-get clean
}

setup_datapath() {
	status_text "Setting up data path..."
	if [ ! -d "$DATAPATH" ]; then
		mkdir -p "$DATAPATH"
	fi
	addgroup --system openclone
	if [ $? -ne 0 ]; then
		status_error "Couldn't create new UNIX group openclone"
	fi
	chown root:openclone "$DATAPATH"
	chmod g+w "$DATAPATH"
	status_ok
}

install_webservice() {
	echo "Installing OpenClone-Webservice..."

	# Install openclone-webservice dependencies
	# TODO: Check if update is needed
	wget -P /tmp "$DIST_MIRROR/pool/universe/s/sqlalchemy/python-sqlalchemy_${SQLALCHEMY_VER}_all.deb"
	if [ $? -ne 0 ]; then
		status_error "Couldn't download SQLAlchemy"
	fi
	dpkg -i "/tmp/python-sqlalchemy_${SQLALCHEMY_VER}_all.deb"
	if [ $? -ne 0 ]; then
		status_error "Couldn't install SQLAlchemy"
	fi
	rm -f "/tmp/python-sqlalchemy_${SQLALCHEMY_VER}_all.deb"
	wget -P /tmp "http://udpcast.linux.lu/download/udpcast_${UDPCAST_VER}_i386.deb"
	if [ $? -ne 0 ]; then
		status_error "Couldn't download UDPcast"
	fi
	dpkg -i "/tmp/udpcast_${UDPCAST_VER}_i386.deb"
	if [ $? -ne 0 ]; then
		status_error "Couldn't install UDPcast"
	fi
	rm -f "/tmp/udpcast_${UDPCAST_VER}_i386.deb"

	# Install openclone-webservice
	cp -a "$SOURCE/openclone-webservice" "/tmp/openclone-webservice"
	if [ $? -ne 0 ]; then
		status_error "Couldn't copy OpenClone-Webservice"
	fi
	if [ "$OPENCLONE_DEV" = "true" ]; then
		DEV="True"
	else
		DEV="False"
	fi
	sed -i -e "s/^major = .*\$/major = $OPENCLONE_MAJOR/;s/^minor = .*\$/minor = $OPENCLONE_MINOR/;s/^patch = .*\$/patch = $OPENCLONE_PATCH/;s/^rev = .*\$/rev = '$OPENCLONE_REV'/;s/^dev = .*\$/dev = $DEV/" "/tmp/openclone-webservice/src/openclone/common/version.py"
	(cd /tmp/openclone-webservice && python setup.py install)
	if [ $? -ne 0 ]; then
		status_error "Couldn't install OpenClone-Webservice"
	fi
	mv /tmp/openclone-webservice/init /etc/init.d/openclone-webservice
	if [ ! -x /etc/init.d/openclone-webservice ]; then
		chmod +x /etc/init.d/openclone-webservice
	fi
	update-rc.d openclone-webservice defaults
	rm -rf "/tmp/openclone-webservice"
}

configure_webservice() {
	echo "Configuring OpenClone-Webservice..."
	WEBSRV_MYSQL_PW="`perl -e 'print map{("a".."z","A".."Z",0..9)[int(rand(62))]}(1..16)'`"
	mysql --defaults-extra-file=/etc/mysql/debian.cnf --execute="GRANT SELECT, INSERT, UPDATE, DELETE ON openclone.* TO 'openclone-websrv'@'localhost' IDENTIFIED BY '$WEBSRV_MYSQL_PW';"
	if [ $? -ne 0 ]; then
		status_error "Couldn't create new MySQL user openclone-websrv"
	fi
	addgroup --system openclone
	if [ $? -ne 0 ]; then
		status_error "Couldn't create new UNIX group openclone"
	fi
	adduser --system --ingroup openclone --home /nonexistent --no-create-home openclone-websrv
	if [ $? -ne 0 ]; then
		status_error "Couldn't create new UNIX user openclone-websrv"
	fi
	if [ ! -d /etc/openclone ]; then
		mkdir -p /etc/openclone
	fi
	cat << _EOF_ > /etc/openclone/webservice.conf
[common]
pidfile=/var/run/openclone/webservice.pid

[database]
db_url=mysql://openclone-websrv:$WEBSRV_MYSQL_PW@localhost/openclone

[network]
multicast_interface=$CLIENT_INTERFACE

[logging]
logfile=/var/log/openclone/webservice.log
loglevel=info
keep_days=0

[storage]
data_path=$DATAPATH

[webservice]
listen_addr=
listen_port=18080
_EOF_
	unset WEBSRV_MYSQL_PW
	chmod g-w,o-rwx /etc/openclone/webservice.conf
	chgrp openclone /etc/openclone/webservice.conf
	if [ ! -d /var/log/openclone ]; then
		mkdir -p /var/log/openclone
		chgrp openclone /var/log/openclone
		chmod g+w /var/log/openclone
	fi
}

install_database() {
	echo "Installing OpenClone-Database..."
	mysql --defaults-extra-file=/etc/mysql/debian.cnf < "$SOURCE/openclone-database/openclone.sql"
	if [ $? -ne 0 ]; then
		status_error "Couldn't create database"
	fi
}

install_webinterface() {
	echo "Installing OpenClone-Webinterface..."
	cp -a "$SOURCE/openclone-webinterface" "/tmp/openclone-webinterface"
	if [ $? -ne 0 ]; then
		status_error "Couldn't copy OpenClone-Webinterface"
	fi
	sed -i -e "s/^\$major = .*;\$/\$major = $OPENCLONE_MAJOR;/;s/^\$minor = .*;\$/\$minor = $OPENCLONE_MINOR;/;s/^\$patch = .*;\$/\$patch = $OPENCLONE_PATCH;/;s/^\$rev = .*;\$/\$rev = '$OPENCLONE_REV';/;s/^\$dev = .*;\$/\$dev = $OPENCLONE_DEV;/" "/tmp/openclone-webinterface/src/functions/version_func.php"
	if [ -f /var/www/openclone-webinterface/config.php ]; then
		mv /var/www/openclone-webinterface/config.php /tmp/openclone-webinterface/src
		if [ $? -ne 0 ]; then
			status_error "Couldn't move old OpenClone-Webinterface configuration"
		fi
	fi
	if [ -d /var/www/openclone-webinterface ]; then
		rm -rf /var/www/openclone-webinterface
		if [ $? -ne 0 ]; then
			status_error "Couldn't remove old OpenClone-Webinterface"
		fi
	fi
	mv /tmp/openclone-webinterface/src /var/www/openclone-webinterface
	if [ $? -ne 0 ]; then
		status_error "Couldn't move OpenClone-Webinterface"
	fi
	mv /tmp/openclone-webinterface/openclone-webinterface-wrapper /usr/bin
	if [ $? -ne 0 ]; then
		status_error "Couldn't move OpenClone-Webinterface wrapper script"
	fi
	chmod +x /usr/bin/openclone-webinterface-wrapper
	if [ $? -ne 0 ]; then
		status_error "Couldn't change rights of OpenClone-Webinterface wrapper script"
	fi
	rm -rf /tmp/openclone-webinterface
}

configure_webinterface() {
	echo "Configureing OpenClone-Webinterface..."
	WEBIF_MYSQL_PW="`perl -e 'print map{("a".."z","A".."Z",0..9)[int(rand(62))]}(1..16)'`"
	mysql --defaults-extra-file=/etc/mysql/debian.cnf --execute="GRANT SELECT, INSERT, UPDATE, DELETE ON openclone.* TO 'openclone-webif'@'localhost' IDENTIFIED BY '$WEBIF_MYSQL_PW';"
	if [ $? -ne 0 ]; then
		status_error "Couldn't create new MySQL user"
	fi
	cat << _EOF_ > /var/www/openclone-webinterface/config.php
<?php
define('DB_URL', 'mysql://openclone-webif:$WEBIF_MYSQL_PW@localhost/openclone');

define('WRAPPERSCRIPT', '/usr/bin/openclone-webinterface-wrapper');
define('WRAPPERUSER', 'openclone-webif');
?>
_EOF_
	unset WEBIF_MYSQL_PW
	addgroup --system openclone
	if [ $? -ne 0 ]; then
		status_error "Couldn't create new UNIX group openclone"
	fi
	adduser --system --ingroup openclone --home /nonexistent --no-create-home openclone-webif
	if [ $? -ne 0 ]; then
		status_error "Couldn't create new UNIX user openclone-webif"
	fi
	# TODO: Change /etc/sudoers
	#www-data ALL=(openclone-webif) NOPASSWD: /usr/bin/openclone-webinterface-wrapper
}

create_user() {
	status_text "Creating OpenClone-Webinterface user..."
	mysql --defaults-extra-file=/etc/mysql/debian.cnf --execute="INSERT INTO user (name, password) VALUES ('admin', sha1('openclone'));" openclone
	if [ $? -ne 0 ]; then
		status_error "Couldn't create new OpenClone-Webinterface user"
	fi
	status_ok
}

case "$MODE" in
	install)
		ask_network
		ask_storage
		save_options
		deb_update
		if [ "$DISTRIB_ID" = "Debian" -a "$DISTRIB_RELEASE" = "5.0.1" ]; then
			deb_install dhcp3-server tftpd-hpa nfs-kernel-server debootstrap syslinux iproute python python-mysqldb python-xml apache2 php5 php5-mysql mysql-server
			wget -P /tmp "$DIST_MIRROR/pool/universe/z/zsi/python-zsi_${ZSI_VER}_all.deb"
			if [ $? -ne 0 ]; then
				status_error "Couldn't download ZSI"
			fi
			dpkg -i "/tmp/python-zsi_${ZSI_VER}_all.deb"
			if [ $? -ne 0 ]; then
				status_error "Couldn't install ZSI"
			fi
			rm -f "/tmp/python-zsi_${ZSI_VER}_all.deb"
			which aptitude > /dev/null
			if [ $? -eq 0 ]; then
				aptitude hold python-zsi
			fi
			echo python-zsi hold | dpkg --set-selections
		else
			deb_install dhcp3-server tftpd-hpa nfs-kernel-server debootstrap syslinux iproute python python-mysqldb python-zsi apache2 php5 php5-mysql mysql-server
		fi
		/etc/init.d/apache2 restart
		configure_dhcpd
		setup_tftpboot
		configure_tftpd
		setup_nfsroot
		configure_nfs
		setup_datapath
		install_webservice
		configure_webservice
		install_database
		install_webinterface
		configure_webinterface
		create_user
		/etc/init.d/openclone-webservice restart
		echo ""
		echo "  *** Installation finished ***"
		echo ""
		echo " Now you can login on http://$SERVER_IP/openclone-webinterface with"
		echo ""
		echo "  Username: admin"
		echo "  Password: openclone"
		echo ""
		echo " For security reasons, please change the password immediatelly."
		;;

	update)
		read_options
		deb_update
		setup_tftpboot
		NFSROOT_DISTRIB_ID="`sed -n "s/^DISTRIB_ID=\(.*\)\$/\1/p" $NFSROOT/etc/lsb-release`"
		NFSROOT_DISTRIB_CODENAME="`sed -n "s/^DISTRIB_CODENAME=\(.*\)\$/\1/p" $NFSROOT/etc/lsb-release`"
		if [ ! "$NFSROOT_DISTRIB_ID" = "Ubuntu" ]; then
			status_error "Your nfsroot uses an unknown distribution."
		fi
		if [ ! "$NFSROOT_DISTRIB_CODENAME" = "$DIST_SUITE" ]; then
			echo "Your nfsroot ($NFSROOT_DISTRIB_CODENAME) needs a reinstallation, because a new Ubuntu release ($DIST_SUITE) is available."
			echo "WARNING: All custom modifications will be dropped in the new nfsroot (complete reinstall)"
			echo "Nevertheless your old nfsroot will remain as backup in $NFSROOT-backup, which can be deleted after the reinstallation has finished."
			question_yn "Do you want to continue?"
			if [ "$QUESTION_RET" = "y" ]; then
				if [ -e "$NFSROOT-backup" ]; then
					echo "An older nfsroot backup resides in $NFSROOT-backup."
					question_yn "Do you want to remove it and continue with the backup?"
					if [ "$QUESTION_RET" = "y" ]; then
						rm -rf "$NFSROOT-backup"
						if [ $? -ne 0 ]; then
							status_error "Couldn't remove old nfsroot backup"
						fi
					else
						exit 0
					fi
				fi
				mv "$NFSROOT" "$NFSROOT-backup"
				if [ $? -ne 0 ]; then
					status_error "Couldn't backup old nfsroot"
				fi
				setup_nfsroot
			else
				exit 0
			fi
		else
			update_nfsroot
		fi
		install_webservice
		# TODO: Update database
		#update_database
		install_webinterface
		/etc/init.d/openclone-webservice restart
		echo ""
		echo "  *** Update finished ***"
		echo ""
		;;

	*)
		status_error "Unexpected mode"

esac

exit 0

