#
# This file is part of OpenClone.
#
# Copyright (C) 2009  David Gnedt
#
# OpenClone is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenClone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with OpenClone.  If not, see <http://www.gnu.org/licenses/>.
#

import base64
import logging
import re
import sys
import os
import subprocess
import time

import client
from webservice.engineclient import ns0
import exception
from ..common import version
from compression import gzip
from compression import lzo
from imaging import dd
from imaging import directswap
from imaging import partimage
from ..common.transfer import udpcast
from misc import hash
from partition import directpart

LOG_DIR = '/var/log/openclone'
LOCK_DIR = '/var/run/openclone'
LOG_FILE = '%s/engine.log' % LOG_DIR
LOCK_FILE = '%s/engine.pid' % LOCK_DIR

def main():
    silent_exit = False
    url = None
    
    # Setup logging
    if not os.path.exists(LOG_DIR):
        old_mask = os.umask(002)
        os.mkdir(LOG_DIR)
        os.umask(old_mask)
    
    root_logger = logging.getLogger()
    root_logger.setLevel(logging.NOTSET)
    formatter = logging.Formatter('%(asctime)s %(name)s %(levelname)s %(message)s')
    # TODO: Configuration?
    handler = logging.FileHandler(LOG_FILE)
    handler.setFormatter(formatter)
    root_logger.addHandler(handler)
    handler = logging.StreamHandler(sys.stderr)
    handler.setFormatter(formatter)
    root_logger.addHandler(handler)
    
    logger = logging.getLogger('main')
    
    try:
        logger.info('OpenClone Engine ' + version.version_long_str)
        
        # Check lock file
        if not os.path.exists(LOCK_DIR):
            old_mask = os.umask(002)
            os.mkdir(LOCK_DIR)
            os.umask(old_mask)
        
        my_pid = os.getpid()
        
        if os.path.exists(LOCK_FILE):
            lock_file = open(LOCK_FILE)
            pid_str = lock_file.read()
            lock_file.close()
            
            try:
                pid = int(pid_str)
                if os.path.exists('/proc/%d' % pid):
                    cmdline_file = open('/proc/%d/cmdline' % pid)
                    cmdline = cmdline_file.read()
                    cmdline_file.close()
                    argv = cmdline.split('\x00')
                    
                    my_cmdline_file = open('/proc/%d/cmdline' % my_pid)
                    my_cmdline = my_cmdline_file.read()
                    my_cmdline_file.close()
                    my_argv = my_cmdline.split('\x00')
                    
                    if argv[0] == my_argv[0] and argv[1] == my_argv[1]:
                        raise Exception('OpenClone Engine is already running.')
            
            except ValueError, e:
                logger.warn('Invalid lock file')
            
            # TODO: Upload log file for debugging?
            
            raw_input('Stale lock file found. Press ENTER to continue.')
            logger.warn('Deleting stale lock file "%s".' % LOCK_FILE)
            os.unlink(LOCK_FILE)
        
        # Write lock file
        lock_file = open(LOCK_FILE, 'w')
        lock_file.write('%d' % my_pid)
        lock_file.close()
        
        # TODO: Configuration?, send with logon/register response
        #root_logger.setLevel(loglevels.get(config.get('logging', 'loglevel'), logging.NOTSET))
        
        if len(sys.argv) > 1:
            url = sys.argv[1]
            logger.info('Found command line parameter: %s' % url)
        
        else:
            f = open('/proc/cmdline')
            for line in f:
                for elem in line.split():
                    # Ignore options without parameter
                    if elem.find('=') != -1:
                        key, value = elem.split('=')
                        if key == 'ocws':
                            url = value
                            logger.info('Found kernel option ocws: %s' % url)
            
            f.close()
        
        if url is None:
            url = 'http://localhost:18080/OpenCloneEngineService/'
            logger.info('Using default url: %s' % url)
        
        if url[-1] != '/':
            url = url + '/'
        
        c = client.Client(url)
        try:
            logger.info('Logging on...')
            c.logon()
            logger.info('Logon successful')
        
        except exception.LogonException:
            logger.warn('Logon failed. Trying to register...')
            c.register()
            logger.info('Registration successful')
        
        while 1:
            logger.info('Requesting next operation...')
            operation = c.nextOperation()
            
            logger.info('Next operation: %s' % str(type(operation.typecode))[8:-2].split('.')[-1])
            
            # IdleOperation
            if isinstance(operation.typecode, ns0.IdleOperation_Def):
                logger.info('Sleeping for %f sec' % (operation.get_attribute_sleeptime()/1000.0))
                time.sleep(operation.get_attribute_sleeptime()/1000.0)
            
            # ImageOperation
            elif isinstance(operation.typecode, ns0.ImageOperation_Def):
                logger.info('Imaging in %s mode (local: %s  remote: %s  program: %s)' % (operation.get_attribute_mode(), operation.get_attribute_address(), operation.get_attribute_url(), operation.get_attribute_program()))
                mode = operation.get_attribute_mode()
                if mode != 'backup' and mode != 'restore':
                    raise exception.NextOperationException('Unknown imaging mode "%s"' % mode)
                
                result = re.match('^(?P<protocol>[a-zA-Z][a-zA-Z0-9+-\.]*)://(?P<host>\S+?)(?::?|:(?P<port>\d+))/$', operation.get_attribute_url())
                if result is None:
                    raise exception.NextOperationException('Wrong URL format')
                
                res = result.groupdict()
                
                logger.debug('Protocol: %s' % res['protocol'])
                if res['protocol'] == 'udpcast':
                    if res['port'] is None:
                        res['port'] = 9000
                    
                    if mode == 'backup':
                        proc_transfer = udpcast.UDPcastSender(port=int(res['port']))
                    
                    elif mode == 'restore':
                        proc_transfer = udpcast.UDPcastReceiver(port=int(res['port']))
                
                else:
                    raise exception.NextOperationException('Unknown transfer protocol "%s"' % res['protocol'])
                
                # TODO: Configuration file?
                hash_algo = 'none' #'sha1'
                if hash_algo == 'none':
                    proc_hash = proc_transfer
                
                else:
                    if mode == 'backup':
                        proc_hash = hash.Hash(hash_algo, output=proc_transfer.getStdin())
                    
                    elif mode == 'restore':
                        proc_hash = hash.Hash(hash_algo, input=proc_transfer.getStdout())
                
                compression = operation.get_attribute_compression()
                logger.debug('Compression: %s' % compression)
                if compression == 'none':
                    proc_compression = proc_hash
                
                elif compression == 'gzip':
                    if mode == 'backup':
                        proc_compression = gzip.GzipCompression(gzip.MODE_COMPRESS, level=1, output=proc_hash.getStdin())
                    
                    elif mode == 'restore':
                        proc_compression = gzip.GzipCompression(gzip.MODE_DECOMPRESS, level=1, input=proc_hash.getStdout())
                
                elif compression == 'lzo':
                    if mode == 'backup':
                        proc_compression = lzo.LZOCompression(lzo.MODE_COMPRESS, level=1, output=proc_hash.getStdin())
                    
                    elif mode == 'restore':
                        proc_compression = lzo.LZOCompression(lzo.MODE_DECOMPRESS, level=1, input=proc_hash.getStdout())
                
                else:
                    raise exception.NextOperationException('Unknown compression "%s"' % compression)
                
                program = operation.get_attribute_program()
                logger.debug('Program: %s' % program)
                if program == 'dd':
                    if mode == 'backup':
                        proc_imaging = dd.DD(dd.MODE_BACKUP, input=operation.get_attribute_address(), output=proc_compression.getStdin())
                    
                    elif mode == 'restore':
                        proc_imaging = dd.DD(dd.MODE_RESTORE, input=proc_compression.getStdout(), output=operation.get_attribute_address())
                    
                    else:
                        raise exception.NextOperationException('Unknown imaging mode "%s"' % mode)
                
                elif program == 'directswap':
                    if mode == 'backup':
                        proc_imaging = directswap.DirectSwap(directswap.MODE_BACKUP, input=operation.get_attribute_address(), output=proc_compression.getStdin())
                    
                    elif mode == 'restore':
                        proc_imaging = directswap.DirectSwap(directswap.MODE_RESTORE, input=proc_compression.getStdout(), output=operation.get_attribute_address())
                
                elif program == 'partimage':
                    if mode == 'backup':
                        proc_imaging = partimage.PartImage(partimage.MODE_BACKUP, input=operation.get_attribute_address(), output=proc_compression.getStdin())
                    
                    elif mode == 'restore':
                        proc_imaging = partimage.PartImage(partimage.MODE_RESTORE, input=proc_compression.getStdout(), output=operation.get_attribute_address())
                
                else:
                    raise exception.NextOperationException('Unknown imaging program "%s"' % program)
                
                nextupdate = time.time()
                
                while proc_imaging.isAlive():
                    status_imaging = proc_imaging.getStatus()
                    status_transfer = proc_transfer.getStatus()
                    
                    imaging_rate = '-'
                    imaging_percent = '-'
                    transfer_bytes = '-'
                    
                    if 'rate' in status_imaging:
                        imaging_rate = status_imaging['rate']
                    
                    if 'percent' in status_imaging:
                        imaging_percent = status_imaging['percent']
                    
                    if 'bytes' in status_transfer:
                        transfer_bytes = status_transfer['bytes']
                    
                    if mode == 'backup':
                        imaging_size = '-'
                        transfer_rexmits = '-'
                        transfer_slice = '-'
                        transfer_timeouts = '-'
                        
                        if 'size' in status_imaging:
                            imaging_size = status_imaging['size']
                        
                        if 'rexmits' in status_transfer:
                            transfer_rexmits = status_transfer['rexmits']
                        
                        if 'slice' in status_transfer:
                            transfer_slice = status_transfer['slice']
                        
                        if 'timeouts' in status_transfer:
                            transfer_timeouts = status_transfer['timeouts']
                        
                        sys.stdout.write('S:%5s R:%9s P:%4s b=%12s rx=%16s s=%4s to=%s \r' % (imaging_size, imaging_rate, imaging_percent, transfer_bytes, transfer_rexmits, transfer_slice, transfer_timeouts))
                    
                    elif mode == 'restore':
                        transfer_rate = '-'
                        
                        if 'rate' in status_transfer:
                            transfer_rate = status_transfer['rate']
                        
                        sys.stdout.write('R:%9s P:%4s b=%12s r=%11s \r' % (imaging_rate, imaging_percent, transfer_bytes, transfer_rate))
                    
                    if time.time() >= nextupdate:
                        kw = {}
                        try:
                            kw['percentage'] = float(imaging_percent.rstrip('%'))
                        
                        except ValueError:
                            pass
                        
                        if imaging_rate is not None and imaging_rate != '' and imaging_rate != '-':
                            kw['speed'] = imaging_rate
                        
                        if len(kw) > 0:
                            nextupdate = c.statusUpdate(operation.get_attribute_id(), operation.get_attribute_context_id(), **kw)/1000.0 + time.time()
                    
                    sys.stdout.flush()
                    proc_imaging.wait(1)
                
                logger.debug('proc_imaging exited')
                
                # TODO: Check if following closes are really needed
                
                if proc_compression is not proc_hash:
                    #if proc_compression.getStdin() is not None and not proc_compression.getStdin().closed:
                    if mode == 'backup':
                        proc_compression.getStdin().close()
                    
                    #if proc_compression.getStdout() is not None and not proc_compression.getStdout().closed:
                    elif mode == 'restore':
                        proc_compression.getStdout().close()
                    
                    proc_compression.wait()
                    logger.debug('proc_compression exited')
                
                if proc_hash is not proc_transfer:
                    if mode == 'backup':
                        logger.debug('closing hash stdin')
                        proc_hash.getStdin().close()
                    
                    elif mode == 'restore':
                        logger.debug('closing hash stdout')
                        proc_hash.getStdout().close()
                    
                    proc_hash.wait()
                    logger.debug('proc_hash exited')
                
                #if proc_transfer.getStdin() is not None and not proc_transfer.getStdin().closed:
                if mode == 'backup':
                    proc_transfer.getStdin().close()
                
                elif mode == 'restore':
                    proc_transfer.getStdout().close()
                
                proc_transfer.wait()
                logger.debug('proc_transfer exited')
                
                status_imaging = proc_imaging.getStatus()
                if 'success' not in status_imaging or not status_imaging['success']:
                    raise exception.NextOperationException('Imaging failed\nstatus: %s' % status_imaging)
                
                if proc_hash is not proc_transfer:
                    status_hash = proc_hash.getStatus()
                    if 'success' not in status_hash or not status_hash['success']:
                        raise exception.NextOperationException('Hash failed\nstatus: %s' % status_hash)
                
                if proc_compression is not proc_hash:
                    status_compression = proc_compression.getStatus()
                    if 'success' not in status_compression or not status_compression['success']:
                        raise exception.NextOperationException('Compression failed\nstatus: %s' % status_compression)
                
                status_transfer = proc_transfer.getStatus()
                if 'success' not in status_transfer or not status_transfer['success']:
                    raise exception.NextOperationException('Transfer failed\nstatus: %s' % status_transfer)
                
                logger.info('Imaging finished')
                
                c.statusUpdate(operation.get_attribute_id(), operation.get_attribute_context_id(), status=0)
            
            # PartitionOperation
            elif isinstance(operation.typecode, ns0.PartitionOperation_Def):
                logger.info('Partitioning %s using %s' % (operation.get_attribute_address(), operation.get_attribute_program()))
                program = operation.get_attribute_program()
                if program == 'direct':
                    partitiontable = operation.Partitiontable
                    if isinstance(partitiontable.typecode, ns0.MBRPartitiontable_Def):
                        direct = directpart.DirectPart(operation.get_attribute_address())
                        
                        kw = {}
                        
                        if partitiontable.get_attribute_extended() is not None:
                            kw['extended'] = base64.b64decode(partitiontable.get_attribute_extended())
                        
                        if operation.get_attribute_restore_bootloader():
                            kw['bootloader'] = base64.b64decode(partitiontable.get_attribute_bootloader())
                        
                        if operation.get_attribute_restore_unused():
                            kw['unused'] = base64.b64decode(partitiontable.get_attribute_unused())
                        
                        direct.writePartitiontable(base64.b64decode(partitiontable.get_attribute_disk_signature()), base64.b64decode(partitiontable.get_attribute_partitions()), **kw)
                        
                        c.statusUpdate(operation.get_attribute_id(), operation.get_attribute_context_id(), status=0)
                    
                    else:
                        raise exception.NextOperationException('Unknown partitiontable type "%s"' % type(partitiontable.typecode))
                
                else:
                    raise exception.NextOperationException('Unknown partitioning program "%s"' % program)
                
                logger.info('Partitioning finished')
                # TODO: Update devices in database...
            
            # ShutdownOperation
            elif isinstance(operation.typecode, ns0.ShutdownOperation_Def):
                logger.info('Shuting down in %s mode' % operation.get_attribute_mode())
                if operation.get_attribute_mode() == 'poweroff':
                    c.statusUpdate(operation.get_attribute_id(), operation.get_attribute_context_id(), status=0)
                    try:
                        c.logoff()
                    
                    except exception.LogoffException:
                        logger.error('Logoff failed')
                    
                    # TODO: Get path (configuration?)
                    args = ['/sbin/shutdown', '-P', 'now']
                    logger.debug('Exec: %s' % ' '.join(args))
                    proc = subprocess.Popen(args, close_fds=True)
                    proc.wait()
                    logger.info('Poweroff requested. Exiting...')
                    silent_exit = True
                    break
                
                elif operation.get_attribute_mode() == 'reboot':
                    c.statusUpdate(operation.get_attribute_id(), operation.get_attribute_context_id(), status=0)
                    try:
                        c.logoff()
                    
                    except exception.LogoffException:
                        logger.error('Logoff failed')
                    
                    # TODO: Get path (configuration?)
                    args = ['/sbin/shutdown', '-r', 'now']
                    logger.debug('Exec: %s' % ' '.join(args))
                    proc = subprocess.Popen(args, close_fds=True)
                    proc.wait()
                    logger.info('Reboot requested. Exiting...')
                    silent_exit = True
                    break
                
                elif operation.get_attribute_mode() == 'exit':
                    c.statusUpdate(operation.get_attribute_id(), operation.get_attribute_context_id(), status=0)
                    try:
                        c.logoff()
                    
                    except exception.LogoffException:
                        logger.error('Logoff failed')
                    
                    logger.info('Exit requested. Exiting...')
                    silent_exit = True
                    break
                
                else:
                    raise exception.NextOperationException('Unknown shutdown operation mode "%s"' % operation.get_attribute_mode())
            
            else:
                raise exception.NextOperationException('Unknown operation type %s' % type(operation.typecode))
    
    except Exception, e:
        logger.exception(e)
        raise
    
    finally:
        if silent_exit:
            os.unlink(LOCK_FILE)
            
        else:
            logger.error('Unexpectedly reached end of code. Exiting...')
        
        # TODO: Cleanup threads

if __name__ == "__main__":
    main()
