#
# This file is part of OpenClone.
#
# Copyright (C) 2009  David Gnedt
#
# OpenClone is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenClone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with OpenClone.  If not, see <http://www.gnu.org/licenses/>.
#

from __future__ import with_statement
import logging
import os
import re
import signal
import subprocess
import threading

from ...common import utils

#
# Example
#
# $ partimage --fully-batch gui=no --volume=0 --nombr --nocheck --nodesc --compress=0 save /dev/sda2 stdout > /dev/null
# Volume size: 0 bytes (0 MiB)
# partimage: status: initializing the operation.
# partimage: status: Partimage: 0.6.1
# partimage: status: Image type: NONE
# partimage: status: Saving partition to the image file...
# partimage: status: reading partition properties
# partimage: status: writing header
# partimage: status: copying used data blocks
# File Name    Size     T:Elapsed/Estimated  Rate/min     Progress
# stdout       S:1.55G  T:00:01:48/00:01:40  R:1.10G/min  P: 51%
# stdout       S:3.83G  T:00:03:24/00:00:00  R:1.13G/min  P:100%
# 
# partimage: status: commiting buffer cache to disk.
# partimage: Success [OK]
# partimage:  Operation  successfully finished:
# 
# Time elapsed: 3m:24sec
# Speed: 1.13 GiB/min
# Data copied: 3.83 GiB

MODE_BACKUP, MODE_RESTORE = range(2)

logger = logging.getLogger('partimage')

class PartImage(threading.Thread):
    proc = None
    mode = None
    status = None
    status_lock = None
    
    def __init__(self, mode, **kw):
        threading.Thread.__init__(self)
        
        # TODO: Get path (configuration?)
        args = ['/usr/sbin/partimage', '--fully-batch', 'gui=no']
        env = dict(os.environ)
        
        # Remove LANG environment variable to force default language
        if 'LANG' in env:
            del env['LANG']
        
        if mode is MODE_BACKUP:
            if 'input' not in kw:
                raise Exception('Keyword input expected in backup mode')
            
            if 'output' not in kw:
                kw['output'] = subprocess.PIPE
            
            args.append('--volume=0')
            args.append('--nombr')
            args.append('--nocheck')
            args.append('--nodesc')
            args.append('--compress=0')
            args.append('save')
            args.append(kw['input'])
            args.append('stdout')
            
            logger.debug('Exec: %s' % ' '.join(args))
            self.proc = subprocess.Popen(args, env=env, stdout=kw['output'], stderr=subprocess.PIPE, close_fds=True)
        
        elif mode is MODE_RESTORE:
            if 'output' not in kw:
                raise Exception('Keyword output expected in restore mode')
            
            if 'input' not in kw:
                kw['input'] = subprocess.PIPE
            
            args.append('restore')
            args.append(kw['output'])
            args.append('stdin')
            
            logger.debug('Exec: %s' % ' '.join(args))
            self.proc = subprocess.Popen(args, env=env, stdin=kw['input'], stderr=subprocess.PIPE, close_fds=True)
        
        else:
            raise Exception('Unknown mode')
        
        self.mode = mode
        self.status = {}
        self.status_lock = threading.RLock()
        
        self.start()
    
    def run(self):
        success = False
        for line in utils.universal_readline_iter(self.proc.stderr, seekable=False):
            line = line.rstrip('\n\r')
            
            if line == 'partimage: Success [OK]':
                success = True
            
            result = re.match('^\t\t\t\b\bT:(?P<time_elapsed>\d{2}:\d{2}:\d{2})/(?P<time_remaining>\d{2}:\d{2}:\d{2})  R: {0,3}(?P<rate>\d{1,4}(?:\.\d{2})?[bKMGT]/min)  P: {0,2}(?P<percent>\d{1,3}%)$', line)
            if result is not None:
                res = result.groupdict()
                with self.status_lock:
                    self.status['time_elapsed'] = res['time_elapsed']
                    self.status['time_remaining'] = res['time_remaining']
                    self.status['rate'] = res['rate']
                    self.status['percent'] = res['percent']
            
            else:
                if self.mode is MODE_BACKUP:
                    result = re.match('^stdout       S: {0,3}(?P<size>\d{1,4}(?:\.\d{2})?[bKMGT]) $', line)
                    if result is not None:
                        res = result.groupdict()
                        with self.status_lock:
                            self.status['size'] = res['size']
                
                elif self.mode is MODE_RESTORE:
                    result = re.match('^stdin        None $', line)
                
                if result is None:
                    logger.debug('out: %s' % line)
        
        ret = self.proc.wait()
        if ret != 0 and success:
            success = False
        
        with self.status_lock:
            self.status['return'] = ret
            self.status['success'] = success
        
        logger.debug('success %s' % success)
    
    def getStdin(self):
        return self.proc.stdin
    
    def getStdout(self):
        return self.proc.stdout
    
    def getStatus(self):
        with self.status_lock:
            return dict(self.status)
    
    def wait(self, timeout=None):
        self.join(timeout)
        # Kill process if thread ended and process is still running
        if not self.isAlive() and self.proc.poll() is None:
            logger.warn('partimage thread ended without process')
            os.kill(self.proc.pid, signal.SIGTERM)
