#
# This file is part of OpenClone.
#
# Copyright (C) 2009  David Gnedt
#
# OpenClone is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenClone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with OpenClone.  If not, see <http://www.gnu.org/licenses/>.
#

#
# Example
#
# $ dd if=/dev/zero count=1000
# 1000+0 records in
# 1000+0 records out
# 512000 bytes (512 kB) copied, 0.643488 s, 796 kB/s

from __future__ import with_statement
import logging
import os
import re
import signal
import subprocess
import threading

MODE_BACKUP, MODE_RESTORE = range(2)

logger = logging.getLogger('dd')

class DD(threading.Thread):
    proc = None
    mode = None
    status = None
    status_lock = None
    
    def __init__(self, mode, **kw):
        threading.Thread.__init__(self)
        
        # TODO: Get path (configuration?)
        args = ['/bin/dd']
        env = dict(os.environ)
        
        # Remove LANG environment variable to force default language
        if 'LANG' in env:
            del env['LANG']
        
        if mode is MODE_BACKUP:
            if 'input' not in kw:
                raise Exception('Keyword input expected in backup mode')
            
            if 'output' not in kw:
                kw['output'] = subprocess.PIPE
            
            args.append('if=%s' % kw['input'])
            
            logger.debug('Exec: %s' % ' '.join(args))
            self.proc = subprocess.Popen(args, env=env, stdout=kw['output'], stderr=subprocess.PIPE, close_fds=True)
        
        elif mode is MODE_RESTORE:
            if 'output' not in kw:
                raise Exception('Keyword output expected in restore mode')
            
            if 'input' not in kw:
                kw['input'] = subprocess.PIPE
            
            args.append('of=%s' % kw['output'])
            
            logger.debug('Exec: %s' % ' '.join(args))
            self.proc = subprocess.Popen(args, env=env, stdin=kw['input'], stderr=subprocess.PIPE, close_fds=True)
        
        else:
            raise Exception('Unknown mode')
        
        self.mode = mode
        self.status = {}
        self.status_lock = threading.RLock()
        
        self.start()
    
    def run(self):
        success = False
        
        # TODO: Poll information from dd
        
        for line in self.proc.stderr:
            line = line.rstrip('\n\r')
            
            result = re.match('^(?P<size>\d+) bytes \(\d+(?:\.\d+) [kMGT]?B\) copied, (?P<time_elapsed>\d+(?:\.\d+)?) s, (?P<rate>\d+(?:\.\d+)? [kMGT]?B/s)$', line)
            if result is not None:
                success = True
                res = result.groupdict()
                with self.status_lock:
                    self.status['size'] = res['size']
                    self.status['time_elapsed'] = res['time_elapsed']
                    self.status['rate'] = res['rate']
            #        self.status['time_remaining'] = res['time_remaining']
            #        self.status['percent'] = res['percent']
            
            else:
                result = re.match('^\d+\+\d+ records (?:in|out)$', line)
                if result is None:
                    logger.debug('out: %s' % line)
        
        ret = self.proc.wait()
        if ret != 0 and success:
            success = False
        
        with self.status_lock:
            self.status['return'] = ret
            self.status['success'] = success
        
        logger.debug('success %s' % success)
    
    def getStdin(self):
        return self.proc.stdin
    
    def getStdout(self):
        return self.proc.stdout
    
    def getStatus(self):
        with self.status_lock:
            return dict(self.status)
    
    def wait(self, timeout=None):
        self.join(timeout)
        # Kill process if thread ended and process is still running
        if not self.isAlive() and self.proc.poll() is None:
            logger.warn('dd thread ended without process')
            os.kill(self.proc.pid, signal.SIGTERM)
