#
# This file is part of OpenClone.
#
# Copyright (C) 2009  David Gnedt
#
# OpenClone is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenClone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with OpenClone.  If not, see <http://www.gnu.org/licenses/>.
#

import base64

from webservice.engineclient import *
import exception
from partition import directpart
from sysinfo import sysinfo

class Client:
    client = None
    mac = None
    ip = None
    
    def __init__(self, url):
        self.client = EngineClient(url)
    
    def __get_mac_ip(self):
        for dev in sysinfo.getEthernetInterfaces():
            if dev.ip is not None:
                return dev.mac, dev.ip
        
        raise Exception('Interface detection failed')
    
    def __get_cpu_devices(self, new_dev):
        devices = []
        for cpu in sysinfo.getProcessors():
            dev = new_dev()
            dev.set_attribute_processor_no(int(cpu['processor']))
            #dev.set_attribute_processor_no(int(cpu['physical id']))
            
            if 'vendor_id' in cpu:
                dev.set_attribute_vendor_id(cpu['vendor_id'])
            
            if 'model name' in cpu:
                dev.set_attribute_model_name(cpu['model name'])
            
            if 'cpu MHz' in cpu:
                dev.set_attribute_mhz(float(cpu['cpu MHz']))
            
            if 'bogomips' in cpu:
                dev.set_attribute_bogomips(float(cpu['bogomips']))
            
            if 'cache size' in cpu:
                dev.set_attribute_cache(int(cpu['cache size'].split()[0]))
            
            if 'core id' in cpu:
                dev.set_attribute_core_id(int(cpu['core id']))
            
            if 'cpu cores' in cpu:
                dev.set_attribute_cpu_cores(int(cpu['cpu cores']))
            
            devices.append(dev)
        
        return devices
    
    def __get_ram_devices(self, new_dev):
        mem = sysinfo.getMemory()
        dev = new_dev()
        dev.set_attribute_size(int(mem['MemTotal'].split()[0]))
        return [dev]
    
    def __get_harddisk_devices(self, new_dev):
        devices = []
        for hd in sysinfo.getHardDisks():
            dev = new_dev()
            dev.set_attribute_address(hd.address)
            dev.set_attribute_size(hd.size)
            dev.set_attribute_sector_size(hd.sector_size)
            
            if hd.model_no is not None:
                dev.set_attribute_model_no(hd.model_no)
            
            if hd.serial_no is not None:
                dev.set_attribute_serial_no(hd.serial_no)
            
            if hd.firmware_rev is not None:
                dev.set_attribute_firmware_rev(hd.firmware_rev)
            
            if hd.wwn is not None:
                dev.set_attribute_wwn(hd.wwn)
            
            if hd.cylinder is not None:
                dev.set_attribute_cylinder(hd.cylinder)
            
            if hd.heads is not None:
                dev.set_attribute_heads(hd.heads)
            
            if hd.sectors is not None:
                dev.set_attribute_sectors(hd.sectors)
            
            try:
                dp = directpart.DirectPart(hd.address)
                mbr = dp.readMBR()
                dev.Partitiontable = self.client.new_harddisk_mbr_partitiontable()
                dev.Partitiontable.set_attribute_disk_signature(base64.b64encode(mbr[0]))
                dev.Partitiontable.set_attribute_bootloader(base64.b64encode(mbr[1]))
                dev.Partitiontable.set_attribute_partitions(base64.b64encode(mbr[2]))
                if mbr[3] is not None:
                    dev.Partitiontable.set_attribute_extended(base64.b64encode(mbr[3]))
                
                dev.Partitiontable.parts = []
                
                first_part_lba = None
                
                for partition in hd.partitions:
                    p = sysinfo.getPartition(hd.address, partition)
                    if first_part_lba is None or p.start_lba < first_part_lba:
                        first_part_lba = p.start_lba
                    
                    part = self.client.new_partitiontable_mbr_partition()
                    if p.fs is not None:
                        part.set_attribute_fs(p.fs)
                    
                    if p.os is not None:
                        part.set_attribute_os(p.os)
                    
                    part.set_attribute_no(p.no)
                    part.set_attribute_record_type(p.record_type)
                    part.set_attribute_bootable(p.bootable)
                    part.set_attribute_partition_type(p.partition_type)
                    part.set_attribute_start_lba(p.start_lba)
                    part.set_attribute_sectors(p.sectors)
                    part.set_attribute_start_cylinder(p.start_cylinder)
                    part.set_attribute_start_head(p.start_head)
                    part.set_attribute_start_sector(p.start_sector)
                    part.set_attribute_end_cylinder(p.end_cylinder)
                    part.set_attribute_end_head(p.end_head)
                    part.set_attribute_end_sector(p.end_sector)
                    dev.Partitiontable.Parts.append(part)
                
                if first_part_lba is not None:
                    # TODO: Implement limit for unused (maybe 1mb)
                    dev.Partitiontable.set_attribute_unused(base64.b64encode(dp.readData(512, first_part_lba*hd.sector_size)))
            
            except directpart.MBRNotFoundException:
                pass
            
            devices.append(dev)
        
        return devices
    
    def __get_hostid(self, new_hostid):
        if self.mac is None or self.ip is None:
            self.mac, self.ip = self.__get_mac_ip()
        
        hostid = new_hostid()
        hostid.set_attribute_mac(self.mac)
        return hostid
    
    def __get_host(self, new_host):
        host = new_host()
        host.Hostid = self.__get_hostid(host.new_hostid)
        if self.mac is None or self.ip is None:
            self.mac, self.ip = self.__get_mac_ip()
        
        host.set_attribute_ip(self.ip)
        serial_no = sysinfo.getSerialNo()
        if len(serial_no) > 0:
            host.set_attribute_serial_no(serial_no)
        
        host.Devices = []
        host.Devices.extend(self.__get_cpu_devices(self.client.new_host_cpu_device))
        host.Devices.extend(self.__get_ram_devices(self.client.new_host_ram_device))
        host.Devices.extend(self.__get_harddisk_devices(self.client.new_host_harddisk_device))
        return host
    
    def register(self):
        host = self.__get_host(self.client.new_register_host)
        status = self.client.register(host)
        if status.get_attribute_value() != 0:
            raise exception.RegisterException()
    
    def logon(self):
        host = self.__get_host(self.client.new_logon_host)
        status = self.client.logon(host)
        if status.get_attribute_value() != 0:
            raise exception.LogonException()
    
    def logoff(self):
        hostid = self.__get_hostid(self.client.new_logoff_hostid)
        status = self.client.logoff(hostid)
        if status.get_attribute_value() != 0:
            raise exception.LogoffException()
        
    def nextOperation(self):
        hostid = self.__get_hostid(self.client.new_nextoperation_hostid)
        status, operation = self.client.nextOperation(hostid)
        if status.get_attribute_value() != 0:
            raise exception.NextOperationException()
        
        return operation
    
    def statusUpdate(self, operation_id, context_id, **kw):
        statusupdate = self.client.new_statusupdate()
        statusupdate.set_attribute_operation_id(operation_id)
        statusupdate.set_attribute_context_id(context_id)
        statusupdate.Hostid = self.__get_hostid(statusupdate.new_hostid)
        
        if 'percentage' in kw:
            statusupdate.set_attribute_percentage(kw['percentage'])
        
        if 'speed' in kw:
            statusupdate.set_attribute_speed(kw['speed'])
        
        if 'status' in kw:
            s = statusupdate.new_status()
            s.set_attribute_value(kw['status'])
            statusupdate.Status = s
        
        status, nextupdate = self.client.statusUpdate(statusupdate)
        if status.get_attribute_value() != 0:
            raise exception.StatusUpdateException()
        
        return nextupdate
