#
# $Id: nettools.py 86 2009-05-02 19:02:20Z lxp $
#
# This file is part of OpenClone.
#
# Copyright (C) 2009  David Gnedt
#
# OpenClone is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenClone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with OpenClone.  If not, see <http://www.gnu.org/licenses/>.
#

import logging
import socket
import struct
import subprocess

logger = logging.getLogger('nettools')

def getDeviceInfo(name, **kw):
    result = None
    
    # TODO: Get path (configuration?)
    args = ['/bin/ip', 'addr', 'show', 'dev', name]
    if 'dest' in kw and 'mask' in kw:
        args.append('to')
        args.append('%s/%s' % (kw['dest'], kw['mask']))
    
    elif 'dest' in kw or 'mask' in kw:
        raise Exception('Keywords dest and mask require each other')
    
    logger.debug('Exec: %s' % ' '.join(args))
    proc = subprocess.Popen(args, stdout=subprocess.PIPE, close_fds=True)
    if proc.wait() == 0:
        for line in proc.stdout:
            #if not line.startswith(' '):
            #    continue
            
            parts = line.split()
            if parts[0] == 'inet':
                result = dict()
                result['ip'], result['mask'] = parts[1].split('/')
                break
    
    else:
        raise Exception('/bin/ip failed')
    
    if result is None:
        raise Exception('No IP address found')
    
    return result

class RoutingTable:
    routing_table = None
    
    def __init__(self):
        # Read routing table
        self.routing_table = list()
        f = open('/proc/net/route')
        line = f.readline()
        parts_name = line.split()
        
        # Check for necessary fields
        parts_name.index('Iface')
        parts_name.index('Destination')
        parts_name.index('Gateway')
        parts_name.index('Metric')
        parts_name.index('Mask')
        
        for line in f:
            parts = line.split()
            entry = dict(zip(parts_name, parts))
            # Hex str to long conversion
            for name in ['Destination', 'Gateway', 'Mask']:
                entry[name] = long(entry[name], 16)
            
            # Dec str to int conversion
            for name in ['Metric']:
                entry[name] = int(entry[name])
            
            self.routing_table.append(entry)
        
        f.close()
    
    def lookup(self, client_ip):
        # Str (dec ip) to unsigned int
        next_hop = struct.unpack('I', socket.inet_aton(client_ip))[0]
        
        while True:
            result = None
            for entry in self.routing_table:
                if next_hop & entry['Mask'] == entry['Destination']:
                    print next_hop, entry
                    if result is not None:
                        # Longest prefix match
                        if result['Mask'] > entry['Mask']:
                            continue
                        elif result['Mask'] == entry['Mask']:
                            # First minimal metric match
                            if result['Metric'] <= entry['Metric']:
                                continue
                    
                    result = entry
            
            if result is None:
                raise Exception('No route to host')
            
            elif result['Gateway'] != 0:
                # Lookup route to gateway
                next_hop = result['Gateway']
            
            else:
                break
        
        result = dict(result)
        
        # Unsigned int to str (dec ip) conversion
        result['Destination'] = socket.inet_ntoa(struct.pack('I', result['Destination']))
        result['Gateway'] = socket.inet_ntoa(struct.pack('I', result['Gateway']))
        result['Mask'] = socket.inet_ntoa(struct.pack('I', result['Mask']))
        
        return result
