# coding=utf8
#
# $Id: sysinfo.py 86 2009-05-02 19:02:20Z lxp $
#
# This file is part of OpenClone.
#
# Copyright (C) 2009  David Gnedt
#
# OpenClone is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenClone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with OpenClone.  If not, see <http://www.gnu.org/licenses/>.
#

import logging
import os
import struct
import subprocess

import blockdev
import ioctl

logger = logging.getLogger('sysinfo')

def getEthernetInterfaces():
    #
    # Example
    #
    # $ ifconfig
    # ath0      Link encap:Ethernet  Hardware Adresse 00:16:cf:cc:39:8e  
    #           inet Adresse:192.168.10.243  Bcast:192.168.10.255  Maske:255.255.255.0
    #           inet6-Adresse: fe80::216:cfff:fecc:398e/64 Gültigkeitsbereich:Verbindung
    #           UP BROADCAST RUNNING MULTICAST  MTU:1500  Metrik:1
    #           RX packets:108 errors:0 dropped:0 overruns:0 frame:0
    #           TX packets:41 errors:0 dropped:0 overruns:0 carrier:0
    #           Kollisionen:0 Sendewarteschlangenlänge:0 
    #           RX bytes:20278 (20.2 KB)  TX bytes:5956 (5.9 KB)
    # 
    # eth0      Link encap:Ethernet  Hardware Adresse 00:1e:83:2e:72:59  
    #           UP BROADCAST MULTICAST  MTU:1500  Metrik:1
    #           RX packets:0 errors:0 dropped:0 overruns:0 frame:0
    #           TX packets:0 errors:0 dropped:0 overruns:0 carrier:0
    #           Kollisionen:0 Sendewarteschlangenlänge:1000 
    #           RX bytes:0 (0.0 B)  TX bytes:0 (0.0 B)
    #           Interrupt:220 Basisadresse:0xc000 
    # 
    # lo        Link encap:Lokale Schleife  
    #           inet Adresse:127.0.0.1  Maske:255.0.0.0
    #           inet6-Adresse: ::1/128 Gültigkeitsbereich:Maschine
    #           UP LOOPBACK RUNNING  MTU:16436  Metrik:1
    #           RX packets:316 errors:0 dropped:0 overruns:0 frame:0
    #           TX packets:316 errors:0 dropped:0 overruns:0 carrier:0
    #           Kollisionen:0 Sendewarteschlangenlänge:0 
    #           RX bytes:119590 (119.5 KB)  TX bytes:119590 (119.5 KB)
    # 
    # wifi0     Link encap:UNSPEC  Hardware Adresse 00-16-CF-CC-39-8E-00-00-00-00-00-00-00-00-00-00  
    #           UP BROADCAST RUNNING MULTICAST  MTU:1500  Metrik:1
    #           RX packets:27454 errors:0 dropped:0 overruns:0 frame:25
    #           TX packets:945 errors:0 dropped:0 overruns:0 carrier:0
    #           Kollisionen:0 Sendewarteschlangenlänge:280 
    #           RX bytes:2972521 (2.9 MB)  TX bytes:55400 (55.4 KB)
    #           Interrupt:16 
    # 
    class EthernetInterface:
        device = None
        mac = None
        ip = None
        netmask = None
    
    # /sys/class/net/eth0/address
    # /sys/class/net/eth0/operationstate (doesn't work before Ubuntu 8.10???)
    devices = []
    dev = None
    # TODO: Unset LANG variable to force default language
    # TODO: Get path (configuration?)
    args = ['/sbin/ifconfig']
    logger.debug('Exec: %s' % ' '.join(args))
    prog = subprocess.Popen(args, stdout=subprocess.PIPE, close_fds=True)
    for line in prog.stdout:
        if line[0].isalnum():
            if dev is not None:
                devices.append(dev)
                dev = None
            
            if line.find('Link encap:Ethernet') != -1:
                parts = line.split()
                dev = EthernetInterface()
                dev.device = parts[0]
                dev.mac = parts[-1].replace(':', '')
        
        elif dev is not None and line.lstrip().startswith('inet '):
            parts = line.split()
            dev.ip = parts[1].split(':')[1]
            dev.netmask = parts[-1].split(':')[1]
    
    if dev is not None:
        devices.append(dev)
    
    return devices

def getSerialNo():
    # TODO: Get path (configuration?)
    args = ['/usr/sbin/dmidecode', '-s', 'system-serial-number']
    logger.debug('Exec: %s' % ' '.join(args))
    proc = subprocess.Popen(args, stdout=subprocess.PIPE, close_fds=True)
    serial_no = proc.stdout.read().strip()
    if proc.wait() != 0:
        raise Exception('dmidecode failed')
    
    return serial_no

def getProcessors():
    #
    # Example
    #
    # $ cat /proc/cpuinfo 
    # processor    : 0
    # vendor_id    : AuthenticAMD
    # cpu family    : 15
    # model        : 104
    # model name    : AMD Turion(tm) 64 X2 Mobile Technology TL-60
    # stepping    : 2
    # cpu MHz        : 800.000
    # cache size    : 512 KB
    # physical id    : 0
    # siblings    : 2
    # core id        : 0
    # cpu cores    : 2
    # apicid        : 0
    # initial apicid    : 0
    # fdiv_bug    : no
    # hlt_bug        : no
    # f00f_bug    : no
    # coma_bug    : no
    # fpu        : yes
    # fpu_exception    : yes
    # cpuid level    : 1
    # wp        : yes
    # flags        : fpu vme de pse tsc msr pae mce cx8 apic sep mtrr pge mca cmov pat pse36 clflush mmx fxsr sse sse2 ht syscall nx mmxext fxsr_opt rdtscp lm 3dnowext 3dnow pni cx16 lahf_lm cmp_legacy svm extapic cr8_legacy 3dnowprefetch
    # bogomips    : 1600.05
    # clflush size    : 64
    # power management: ts fid vid ttp tm stc 100mhzsteps
    # 
    # processor    : 1
    # vendor_id    : AuthenticAMD
    # cpu family    : 15
    # model        : 104
    # model name    : AMD Turion(tm) 64 X2 Mobile Technology TL-60
    # stepping    : 2
    # cpu MHz        : 800.000
    # cache size    : 512 KB
    # physical id    : 0
    # siblings    : 2
    # core id        : 1
    # cpu cores    : 2
    # apicid        : 1
    # initial apicid    : 1
    # fdiv_bug    : no
    # hlt_bug        : no
    # f00f_bug    : no
    # coma_bug    : no
    # fpu        : yes
    # fpu_exception    : yes
    # cpuid level    : 1
    # wp        : yes
    # flags        : fpu vme de pse tsc msr pae mce cx8 apic sep mtrr pge mca cmov pat pse36 clflush mmx fxsr sse sse2 ht syscall nx mmxext fxsr_opt rdtscp lm 3dnowext 3dnow pni cx16 lahf_lm cmp_legacy svm extapic cr8_legacy 3dnowprefetch
    # bogomips    : 1600.05
    # clflush size    : 64
    # power management: ts fid vid ttp tm stc 100mhzsteps
    # 
    cpus = []
    cpu = {}
    f = open('/proc/cpuinfo')
    for line in f:
        if line.startswith('\n'):
            cpus.append(cpu)
            cpu = {}
        else:
            parts = line.split(':')
            cpu[parts[0].strip()] = parts[1].strip()
    
    f.close()
    if len(cpu) != 0:
        cpus.append(cpu)
    
    return cpus

def getMemory():
    #
    # Example
    #
    # $ cat /proc/meminfo 
    # MemTotal:      1942760 kB
    # MemFree:       1041992 kB
    # Buffers:         26308 kB
    # Cached:         389944 kB
    # SwapCached:          0 kB
    # Active:         528952 kB
    # Inactive:       265460 kB
    # HighTotal:     1048320 kB
    # HighFree:       267320 kB
    # LowTotal:       894440 kB
    # LowFree:        774672 kB
    # SwapTotal:     4000144 kB
    # SwapFree:      4000144 kB
    # Dirty:              32 kB
    # Writeback:           0 kB
    # AnonPages:      378196 kB
    # Mapped:          96156 kB
    # Slab:            41304 kB
    # SReclaimable:    24856 kB
    # SUnreclaim:      16448 kB
    # PageTables:       3232 kB
    # NFS_Unstable:        0 kB
    # Bounce:              0 kB
    # WritebackTmp:        0 kB
    # CommitLimit:   4971524 kB
    # Committed_AS:  1033808 kB
    # VmallocTotal:   110584 kB
    # VmallocUsed:     22172 kB
    # VmallocChunk:    87944 kB
    # HugePages_Total:     0
    # HugePages_Free:      0
    # HugePages_Rsvd:      0
    # HugePages_Surp:      0
    # Hugepagesize:     4096 kB
    # DirectMap4k:     49152 kB
    # DirectMap4M:    868352 kB
    mem = {}
    f = open('/proc/meminfo')
    for line in f:
        parts = line.split(':')
        mem[parts[0].strip()] = parts[1].strip()
    
    f.close()
    return mem

def getHardDisks():
    #
    # Example
    #
    # $ cat /proc/partitions
    # major minor  #blocks  name
    # 
    #    8     0  156290904 sda
    #    8     1    4000153 sda1
    #    8     2   54629032 sda2
    #    8     3   97659135 sda3
    class HardDisk:
        address = None
        size = None
        sector_size = None
        model_no = None
        serial_no = None
        firmware_rev = None
        wwn = None
        cylinder = None
        heads = None
        sectors = None
        partitions = None
    
    hds = []
    hd = None
    f = open('/proc/partitions')
    for line in f:
        parts = line.split()
        if len(parts) != 4:
            continue
        
        if parts[3] == 'name':
            continue
        
        address = '/dev/' + parts[3]
        
        dev = open(address)
        #try:
        #    dev = open(address)
        #except Exception:
        #    continue
        
        #
        # Example
        #
        # geometry = (heads, sectors, cylinder, start)
        geometry = ioctl.getStruct(dev, blockdev.HDIO_GETGEO, blockdev.FORMAT_HD_GEOMETRY)
        
        # blockdev.HDIO_GETGEO_BIG fails at this time
        #geometry = ioctl.getStruct(disk, blockdev.HDIO_GETGEO_BIG, blockdev.FORMAT_HD_BIG_GEOMETRY)
        
        if geometry[3] == 0:
            hd = HardDisk()
            hd.address = address
            hd.partitions = []
            
            hd.heads = geometry[0]
            hd.sectors = geometry[1]
            hd.cylinder = geometry[2]
            
            hd.size = ioctl.getULongLong(dev, blockdev.BLKGETSIZE64)
            
            # /sys/block/sda/queue/hw_sector_size (available since Ubuntu 8.10)
            hd.sector_size = ioctl.getInt(dev, blockdev.BLKSSZGET)
            
            # Do calculation for large hard disks
            # because blockdev.HDIO_GETGEO_BIG fails at this time
            # with IOError: [Errno 25] Inappropriate ioctl for device
            total_sectors = hd.size / hd.sector_size
            cylindersize = hd.heads * hd.sectors
            cylinder = total_sectors / cylindersize
            if cylinder > 65535:
                logger.warn('Hard disk size exceeds HDIO_GETGEO cylinder limit! (HDIO_GETGEO: %d, Calculation: %d)' % (hd.cylinder, cylinder))
                hd.cylinder = cylinder
            else:
                if cylinder != hd.cylinder:
                    raise Exception('Cylinder missmatch (HDIO_GETGEO: %d, Calculation: %d)' % (hd.cylinder, cylinder))
            
            # TODO: Get path (configuration?)
            args = ['/sbin/hdparm', '-I', address]
            logger.debug('Exec: %s' % ' '.join(args))
            hdparm = subprocess.Popen(args, stdout=subprocess.PIPE, close_fds=True)
            for line in hdparm.stdout:
                parts = line.split(':')
                parts[0] = parts[0].strip()
                if parts[0] == 'Model Number':
                    hd.model_no = parts[1].strip()
                elif parts[0] == 'Serial Number':
                    hd.serial_no = parts[1].strip()
                elif parts[0] == 'Firmware Revision':
                    hd.firmware_rev = parts[1].strip()
                elif parts[0] == 'Logical Unit WWN Device Identifier':
                    wwn = parts[1].strip()
                    if len(wwn) != 16:
                        # TODO: Workaround the WWN issue of hdparm < 9.2 (maybe with hdparm --Istdout or direct ioctls)
                        logger.warn('WWN may be wrong (bug of hdparm < 9.2)')
                    
                    hd.wwn = int(wwn, 16)
            
            # TODO: Maybe use WWN field in ata identify also if wwn feature is not defined?
            
            hds.append(hd)
        
        else:
            if hd is not None and address.startswith(hd.address):
                hd.partitions.append(address)
        
        dev.close()
    
    f.close()
    return hds

def getPartition(hd_address, part_address):
    class Partition:
        no = None
        record_type = None
        bootable = None
        partition_type = None
        start_lba = None
        sectors = None
        start_cylinder = None
        start_head = None
        start_sector = None
        end_cylinder = None
        end_head = None
        end_sector = None
        fs = None
        os = None
    
    part = Partition()
    if part_address[-1].isdigit():
        part.no = int(part_address[-1])
    else:
        raise Exception('Unable to detect partition number')
    
    # TODO: Implement direct disk access
    
    # TODO: Get path (configuration?)
    args = ['/sbin/fdisk', hd_address]
    logger.debug('Exec: %s' % ' '.join(args))
    fdisk = subprocess.Popen(args, stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.PIPE, close_fds=True)
    fdisk.stdin.write('x\n')
    fdisk.stdin.write('p\n')
    fdisk.stdin.write('q\n')
    for line in fdisk.stdout:
        parts = line.split()
        if len(parts) == 11 and parts[0].isdigit() and int(parts[0]) == part.no:
            if parts[1] == '80':
                part.bootable = True
            elif parts[1] == '00':
                part.bootable = False
            else:
                raise Exception('Unknown partition state %s', parts[1])
            
            part.start_head = int(parts[2])
            part.start_sector = int(parts[3])
            part.start_cylinder = int(parts[4])
            
            part.end_head = int(parts[5])
            part.end_sector = int(parts[6])
            part.end_cylinder = int(parts[7])
            
            part.start_lba = int(parts[8])
            part.sectors = int(parts[9])
            
            part.partition_type = int(parts[10], 16)
    
    if part.no > 4:
        # Logical partition
        part.record_type = 'logical'
    
    elif part.partition_type == 0x05 or part.partition_type == 0x0f:
        # Extended partition
        #
        # 0x05 DOS 3.3+ Extended Partition
        # 0x0f WIN95: Extended partition, LBA-mapped
        # 0x85 Linux extended partition
        # TODO: Linux extended 0x85 used?
        # Source: http://www.win.tue.nl/~aeb/partitions/partition_types-1.html
        part.record_type = 'extended'
    
    else:
        # Primary partition
        part.record_type = 'primary'
    
    # TODO: Get path (configuration?)
    args = ['/lib/udev/vol_id', '--type', part_address]
    logger.debug('Exec: %s' % ' '.join(args))
    # TODO: Handle stderr (unknown volume type)
    volid = subprocess.Popen(args, stdout=subprocess.PIPE, close_fds=True)
    fs = volid.stdout.read().strip()
    if len(fs) > 0:
        part.fs = fs
    
    # Alternative for NTFS detection: ntfs-3g.probe
    # TODO: Implement operating system detection
    
    return part
