#
# $Id: directpart.py 86 2009-05-02 19:02:20Z lxp $
#
# This file is part of OpenClone.
#
# Copyright (C) 2009  David Gnedt
#
# OpenClone is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenClone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with OpenClone.  If not, see <http://www.gnu.org/licenses/>.
#

import fcntl
import logging
import os
import struct

from ..sysinfo import blockdev

#
# MBR structure
#
# Bootloader          440 bytes              -> bootloader
# Disk signature        4 bytes              -> disk_signature
# Unused (0x00)         2 bytes              -> disk_signature
# 4 * Partition entry  64 bytes (4*16 bytes) -> partitions
# 0xaa55                2 bytes              X
#
#
# EBR structure
#
# Unused (0x00)            446 bytes              -> extended
# Logical partition entry   16 bytes              -> extended
# Next EBR partition entry  16 bytes              -> extended
# Unused (0x00)             32 bytes (2*16 bytes) -> extended
# 0xaa55                     2 bytes              X
#
# For further EBR details see http://en.wikipedia.org/wiki/Extended_boot_record
#

logger = logging.getLogger('directpart')

class MBRNotFoundException(Exception):
    pass

class DirectPart:
    address = None
    
    def __init__(self, address):
        self.address = address
    
    def readMBR(self):
        disk_signature = None
        bootloader = None
        partitions = None
        extended = None
        
        dev = open(self.address)
        
        try:
            dev.seek(510, os.SEEK_SET)
            if struct.unpack('H', dev.read(2))[0] != 0xaa55:
                raise MBRNotFoundException('MBR signature not found')
            
            dev.seek(0, os.SEEK_SET)
            bootloader = dev.read(440)
            disk_signature = dev.read(6)
            
            partitions = dev.read(64)
            
            part = partitions
            extended_start = None
            next = None
            
            while len(part) >= 16:
                # TODO: Correct extended types?
                if part[4] == '\x05' or part[4] == '\x0f':
                    next = struct.unpack('L', part[8:12])[0]
                    if next < 1:
                        raise Exception('Extended boot record failure')
                    
                    if extended is None:
                        extended_start = next
                    
                    else:
                        next = extended_start + next
                    
                    logger.debug('Next extended boot record: %d' % next)
                    
                    # TODO: Get harddisk sector size
                    dev.seek(next*512, os.SEEK_SET)
                    extended_bootloader = dev.read(446)
                    part = dev.read(64)
                    if struct.unpack('H', dev.read(2))[0] != 0xaa55:
                        raise Exception('MBR signature not found in extended boot record')
                    
                    if extended is None:
                        extended = extended_bootloader + part
                    
                    else:
                        extended = extended + extended_bootloader + part
                
                else:
                    part = part[16:]
        
        finally:
            dev.close()
        
        return (disk_signature, bootloader, partitions, extended)
    
    def writeMBR(self, disk_signature, partitions, **kw):
        if len(disk_signature) > 6:
            raise Exception('Disk signature exceeds 6 byte limit')
        
        if len(partitions) > 64:
            raise Exception('Partitions exceed 64 byte limit')
        
        if 'bootloader' in kw and len(kw['bootloader']) > 440:
            raise Exception('Bootloader exceeds 440 byte limit')
        
        # TODO: Use direct IO?
        #try:
        #    # TODO: Fix direct IO
        #    import directio
        #    dev = directio.open(self.address, directio.O_WRONLY)
        #    
        #    if 'bootloader' in kw:
        #        # Write bootloader
        #        os.lseek(dev, 0, os.SEEK_SET)
        #        directio.write(dev, kw['bootloader'])
        #    
        #    # Write disk signature
        #    os.lseek(dev, 440, os.SEEK_SET)
        #    directio.write(dev, disk_signature)
        #    
        #    # Write partitions
        #    os.lseek(dev, 446, os.SEEK_SET)
        #    directio.write(dev, partitions)
        #    
        #    # Write MBR signature
        #    os.lseek(dev, 510, os.SEEK_SET)
        #    directio.write(dev, struct.pack('H', 0xaa55))
        #    
        #    directio.close(dev)
        #
        #except ImportError:
        dev = open(self.address, 'w')
        
        if 'bootloader' in kw:
            # Write bootloader
            dev.seek(0, os.SEEK_SET)
            dev.write(kw['bootloader'])
        
        # Write disk signature
        dev.seek(440, os.SEEK_SET)
        dev.write(disk_signature)
        
        # Write partitions
        dev.seek(446, os.SEEK_SET)
        dev.write(partitions)
        
        # Write MBR signature
        dev.seek(510, os.SEEK_SET)
        dev.write(struct.pack('H', 0xaa55))
        
        # Write extended boot record
        if 'extended' in kw and len(kw['extended']) > 0:
            if len(kw['extended']) % 510 == 0:
                extended_len = 510
            
            elif len(kw['extended']) % 64 == 0:
                extended_len = 64
            
            else:
                raise Exception('Unexpected extended length')
            
            part = partitions
            extended_start = None
            next = None
            
            for i in range(0, len(kw['extended']), extended_len):
                while len(part) >= 16:
                    # TODO: Correct extended types?
                    if part[4] == '\x05' or part[4] == '\x0f':
                        if next is not None:
                            raise Exception('Multiple extended partitions found')
                        
                        next = struct.unpack('L', part[8:12])[0]
                        if next < 1:
                            raise Exception('Extended boot record failure')
                        
                        if extended_start is None:
                            extended_start = next
                        
                        else:
                            next = extended_start + next
                        
                        logger.debug('Next extended boot record: %d' % next)
                    
                    part = part[16:]
                
                if next is not None:
                    extended_bootloader = None
                    
                    if extended_len == 510:
                        extended_bootloader = kw['extended'][i:i+extended_len-64]
                        if len(extended_bootloader) != 446:
                            raise Exception('Unexpected extended bootloader size %d' % len(extended_bootloader))
                    
                    part = kw['extended'][i+extended_len-64:i+extended_len]
                    if len(part) != 64:
                        raise Exception('Unexpected partition record size %d' % len(part))
                    
                    # TODO: Get harddisk sector size
                    dev.seek(next*512, os.SEEK_SET)
                    if extended_bootloader is None:
                        dev.write('\x00' * 446)
                    
                    else:
                        dev.write(extended_bootloader)
                    
                    dev.write(part)
                    dev.write(struct.pack('H', 0xaa55))
                    
                    next = None
                
                else:
                    raise Exception('Position for next extended boot record unknown')
        
        #dev.flush()
        # TODO: Flush disk buffer
        dev.close()
    
    def readData(self, start, end):
        # TODO: Use direct IO?
        #try:
        #    import directio
        #    dev = directio.open(self.address, directio.O_RDONLY)
        #    os.lseek(dev, start, os.SEEK_SET)
        #    data = directio.read(dev, end-start)
        #    directio.close(dev)
        #
        #except ImportError:
        dev = open(self.address)
        dev.seek(start, os.SEEK_SET)
        data = dev.read(end-start)
        dev.close()
        
        return data
    
    def writeData(self, data, start):
        # TODO: Use direct IO?
        #try:
        #    raise ImportError()
        #    import directio
        #    dev = directio.open(self.address, directio.O_WRONLY)
        #    os.lseek(dev, start, os.SEEK_SET)
        #    directio.write(dev, data)
        #    directio.close(dev)
        #
        #except ImportError:
        dev = open(self.address, 'w')
        dev.seek(start, os.SEEK_SET)
        dev.write(data)
        dev.close()
    
    def writePartitiontable(self, disk_signature, partitions, **kw):
        self.writeMBR(disk_signature, partitions, **kw)
        
        if 'unused' in kw:
            # TODO: Validate length
            self.writeData(kw['unused'], 512)
        
        # Reload partition table
        dev = open(self.address)
        fcntl.ioctl(dev, blockdev.BLKRRPART)
        dev.close()
